"use client";

import React, { useState, useEffect } from "react";
import {
  Button,
  Card,
  Row,
  Tabs,
  Tag,
  Tooltip,
  Breadcrumb,
  Descriptions,
  Empty,
  Table,
  Badge,
  Collapse,
} from "antd";
import {
  HomeOutlined,
  EyeOutlined,
  EditOutlined,
  DeleteOutlined,
  FileTextOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import { useSearchParams, useRouter } from "next/navigation";
import { motion } from "framer-motion";
import dayjs from "dayjs";

import { ColumnsType } from "antd/es/table";
import { Supplier } from "@/app/types/Supplier";
import { SupplierAssessment } from "@/app/types/SupplierAssessment";
import { SupplierCertification } from "@/app/types/SupplierCertification";
import { ArrowBigLeft } from "lucide-react";

const SupplierDetailPage = () => {
  const searchParams = useSearchParams();
  const router = useRouter();
  const supplierId = searchParams.get("id");
  const supplierName = searchParams.get("name") || "Unknown Supplier";

  const [supplier, setSupplier] = useState<Supplier | null>(null);
  const [assessments, setAssessments] = useState<SupplierAssessment[]>([]);
  const [certifications, setCertifications] = useState<SupplierCertification[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (!supplierId) router.push("/suppliers");
    const fetchData = () => {
      const mockSupplier: Supplier = {
        id: supplierId || "1",
        supplierCode: "SUP001",
        companyName: supplierName,
        contactPerson: "John Doe",
        email: "john.doe@example.com",
        phone: "+1-555-123-4567",
        country: "USA",
        address: "123 Business St, New York",
        companyProfile: "Leading supplier of industrial goods",
        ownershipDetails: "Privately owned",
        gstTaxId: "GST123456789",
        riskClassification: "Low",
        status: "Active",
        created_at: dayjs().subtract(30, "days").toISOString(),
        updated_at: dayjs().toISOString(),
      };
      const mockAssessments: SupplierAssessment[] = [
        {
          id: "1",
          supplierId: supplierId || "1",
          supplierName,
          assessmentDate: dayjs().subtract(10, "days").toISOString(),
          assessmentType: "Routine",
          assessor: "Jane Smith",
          score: 85,
          findings: "Met all standards",
          recommendations: "Continue monitoring",
          status: "Pass",
          created_at: dayjs().subtract(10, "days").toISOString(),
        },
      ];
      const mockCertifications: SupplierCertification[] = [
        {
          id: "1",
          supplierId: supplierId || "1",
          supplierName,
          certificationType: "ISO 9001",
          certificationNumber: "CERT001",
          issuingBody: "ISO",
          issueDate: dayjs().subtract(20, "days").toISOString(),
          expiryDate: dayjs().add(10, "months").toISOString(),
          scope: "Quality Management",
          status: "Valid",
          created_at: dayjs().subtract(20, "days").toISOString(),
        },
      ];
      setSupplier(mockSupplier);
      setAssessments(mockAssessments);
      setCertifications(mockCertifications);
      setLoading(false);
    };
    fetchData();
  }, [supplierId, router, supplierName]);

  const renderTag = (value: string) => {
    const colorMap: { [key: string]: string } = {
      Low: "success",
      Medium: "warning",
      High: "error",
      Active: "success",
      Inactive: "warning",
      BlackListed: "error",
      Pass: "success",
      Fail: "error",
      Conditional: "warning",
      Valid: "success",
      Expired: "error",
      Pending: "warning",
    };
    const iconMap: { [key: string]: React.JSX.Element } = {
      Low: <EyeOutlined />,
      Medium: <EditOutlined />,
      High: <DeleteOutlined />,
      Active: <EyeOutlined />,
      Inactive: <EditOutlined />,
      BlackListed: <DeleteOutlined />,
      Pass: <EyeOutlined />,
      Fail: <DeleteOutlined />,
      Conditional: <EditOutlined />,
      Valid: <EyeOutlined />,
      Expired: <DeleteOutlined />,
      Pending: <EditOutlined />,
    };
    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const assessmentColumns: ColumnsType<SupplierAssessment> = [
    { title: "Date", dataIndex: "assessmentDate", render: (text) => dayjs(text).format("YYYY-MM-DD") },
    { title: "Type", dataIndex: "assessmentType" },
    { title: "Assessor", dataIndex: "assessor" },
    { title: "Score (%)", dataIndex: "score", render: (text) => `${text || 0}%` },
    { title: "Status", dataIndex: "status", render: renderTag },
  ];

  const certificationColumns: ColumnsType<SupplierCertification> = [
    { title: "Type", dataIndex: "certificationType" },
    { title: "Number", dataIndex: "certificationNumber" },
    { title: "Issue Date", dataIndex: "issueDate", render: (text) => dayjs(text).format("YYYY-MM-DD") },
    { title: "Expiry Date", dataIndex: "expiryDate", render: (text) => dayjs(text).format("YYYY-MM-DD") },
    { title: "Status", dataIndex: "status", render: renderTag },
  ];

  if (loading || !supplier) return <div className="text-center py-10">Loading...</div>;

   return (
    <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-6 min-h-screen bg-gradient-to-br from-gray-100 to-white dark:from-gray-900 dark:to-gray-800">
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.6, ease: "easeOut" }}
      >
        <Button
  icon={<ArrowBigLeft className="text-green-600" />}
  onClick={() => router.push("/suppliers")}
  className="mb-3 text-green-600 hover:text-green-700 border border-green-600 hover:border-green-700 bg-transparent shadow-none"
>
  Back to Suppliers
</Button>
        <h2 className="text-2xl sm:text-3xl md:text-4xl font-extrabold text-gray-900 dark:text-white mb-3 bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
          Supplier Details - {supplier.companyName}
        </h2>
        <p className="text-gray-600 dark:text-gray-400 text-base sm:text-lg mb-5">
          Comprehensive overview of {supplier.companyName}.
        </p>
        <Breadcrumb
          className="mb-5 sm:mb-6 text-sm sm:text-base text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            { title: <Tooltip title="Dashboard"><HomeOutlined /></Tooltip>, href: "/dashboard" },
            { title: "Suppliers", href: "/suppliers" },
            { title: "Details" },
          ]}
        />
      </motion.div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 mt-3">
        {/* Summary Card */}
        <motion.div
          className="lg:col-span-1 flex flex-col"
          initial={{ scale: 0.95, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          transition={{ delay: 0.2, duration: 0.5 }}
        >
          <Card className="shadow-2xl bg-white dark:bg-gray-800 rounded-2xl p-6 border-0 transform hover:-translate-y-2 transition-all duration-300 flex-1">
            <h3 className="text-xl font-semibold text-gray-800 dark:text-white mb-4 border-b-2 border-blue-200 pb-2 flex justify-between items-center">
          Summary
           <div className="flex gap-2">
            <Tooltip title="Assessment">
              <Button
                icon={<FileTextOutlined />}
                onClick={() => router.push(`/suppliers/supplierassessment?id=${supplierId}&name=${encodeURIComponent(supplierName)}`)}
                size="small"
                className="bg-purple-500 hover:bg-purple-600 text-white"
              />
            </Tooltip>
            <Tooltip title="Certification">
              <Button
                icon={<IdcardOutlined />}
                onClick={() => router.push(`/suppliers/supplierscertification?id=${supplierId}&name=${encodeURIComponent(supplierName)}`)}
                size="small"
                className="bg-teal-500 hover:bg-teal-600 text-white"
              />
            </Tooltip>
          </div>
        </h3>
            <div className="space-y-4">
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">Code:</span>
                <span className="text-gray-900 dark:text-white font-medium">{supplier.supplierCode}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">Risk:</span>
                {renderTag(supplier.riskClassification)}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">Status:</span>
                {renderTag(supplier.status)}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">Created:</span>
                <span className="text-gray-900 dark:text-white font-medium">{dayjs(supplier.created_at).format("YYYY-MM-DD")}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">Company Profile:</span>
                <span className="text-gray-900 dark:text-white font-medium">{supplier.companyProfile || "-"}</span>
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">Ownership:</span>
                <span className="text-gray-900 dark:text-white font-medium">{supplier.ownershipDetails || "-"}</span>
              </div>
            </div>
          </Card>
        </motion.div>

        {/* Details and Tabs */}
        <motion.div
          className="lg:col-span-2 flex flex-col"
          initial={{ scale: 0.95, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          transition={{ delay: 0.3, duration: 0.5 }}
        >
          <Card className="shadow-2xl bg-white dark:bg-gray-800 rounded-2xl p-6 border-0 transform hover:-translate-y-2 transition-all duration-300 flex-1">
            <Tabs
              defaultActiveKey="1"
              className="mb-6"
              items={[
                {
                  key: "1",
                  label: "Overview",
                  children: (
                    <div className="space-y-4">
                      <Collapse
                        items={[
                          {
                            key: "1",
                            label: "Basic Information",
                            children: (
                              <Descriptions column={1} size="small" className="text-sm text-gray-700 dark:text-gray-300">
                                <Descriptions.Item label="Contact">{supplier.contactPerson}</Descriptions.Item>
                                <Descriptions.Item label="Email">{supplier.email}</Descriptions.Item>
                                <Descriptions.Item label="Phone">{supplier.phone}</Descriptions.Item>
                                <Descriptions.Item label="Country">{supplier.country}</Descriptions.Item>
                                <Descriptions.Item label="Address">{supplier.address}</Descriptions.Item>
                                <Descriptions.Item label="GST/Tax ID">{supplier.gstTaxId}</Descriptions.Item>
                              </Descriptions>
                            ),
                          },
                          {
                            key: "2",
                            label: "Company Details",
                            children: (
                              <div className="text-sm text-gray-700 dark:text-gray-300">
                                <p><strong>Profile:</strong> {supplier.companyProfile || "-"}</p>
                                <p><strong>Ownership:</strong> {supplier.ownershipDetails || "-"}</p>
                              </div>
                            ),
                          },
                        ]}
                      />
                    </div>
                  ),
                },
                {
                  key: "2",
                  label: "Assessments",
                  children: assessments.length > 0 ? (
                    <Table
                      columns={assessmentColumns}
                      dataSource={assessments}
                      rowKey="id"
                      pagination={{ pageSize: 5 }}
                      scroll={{ x: "max-content" }}
                      className="mb-4 border border-gray-200 dark:border-gray-700 rounded-lg"
                    />
                  ) : (
                    <Empty description="No assessments available" className="py-10" />
                  ),
                },
                {
                  key: "3",
                  label: "Certifications",
                  children: certifications.length > 0 ? (
                    <Table
                      columns={certificationColumns}
                      dataSource={certifications}
                      rowKey="id"
                      pagination={{ pageSize: 5 }}
                      scroll={{ x: "max-content" }}
                      className="mb-4 border border-gray-200 dark:border-gray-700 rounded-lg"
                    />
                  ) : (
                    <Empty description="No certifications available" className="py-10" />
                  ),
                },
              ]}
            />
          </Card>
        </motion.div>
      </div>
    </div>
  );
};

export default SupplierDetailPage;