"use client";

import React, { useCallback, useEffect, useState } from "react";
import {
  Button,
  Form,
  Input,
  Modal,
  Select,
  Space,
  Row,
  Col,
  Tooltip,
  Tag,
  Breadcrumb,
  Card,
  DatePicker,
  Switch,
  Pagination,
  Dropdown,
} from "antd";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  EyeOutlined,
  SearchOutlined,
  DownloadOutlined,
  InfoCircleOutlined,
  ExclamationCircleOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
  FileTextOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import { Supplier } from "../types/Supplier";
import DataTable from "../components/common/DataTable";
import WidgetCard from "../components/common/WidgetCard";
import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import dayjs from "dayjs";
import { FilterValue } from "antd/es/table/interface";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useRouter } from "next/navigation";


const { RangePicker } = DatePicker;
const riskLevelOptions = ["Low", "Medium", "High"] as const;
const statusOptions = ["Active", "Inactive", "BlackListed"] as const;

export default function SuppliersPage() {
  const [form] = Form.useForm();
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingSupplier, setEditingSupplier] = useState<Supplier | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(null);
  const [filters, setFilters] = useState<Record<string, FilterValue | null>>({});
  const [viewingSupplier, setViewingSupplier] = useState<Supplier | null>(null);
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    riskClassification: undefined,
    status: undefined,
  });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [savedFilters, setSavedFilters] = useState<{ name: string; filters: typeof advancedFilters }[]>([]);
  const isMobile = useIsMobile();
    const router = useRouter();

  const openAddModal = () => {
    setEditingSupplier(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (supplier: Supplier) => {
    setEditingSupplier(supplier);
    form.setFieldsValue(supplier);
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this supplier?",
      onOk: () => {
        setSuppliers((prev) => prev.filter((s) => s.id !== id));
        toast.success("Supplier deleted successfully");
      },
      onCancel: () => { },
    });
  };

  const handleBulkDelete = () => {
    Modal.confirm({
      title: `Are you sure you want to delete ${selectedRows.length} suppliers?`,
      onOk: () => {
        setSuppliers((prev) => prev.filter((s) => !selectedRows.includes(s.id.toString())));
        setSelectedRows([]);
        toast.success("Suppliers deleted successfully");
      },
    });
  };

  const handleFormSubmit = () => {
    form.validateFields().then((values: Supplier) => {
      const formattedValues: Supplier = {
        ...values,
        companyName: values.companyName.trim().replace(/^\w/, (c) => c.toUpperCase()),
        created_at: editingSupplier?.created_at ?? dayjs().toISOString(),
        updated_at: dayjs().toISOString(),
        id: editingSupplier?.id ?? Date.now().toString(),
      };

      if (editingSupplier) {
        setSuppliers((prev) =>
          prev.map((s) => (s.id === editingSupplier.id ? formattedValues : s))
        );
        toast.success("Supplier updated successfully");
      } else {
        setSuppliers((prev) => [...prev, formattedValues]);
        toast.success("Supplier added successfully");
      }

      setIsModalOpen(false);
      form.resetFields();
    });
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Low: "green",
      Medium: "orange",
      High: "red",
      Active: "green",
      Inactive: "gold",
      BlackListed: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Low: <CheckCircleOutlined />,
      Medium: <PauseCircleOutlined />,
      High: <FireOutlined />,
      Active: <CheckCircleOutlined />,
      Inactive: <MinusCircleOutlined />,
      BlackListed: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredSuppliers = suppliers.filter((supplier) => {
    const matchesSearch =
      supplier.companyName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      supplier.supplierCode.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (supplier.created_at &&
        (!dateRange[0] ||
          dayjs(supplier.created_at).isSame(dateRange[0]) ||
          dayjs(supplier.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(supplier.created_at).isSame(dateRange[1]) ||
          dayjs(supplier.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.riskClassification ||
        supplier.riskClassification === advancedFilters.riskClassification) &&
      (!advancedFilters.status || supplier.status === advancedFilters.status);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedSuppliers = filteredSuppliers.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<Supplier> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredSuppliers.map((s) => s.id.toString()) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.id.toString()]
                : prev.filter((id) => id !== record.id.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Code",
      dataIndex: "supplierCode",
      key: "supplierCode",
      sorter: (a, b) => a.supplierCode.localeCompare(b.supplierCode),
    },
    {
  title: "Company",
  dataIndex: "companyName",
  key: "companyName",
  sorter: (a, b) => a.companyName.localeCompare(b.companyName),
  render: (text, record) => (
    <Button
      type="link"
      className="text-green-600 font-medium p-0"
      onClick={() => router.push(`/suppliers/supplierdetail?id=${record.id}&name=${encodeURIComponent(record.companyName)}`)}
    >
      {text}
    </Button>
  ),
  filterSearch: true,
  onFilter: (value, record) =>
    record.companyName.toLowerCase().includes((value as string).toLowerCase()),
  filters: suppliers.map((s) => ({ text: s.companyName, value: s.companyName })),
},
    {
      title: "Contact",
      dataIndex: "contactPerson",
      key: "contactPerson",
      sorter: (a, b) => (a.contactPerson || "").localeCompare(b.contactPerson || ""),
      filterSearch: true,
      onFilter: (value, record) =>
        (record.contactPerson || "").toLowerCase().includes((value as string).toLowerCase()),
      filters: suppliers.map((s) => ({ text: s.contactPerson || "", value: s.contactPerson || "" })),
    },
    {
      title: "Email",
      dataIndex: "email",
      key: "email",
      sorter: (a, b) => (a.email || "").localeCompare(b.email || ""),
    },
    {
      title: "Phone",
      dataIndex: "phone",
      key: "phone",
      sorter: (a, b) => (a.phone || "").localeCompare(b.phone || ""),
    },
    {
      title: "Country",
      dataIndex: "country",
      key: "country",
      sorter: (a, b) => (a.country || "").localeCompare(b.country || ""),
      filterSearch: true,
      onFilter: (value, record) =>
        (record.country || "").toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(suppliers.map((s) => s.country || ""))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Address",
      dataIndex: "address",
      key: "address",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "GST/Tax ID",
      dataIndex: "gstTaxId",
      key: "gstTaxId",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Risk",
      dataIndex: "riskClassification",
      key: "riskClassification",
      filters: riskLevelOptions.map((r) => ({ text: r, value: r })),
      onFilter: (value, record) => record.riskClassification === value,
      render: renderTag,
      sorter: (a, b) => a.riskClassification.localeCompare(b.riskClassification),
    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      filters: statusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.status === value,
      render: renderTag,
      sorter: (a, b) => a.status.localeCompare(b.status),
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      sorter: (a, b) => dayjs(a.created_at).unix() - dayjs(b.created_at).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
        <Button
          icon={<EyeOutlined />}
          // onClick={() => setViewingSupplier(record)}
          onClick={() => router.push(`/suppliers/supplierdetail?id=${record.id}&name=${encodeURIComponent(record.companyName)}`)}
          size="small"
          className="text-blue-600"
        />
      </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id.toString())}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
          <Tooltip title="Supplier Assessment">
            <Button
              icon={<FileTextOutlined />}
              onClick={() => router.push(`/suppliers/supplierassessment?id=${record.id}&name=${encodeURIComponent(record.companyName)}`)}
              size="small"
              className="text-purple-600"
            />
          </Tooltip>
          <Tooltip title="Supplier Certification">
            <Button
              icon={<IdcardOutlined />}
              onClick={() => router.push(`/suppliers/supplierscertification?id=${record.id}&name=${encodeURIComponent(record.companyName)}`)}
              size="small"
              className="text-teal-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "supplierCode", label: "Code" },
    { key: "companyName", label: "Company" },
    { key: "contactPerson", label: "Contact" },
    { key: "email", label: "Email" },
    { key: "phone", label: "Phone" },
    { key: "country", label: "Country" },
    { key: "address", label: "Address" },
    { key: "gstTaxId", label: "GST/Tax ID" },
    { key: "riskClassification", label: "Risk" },
    { key: "status", label: "Status" },
    { key: "created_at", label: "Created At" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const SupplierCard = ({ supplier }: { supplier: Supplier }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">{supplier.companyName}</h3>
        <Tag color="blue">{supplier.country || "N/A"}</Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Code:</strong> {supplier.supplierCode}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Risk:</strong> {renderTag(supplier.riskClassification)}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Status:</strong> {renderTag(supplier.status)}</p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs"><strong>Created:</strong> {dayjs(supplier.created_at).format("YYYY-MM-DD")}</p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => setViewingSupplier(supplier)} size="small" className="text-blue-600" />
        </Tooltip>
        <Tooltip title="Edit">
          <Button icon={<EditOutlined />} onClick={() => openEditModal(supplier)} size="small" type="primary" className="bg-blue-600 hover:bg-blue-700" />
        </Tooltip>
        <Tooltip title="Delete">
          <Button icon={<DeleteOutlined />} onClick={() => handleDelete(supplier.id.toString())} size="small" danger className="text-red-600" />
        </Tooltip>
      </div>
    </div>
  );

  const total = suppliers.length;
  const active = suppliers.filter((s) => s.status === "Active").length;
  const blackListed = suppliers.filter((s) => s.status === "BlackListed").length;
  const highRisk = suppliers.filter((s) => s.riskClassification === "High").length;

  return (
    <>
      <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
        <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          Suppliers <span className="text-gray-500 text-base sm:text-lg font-medium">({suppliers.length})</span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage supplier master data used throughout the QMS system.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Suppliers",
            },
          ]}
        />

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Suppliers"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All registered suppliers"
          />
          <WidgetCard
            title="Active"
            value={active}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Currently active suppliers"
          />
          <WidgetCard
            title="BlackListed"
            value={blackListed}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Suppliers no longer in use"
          />
          <WidgetCard
            title="High Risk"
            value={highRisk}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="High-risk suppliers"
          />
        </div>

        <div className="mt-4">
          <Card title="Supplier List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier by name or code"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Supplier
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Risk Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, riskClassification: value }))
                    }
                    allowClear
                    options={riskLevelOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="supplier_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.riskClassification || advancedFilters.status) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.riskClassification && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          riskClassification: undefined,
                        }))
                      }
                    >
                      Risk Level: {advancedFilters.riskClassification}
                    </Tag>
                  )}
                  {advancedFilters.status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.status}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
                {paginatedSuppliers.map((supplier) => (
                  <SupplierCard key={supplier.id} supplier={supplier} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<Supplier>
                  columns={filteredColumns}
                  dataSource={paginatedSuppliers}
                  rowKey="id"
                  scrollX="max-content"
                  className="mt-4"
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing{" "}
                <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>{Math.min(currentPage * pageSize, filteredSuppliers.length)}</strong> of{" "}
                <strong>{filteredSuppliers.length}</strong> suppliers
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredSuppliers.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={editingSupplier ? "Edit Supplier" : "Add Supplier"}
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            className="w-full max-w-[90vw] sm:max-w-[600px] md:max-w-[800px] lg:max-w-[1000px] top-4"
          >
            <Form form={form} layout="vertical" className="px-2 sm:px-4">
              <Row gutter={[16, 16]}>
                <Col xs={24} sm={12}>
                  <Form.Item
                    name="supplierCode"
                    label="Supplier Code"
                    rules={[{ required: true, message: "Supplier Code is required" }]}
                  >
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item
                    name="companyName"
                    label="Company Name"
                    rules={[{ required: true, message: "Company Name is required" }]}
                  >
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="contactPerson" label="Contact Person">
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="email" label="Email">
                    <Input type="email" />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="phone" label="Phone">
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="country" label="Country">
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24}>
                  <Form.Item name="address" label="Address">
                    <Input.TextArea rows={2} />
                  </Form.Item>
                </Col>
                <Col xs={24}>
                  <Form.Item name="companyProfile" label="Company Profile">
                    <Input.TextArea rows={2} />
                  </Form.Item>
                </Col>
                <Col xs={24}>
                  <Form.Item name="ownershipDetails" label="Ownership Details">
                    <Input.TextArea rows={2} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="gstTaxId" label="GST/Tax ID">
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item
                    name="riskClassification"
                    label="Risk Classification"
                    rules={[{ required: true, message: "Classification is required" }]}
                  >
                    <Select options={riskLevelOptions.map((v) => ({ label: v, value: v }))} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item
                    name="status"
                    label="Status"
                    rules={[{ required: true, message: "Status is required" }]}
                  >
                    <Select options={statusOptions.map((v) => ({ label: v, value: v }))} />
                  </Form.Item>
                </Col>
              </Row>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Supplier Details"
          open={!!viewingSupplier}
          onCancel={() => setViewingSupplier(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingSupplier && (
            <div className="space-y-2 text-sm sm:text-base">
              <p><strong>Code:</strong> {viewingSupplier.supplierCode}</p>
              <p><strong>Name:</strong> <span className="text-green-600">{viewingSupplier.companyName}</span></p>
              <p><strong>Contact Person:</strong> {viewingSupplier.contactPerson || "-"}</p>
              <p><strong>Email:</strong> {viewingSupplier.email || "-"}</p>
              <p><strong>Phone:</strong> {viewingSupplier.phone || "-"}</p>
              <p><strong>Country:</strong> {viewingSupplier.country || "-"}</p>
              <p><strong>Address:</strong> {viewingSupplier.address || "-"}</p>
              <p><strong>Company Profile:</strong> {viewingSupplier.companyProfile || "-"}</p>
              <p><strong>Ownership Details:</strong> {viewingSupplier.ownershipDetails || "-"}</p>
              <p><strong>GST/Tax ID:</strong> {viewingSupplier.gstTaxId || "-"}</p>
              <p><strong>Risk Classification:</strong> {viewingSupplier.riskClassification}</p>
              <p><strong>Status:</strong> {viewingSupplier.status}</p>
              <p><strong>Created:</strong> {dayjs(viewingSupplier.created_at).format("YYYY-MM-DD")}</p>
            </div>
          )}
        </Modal>
      </div>
    </>
  );
}