"use client";

import React, { useCallback, useState } from "react";
import {
  Button,
  Card,
  Col,
  DatePicker,
  Form,
  Input,
  InputNumber,
  Modal,
  Row,
  Space,
  Tooltip,
  Tag,
  Breadcrumb,
  Switch,
  Pagination,
  Dropdown,
  Select,
} from "antd";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleTwoTone,
  WarningOutlined,
  StarOutlined,
  CheckCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import dayjs from "dayjs";
import DataTable from "../components/common/DataTable";
import WidgetCard from "../components/common/WidgetCard";
import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import type { SupplierRating } from "../types/SupplierRating";
import { toast } from "sonner";
import { motion } from "framer-motion";

export default function SupplierRatingPage() {
  const [form] = Form.useForm();
  const [ratings, setRatings] = useState<SupplierRating[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingRating, setEditingRating] = useState<SupplierRating | null>(null);
  const [viewingRating, setViewingRating] = useState<SupplierRating | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    materialDescription: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();

  const openAddModal = () => {
    form.resetFields();
    setEditingRating(null);
    setIsModalOpen(true);
  };

  const openEditModal = (rating: SupplierRating) => {
    form.setFieldsValue({
      ...rating,
      ratingDate: rating.ratingDate ? dayjs(rating.ratingDate) : undefined,
    });
    setEditingRating(rating);
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this rating?",
      onOk: () => {
        setRatings((prev) => prev.filter((r) => r.id !== id));
        toast.success("Rating deleted successfully");
      },
      onCancel: () => {},
    });
  };

  const handleBulkDelete = () => {
    Modal.confirm({
      title: `Are you sure you want to delete ${selectedRows.length} ratings?`,
      onOk: () => {
        setRatings((prev) => prev.filter((r) => !selectedRows.includes(r.id.toString())));
        setSelectedRows([]);
        toast.success("Ratings deleted successfully");
      },
    });
  };

  const computeDerivedValues = (entry: Omit<SupplierRating, 'id' | 'acceptedPercent' | 'overallScore' | 'created_at'>): SupplierRating => {
    const acceptedPercent = entry.lotReceived ? (entry.lotAccepted / entry.lotReceived) * 100 : 0;
    const overallScore =
      (entry.qualityRating || 0) + (entry.deliveryRating || 0) + (entry.serviceRating || 0) + (entry.priceRating || 0);
    return {
      ...entry,
      id: editingRating ? editingRating.id : Date.now().toString(),
      acceptedPercent: +acceptedPercent.toFixed(2),
      overallScore,
      ratingDate: dayjs(entry.ratingDate).format("YYYY-MM-DD"),
      created_at: editingRating ? editingRating.created_at : dayjs().toISOString(),
    };
  };

  const handleFormSubmit = () => {
    form.validateFields().then((values) => {
      const newRating = computeDerivedValues(values);
      if (editingRating) {
        setRatings((prev) => prev.map((r) => (r.id === editingRating.id ? newRating : r)));
        toast.success("Rating updated successfully");
      } else {
        setRatings((prev) => [...prev, newRating]);
        toast.success("Rating added successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
    });
  };

  const filteredRatings = ratings.filter((rating) => {
    const matchesSearch =
      rating.supplier.toLowerCase().includes(searchTerm.toLowerCase()) ||
      rating.materialDescription.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (rating.created_at &&
        (!dateRange[0] ||
          dayjs(rating.created_at).isSame(dateRange[0]) ||
          dayjs(rating.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(rating.created_at).isSame(dateRange[1]) ||
          dayjs(rating.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      !advancedFilters.materialDescription || rating.materialDescription === advancedFilters.materialDescription;
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedRatings = filteredRatings.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<SupplierRating> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredRatings.map((r) => r.id.toString()) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.id.toString()]
                : prev.filter((id) => id !== record.id.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Supplier",
      dataIndex: "supplier",
      key: "supplier",
      sorter: (a, b) => a.supplier.localeCompare(b.supplier),
      render: (text) => <span className="text-green-600 font-medium">{text}</span>,
      filterSearch: true,
      onFilter: (value, record) =>
        record.supplier.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(ratings.map((r) => r.supplier))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Material",
      dataIndex: "materialDescription",
      key: "materialDescription",
      sorter: (a, b) => a.materialDescription.localeCompare(b.materialDescription),
      filterSearch: true,
      onFilter: (value, record) =>
        record.materialDescription.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(ratings.map((r) => r.materialDescription))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Lot Received",
      dataIndex: "lotReceived",
      key: "lotReceived",
      sorter: (a, b) => Number(a.lotReceived ?? 0) - Number(b.lotReceived ?? 0),
    },
    {
      title: "Lot Accepted",
      dataIndex: "lotAccepted",
      key: "lotAccepted",
      sorter: (a, b) => Number(a.lotAccepted ?? 0) - Number(b.lotAccepted ?? 0),
    },
    {
      title: "Accepted %",
      dataIndex: "acceptedPercent",
      key: "acceptedPercent",
      render: (v) => `${v}%`,
      sorter: (a, b) => Number(a.acceptedPercent ?? 0) - Number(b.acceptedPercent ?? 0),
    },
    {
      title: "Overall Score",
      dataIndex: "overallScore",
      key: "overallScore",
      sorter: (a, b) => Number(a.overallScore ?? 0) - Number(b.overallScore ?? 0),
      render: (v) => (
        <span className="flex items-center gap-1">
          {v} <StarOutlined className="text-yellow-500" />
        </span>
      ),
    },
    {
      title: "Rating Date",
      dataIndex: "ratingDate",
      key: "ratingDate",
      sorter: (a, b) =>
        new Date(a.ratingDate ?? 0).getTime() - new Date(b.ratingDate ?? 0).getTime(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "On Time Delivery",
      dataIndex: "onTimeDelivery",
      key: "onTimeDelivery",
      sorter: (a, b) => (a.onTimeDelivery || "").localeCompare(b.onTimeDelivery || ""),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text || "-"}</span>
        </Tooltip>
      ),
    },
    {
      title: "Short Delivery",
      dataIndex: "shortDelivery",
      key: "shortDelivery",
      sorter: (a, b) => (a.shortDelivery || "").localeCompare(b.shortDelivery || ""),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text || "-"}</span>
        </Tooltip>
      ),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingRating(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "supplier", label: "Supplier" },
    { key: "materialDescription", label: "Material" },
    { key: "lotReceived", label: "Lot Received" },
    { key: "lotAccepted", label: "Lot Accepted" },
    { key: "acceptedPercent", label: "Accepted %" },
    { key: "overallScore", label: "Overall Score" },
    { key: "ratingDate", label: "Rating Date" },
    { key: "onTimeDelivery", label: "On Time Delivery" },
    { key: "shortDelivery", label: "Short Delivery" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const RatingCard = ({ rating }: { rating: SupplierRating }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">{rating.supplier}</h3>
        <Tag color="blue">{rating.overallScore} <StarOutlined /></Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Material:</strong> {rating.materialDescription}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Accepted:</strong> {rating.acceptedPercent}%</p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs"><strong>Score:</strong> {rating.overallScore} <StarOutlined className="text-yellow-500" /></p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => setViewingRating(rating)} size="small" className="text-blue-600" />
        </Tooltip>
        <Tooltip title="Edit">
          <Button icon={<EditOutlined />} onClick={() => openEditModal(rating)} size="small" type="primary" className="bg-blue-600 hover:bg-blue-700" />
        </Tooltip>
        <Tooltip title="Delete">
          <Button icon={<DeleteOutlined />} onClick={() => handleDelete(rating.id)} size="small" danger className="text-red-600" />
        </Tooltip>
      </div>
    </div>
  );

  const total = ratings.length;
  const highQuality = ratings.filter((r) => (r.qualityRating || 0) >= 30).length;
  const highAcceptance = ratings.filter((r) => (r.acceptedPercent || 0) >= 80).length;
  const highScore = ratings.filter((r) => (r.overallScore || 0) >= 80).length;

  // Sample trend data for widgets
  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
        <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          Supplier Ratings <span className="text-gray-500 text-base sm:text-lg font-medium">({ratings.length})</span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Evaluate supplier performance based on delivery, quality, and service.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Supplier Ratings",
            },
          ]}
        />

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Ratings"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All supplier rating records"
            trendData={trendData}
            className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900"
          />
          <WidgetCard
            title="High Quality"
            value={highQuality}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Suppliers with quality rating ≥ 30"
            trendData={trendData}
            className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900"
          />
          <WidgetCard
            title="High Acceptance"
            value={highAcceptance}
            icon={<CheckCircleOutlined />}
            color="orange"
            percentage={5}
            description="Suppliers with ≥ 80% acceptance"
            trendData={trendData}
            className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900"
          />
          <WidgetCard
            title="Top Performers"
            value={highScore}
            icon={<StarOutlined />}
            color="red"
            percentage={3}
            description="Suppliers with overall score ≥ 80"
            trendData={trendData}
            className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900"
          />
        </div>

        <div className="mt-4">
          <Card title="Supplier Rating List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier or Material"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Rating
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Material Description"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, materialDescription: value }))
                    }
                    allowClear
                    options={Array.from(new Set(ratings.map((r) => r.materialDescription))).map((value) => ({
                      label: value,
                      value: value,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="supplier_rating_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <DatePicker.RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {advancedFilters.materialDescription && (
                <div className="flex flex-wrap gap-2 mt-2">
                  <Tag
                    closable
                    onClose={() =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        materialDescription: undefined,
                      }))
                    }
                  >
                    Material: {advancedFilters.materialDescription}
                  </Tag>
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
                {paginatedRatings.map((rating) => (
                  <RatingCard key={rating.id} rating={rating} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SupplierRating>
                  columns={filteredColumns}
                  dataSource={paginatedRatings}
                  rowKey="id"
                  scrollX="max-content"
                  className="mt-4"
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>{Math.min(currentPage * pageSize, filteredRatings.length)}</strong> of{" "}
                <strong>{filteredRatings.length}</strong> ratings
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredRatings.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={editingRating ? "Edit Supplier Rating" : "Add Supplier Rating"}
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            className="w-full max-w-[90vw] sm:max-w-[600px] md:max-w-[800px] lg:max-w-[1000px] top-4"
          >
            <Form form={form} layout="vertical" className="px-2 sm:px-4">
              <Row gutter={[16, 16]}>
                <Col xs={24} sm={12}>
                  <Form.Item name="supplier" label="Supplier" rules={[{ required: true, message: "Supplier is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="materialDescription" label="Material Description" rules={[{ required: true, message: "Material Description is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="lotReceived" label="Lot Received" rules={[{ required: true, message: "Lot Received is required" }]}>
                    <InputNumber className="w-full" min={0} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="lotAccepted" label="Lot Accepted" rules={[{ required: true, message: "Lot Accepted is required" }]}>
                    <InputNumber className="w-full" min={0} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="qualityRating" label="Quality Rating (40 pts)" rules={[{ required: true, message: "Quality Rating is required" }]}>
                    <InputNumber className="w-full" max={40} min={0} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="deliveryRating" label="Delivery Rating (30 pts)" rules={[{ required: true, message: "Delivery Rating is required" }]}>
                    <InputNumber className="w-full" max={30} min={0} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="serviceRating" label="Service Rating (20 pts)" rules={[{ required: true, message: "Service Rating is required" }]}>
                    <InputNumber className="w-full" max={20} min={0} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="priceRating" label="Price Rating (10 pts)" rules={[{ required: true, message: "Price Rating is required" }]}>
                    <InputNumber className="w-full" max={10} min={0} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="ratingDate" label="Rating Date" rules={[{ required: true, message: "Rating Date is required" }]}>
                    <DatePicker className="w-full" />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="onTimeDelivery" label="On Time Delivery">
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24}>
                  <Form.Item name="shortDelivery" label="Short Delivery">
                    <Input />
                  </Form.Item>
                </Col>
              </Row>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Supplier Rating Details"
          open={!!viewingRating}
          onCancel={() => setViewingRating(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingRating && (
            <div className="space-y-2 text-sm sm:text-base">
              <p><strong>Supplier:</strong> <span className="text-green-600">{viewingRating.supplier}</span></p>
              <p><strong>Material:</strong> {viewingRating.materialDescription}</p>
              <p><strong>Lot Received:</strong> {viewingRating.lotReceived}</p>
              <p><strong>Lot Accepted:</strong> {viewingRating.lotAccepted}</p>
              <p><strong>Accepted %:</strong> {viewingRating.acceptedPercent}%</p>
              <p><strong>Quality:</strong> {viewingRating.qualityRating}</p>
              <p><strong>Delivery:</strong> {viewingRating.deliveryRating}</p>
              <p><strong>Service:</strong> {viewingRating.serviceRating}</p>
              <p><strong>Price:</strong> {viewingRating.priceRating}</p>
              <p><strong>Overall Score:</strong> {viewingRating.overallScore} <StarOutlined className="text-yellow-500" /></p>
              <p><strong>Rating Date:</strong> {dayjs(viewingRating.ratingDate).format("YYYY-MM-DD")}</p>
              <p><strong>On Time Delivery:</strong> {viewingRating.onTimeDelivery || "-"}</p>
              <p><strong>Short Delivery:</strong> {viewingRating.shortDelivery || "-"}</p>
              <p><strong>Created:</strong> {dayjs(viewingRating.created_at).format("YYYY-MM-DD")}</p>
            </div>
          )}
        </Modal>
      </div>
    </>
  );
}