"use client";

import React, { useCallback, useEffect, useState } from "react";
import {
  Button,
  Form,
  Input,
  Modal,
  Select,
  Space,
  Row,
  Col,
  Tooltip,
  Tag,
  Breadcrumb,
  Card,
  DatePicker,
  Switch,
  Pagination,
  Dropdown,
  Table,
} from "antd";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  EyeOutlined,
  SearchOutlined,
  DownloadOutlined,
  InfoCircleOutlined,
  ExclamationCircleOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import dayjs from "dayjs";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useIsMobile } from "@/data/useIsMobile";
import { FilterValue } from "antd/es/table/interface";
import WidgetCard from "@/app/components/common/WidgetCard";
import ColumnVisibilityToggle from "@/app/components/common/ManageColumn";
import DataTable from "@/app/components/common/DataTable";
import { Item, PurchaseRequisition } from "@/app/types/PurchaseRequisition";


const { RangePicker } = DatePicker;
const statusOptions = ["Pending", "Approved", "Rejected"] as const;
const materialCenterOptions = ["Warehouse A", "Warehouse B", "Central Store"] as const;
const plantDeptOptions = ["Production", "Maintenance", "Quality Control"] as const;

export default function PurchaseRequisitionPage() {
  const [requisitions, setRequisitions] = useState<PurchaseRequisition[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingRequisition, setEditingRequisition] = useState<PurchaseRequisition | null>(null);
  const [form] = Form.useForm();
  const [searchTerm, setSearchTerm] = useState("");
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(null);
  const [filters, setFilters] = useState<Record<string, FilterValue | null>>({});
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    materialCenter: undefined,
    plantDept: undefined,
    status: undefined,
  });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [viewingRequisition, setViewingRequisition] = useState<PurchaseRequisition | null>(null);
  const [loading, setLoading] = useState(false);
  const isMobile = useIsMobile();

  const handleView = (requisition: PurchaseRequisition) => {
    setViewingRequisition(requisition);
  };

  const openAddModal = () => {
    setEditingRequisition(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (requisition: PurchaseRequisition) => {
    setEditingRequisition(requisition);
    form.setFieldsValue({
      ...requisition,
      indentDate: requisition.indentDate ? dayjs(requisition.indentDate) : null,
      items: requisition.items,
    });
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this purchase requisition?",
      onOk: () => {
        setRequisitions((prev) => prev.filter((p) => p.id !== id));
        toast.success("Purchase requisition deleted successfully");
      },
      onCancel: () => {},
    });
  };

  const handleBulkDelete = () => {
    Modal.confirm({
      title: `Are you sure you want to delete ${selectedRows.length} purchase requisitions?`,
      onOk: () => {
        setRequisitions((prev) => prev.filter((p) => !selectedRows.includes(p.id.toString())));
        setSelectedRows([]);
        toast.success("Purchase requisitions deleted successfully");
      },
    });
  };

  const handleFormSubmit = () => {
    form.validateFields().then((values: PurchaseRequisition) => {
      const capitalized = {
        ...values,
        executive: values.executive.charAt(0).toUpperCase() + values.executive.slice(1),
        indentDate: values.indentDate ? dayjs(values.indentDate).toISOString() : dayjs().toISOString(),
      };

      if (editingRequisition) {
        setRequisitions((prev) =>
          prev.map((p) => (p.id === editingRequisition.id ? { ...editingRequisition, ...capitalized } : p))
        );
        toast.success("Purchase requisition updated successfully");
      } else {
        const newRequisition: PurchaseRequisition = {
          ...capitalized,
          created_at: dayjs().toISOString(),
          id: Date.now().toString(),
        };
        setRequisitions((prev) => [...prev, newRequisition]);
        toast.success("Purchase requisition added successfully");
      }

      setIsModalOpen(false);
      form.resetFields();
    });
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Pending: "gold",
      Approved: "green",
      Rejected: "red",
      "Warehouse A": "blue",
      "Warehouse B": "cyan",
      "Central Store": "purple",
      Production: "orange",
      Maintenance: "volcano",
      "Quality Control": "geekblue",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Pending: <PauseCircleOutlined />,
      Approved: <CheckCircleOutlined />,
      Rejected: <CloseCircleOutlined />,
      "Warehouse A": <InfoCircleOutlined />,
      "Warehouse B": <InfoCircleOutlined />,
      "Central Store": <InfoCircleOutlined />,
      Production: <FireOutlined />,
      Maintenance: <WarningOutlined />,
      "Quality Control": <CheckCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredRequisitions = requisitions.filter((requisition) => {
    const matchesSearch =
      requisition.executive.toLowerCase().includes(searchTerm.toLowerCase()) ||
      requisition.indentNo.toLowerCase().includes(searchTerm.toLowerCase()) ||
      requisition.requisitionNo.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (requisition.created_at &&
        (!dateRange[0] ||
          dayjs(requisition.created_at).isSame(dateRange[0]) ||
          dayjs(requisition.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(requisition.created_at).isSame(dateRange[1]) ||
          dayjs(requisition.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.materialCenter || requisition.materialCenter === advancedFilters.materialCenter) &&
      (!advancedFilters.plantDept || requisition.plantDept === advancedFilters.plantDept) &&
      (!advancedFilters.status || requisition.status === advancedFilters.status);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedRequisitions = filteredRequisitions.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const itemColumns: ColumnsType<Item> = [
    {
      title: "Description",
      dataIndex: "description",
      key: "description",
      render: (text) => <span className="text-green-600 font-medium">{text}</span>,
    },
    {
      title: "Quantity Required",
      dataIndex: "quantityRequired",
      key: "quantityRequired",
    },
    {
      title: "Stock in Hand",
      dataIndex: "stockInHand",
      key: "stockInHand",
    },
    {
      title: "Ordered Quantity",
      dataIndex: "orderedQuantity",
      key: "orderedQuantity",
    },
    {
      title: "Unit",
      dataIndex: "unit",
      key: "unit",
    },
  ];

  const columns: ColumnsType<PurchaseRequisition> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredRequisitions.map((p) => p.id.toString()) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.id.toString()]
                : prev.filter((id) => id !== record.id.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Indent No",
      dataIndex: "indentNo",
      key: "indentNo",
      sorter: (a, b) => a.indentNo.localeCompare(b.indentNo),
    },
    {
      title: "Executive",
      dataIndex: "executive",
      key: "executive",
      sorter: (a, b) => a.executive.localeCompare(b.executive),
      render: (text) => <span className="text-green-600 font-medium">{text}</span>,
      filterSearch: true,
      onFilter: (value, record) =>
        record.executive.toLowerCase().includes((value as string).toLowerCase()),
      filters: requisitions.map((p) => ({ text: p.executive, value: p.executive })),
    },
    {
      title: "Plant/Dept",
      dataIndex: "plantDept",
      key: "plantDept",
      render: renderTag,
      filters: Array.from(new Set(requisitions.map((p) => p.plantDept))).map((value) => ({
        text: value,
        value: value,
      })),
      onFilter: (value, record) => record.plantDept === value,
    },
    {
      title: "Indent Date",
      dataIndex: "indentDate",
      key: "indentDate",
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
      sorter: (a, b) => dayjs(a.indentDate).unix() - dayjs(b.indentDate).unix(),
    },
    {
      title: "Material Center",
      dataIndex: "materialCenter",
      key: "materialCenter",
      render: renderTag,
      filters: Array.from(new Set(requisitions.map((p) => p.materialCenter))).map((value) => ({
        text: value,
        value: value,
      })),
      onFilter: (value, record) => record.materialCenter === value,
    },
    {
      title: "Requisition No",
      dataIndex: "requisitionNo",
      key: "requisitionNo",
      sorter: (a, b) => a.requisitionNo.localeCompare(b.requisitionNo),
    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      render: renderTag,
      filters: Array.from(new Set(requisitions.map((p) => p.status))).map((value) => ({
        text: value,
        value: value,
      })),
      onFilter: (value, record) => record.status === value,
    },
    {
      title: "Any Other",
      dataIndex: "anyOther",
      key: "anyOther",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => handleView(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id.toString())}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "indentNo", label: "Indent No" },
    { key: "executive", label: "Executive" },
    { key: "plantDept", label: "Plant/Dept" },
    { key: "indentDate", label: "Indent Date" },
    { key: "materialCenter", label: "Material Center" },
    { key: "requisitionNo", label: "Requisition No" },
    { key: "status", label: "Status" },
    { key: "anyOther", label: "Any Other" },
    { key: "actions", label: "Action" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const RequisitionCard = ({ requisition }: { requisition: PurchaseRequisition }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">{requisition.indentNo}</h3>
        <Tag color="blue">{requisition.plantDept}</Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Executive:</strong> {requisition.executive}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Status:</strong> {renderTag(requisition.status)}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Material Center:</strong> {renderTag(requisition.materialCenter)}</p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs"><strong>Created:</strong> {dayjs(requisition.created_at).format("YYYY-MM-DD")}</p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => handleView(requisition)} size="small" className="text-blue-600" />
        </Tooltip>
        <Tooltip title="Edit">
          <Button icon={<EditOutlined />} onClick={() => openEditModal(requisition)} size="small" type="primary" className="bg-blue-600 hover:bg-blue-700" />
        </Tooltip>
        <Tooltip title="Delete">
          <Button icon={<DeleteOutlined />} onClick={() => handleDelete(requisition.id.toString())} size="small" danger className="text-red-600" />
        </Tooltip>
      </div>
    </div>
  );

  const total = requisitions.length;
  const pending = requisitions.filter((p) => p.status === "Pending").length;
  const approved = requisitions.filter((p) => p.status === "Approved").length;
  const rejected = requisitions.filter((p) => p.status === "Rejected").length;

  return (
    <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
      <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
        Purchase Requisitions <span className="text-gray-500 text-base sm:text-lg font-medium">({requisitions.length})</span>
      </h2>
      <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
        Manage purchase requisitions for material procurement.
      </p>
      <Breadcrumb
        className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
        separator=">"
        items={[
          {
            title: (
              <Tooltip title="Dashboard">
                <HomeOutlined />
              </Tooltip>
            ),
            href: "/dashboard",
          },
          {
            title: "Procurement",
          },
          {
            title: "Purchase Requisitions",
          },
        ]}
      />

      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
        <WidgetCard
          title="Total Requisitions"
          value={total}
          icon={<AppstoreOutlined />}
          color="green"
          percentage={10}
          description="All purchase requisitions"
        />
        <WidgetCard
          title="Pending"
          value={pending}
          icon={<PauseCircleOutlined />}
          color="gold"
          percentage={5}
          description="Awaiting approval"
        />
        <WidgetCard
          title="Approved"
          value={approved}
          icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
          color="blue"
          percentage={8}
          description="Approved requisitions"
        />
        <WidgetCard
          title="Rejected"
          value={rejected}
          icon={<StopOutlined />}
          color="red"
          percentage={-2}
          description="Rejected requisitions"
        />
      </div>

      <div className="mt-4">
        <Card title="Requisition List" className="p-0 mb-6 shadow-sm">
          <div className="p-4 flex flex-col gap-4">
            <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
              <Input
                prefix={<SearchOutlined />}
                allowClear
                placeholder="Search by Indent No, Executive, or Requisition No"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
              />
              <Dropdown
                className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                menu={{
                  items: [
                    { label: "Export CSV", key: "csv" },
                    { label: "Export PDF", key: "pdf" },
                  ],
                }}
              >
                <Button icon={<DownloadOutlined />}>Export</Button>
              </Dropdown>
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={openAddModal}
                className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
              >
                Add Requisition
              </Button>
            </div>
            <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                <Select
                  placeholder="Material Center"
                  className="w-full max-w-[160px]"
                  onChange={(value) =>
                    setAdvancedFilters((prev) => ({ ...prev, materialCenter: value }))
                  }
                  allowClear
                  options={materialCenterOptions.map((v) => ({ label: v, value: v }))}
                />
                <Select
                  placeholder="Plant/Dept"
                  className="w-full max-w-[160px]"
                  onChange={(value) =>
                    setAdvancedFilters((prev) => ({ ...prev, plantDept: value }))
                  }
                  allowClear
                  options={plantDeptOptions.map((v) => ({ label: v, value: v }))}
                />
                <Select
                  placeholder="Status"
                  className="w-full max-w-[160px]"
                  onChange={(value) =>
                    setAdvancedFilters((prev) => ({ ...prev, status: value }))
                  }
                  allowClear
                  options={statusOptions.map((v) => ({ label: v, value: v }))}
                />
              </div>
              <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                <ColumnVisibilityToggle
                  columns={columnKeys}
                  labels={labelsMap}
                  storageKey="requisition_table_visibility"
                  onChange={handleVisibleColumnChange}
                />
                <RangePicker
                  value={dateRange}
                  onChange={(dates) =>
                    setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                  }
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                />
                <Button
                  onClick={handleBulkDelete}
                  disabled={selectedRows.length === 0}
                  danger
                  className="w-full sm:w-auto"
                >
                  Bulk Delete
                </Button>
                <div className="hidden sm:block">
                  <Switch
                    checkedChildren={<AppstoreOutlined />}
                    unCheckedChildren={<TableOutlined />}
                    checked={isGridView}
                    onChange={setIsGridView}
                    className="w-full sm:w-auto"
                  />
                </div>
              </div>
            </div>

            {(advancedFilters.materialCenter ||
              advancedFilters.plantDept ||
              advancedFilters.status) && (
              <div className="flex flex-wrap gap-2 mt-2">
                {advancedFilters.materialCenter && (
                  <Tag
                    closable
                    onClose={() =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        materialCenter: undefined,
                      }))
                    }
                  >
                    Material Center: {advancedFilters.materialCenter}
                  </Tag>
                )}
                {advancedFilters.plantDept && (
                  <Tag
                    closable
                    onClose={() =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        plantDept: undefined,
                      }))
                    }
                  >
                    Plant/Dept: {advancedFilters.plantDept}
                  </Tag>
                )}
                {advancedFilters.status && (
                  <Tag
                    closable
                    onClose={() =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        status: undefined,
                      }))
                    }
                  >
                    Status: {advancedFilters.status}
                  </Tag>
                )}
              </div>
            )}
          </div>

          {isGridView || isMobile ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
              {paginatedRequisitions.map((requisition) => (
                <RequisitionCard key={requisition.id} requisition={requisition} />
              ))}
            </div>
          ) : (
            <div className="overflow-auto">
              <DataTable<PurchaseRequisition>
                columns={filteredColumns}
                dataSource={paginatedRequisitions}
                rowKey="id"
                scrollX="max-content"
                className="mt-4"
              />
            </div>
          )}
          <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
            <div className="mb-2 sm:mb-0">
              Showing{" "}
              <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
              <strong>{Math.min(currentPage * pageSize, filteredRequisitions.length)}</strong> of{" "}
              <strong>{filteredRequisitions.length}</strong> requisitions
            </div>
            <Pagination
              current={currentPage}
              pageSize={pageSize}
              total={filteredRequisitions.length}
              onChange={(page, size) => {
                setCurrentPage(page);
                setPageSize(size);
              }}
              showSizeChanger
              pageSizeOptions={["10", "30", "50", "100"]}
              size="small"
            />
          </div>
        </Card>
      </div>

      <motion.div initial={{ opacity: 0, scale: 0.95 }} animate={{ opacity: 1, scale: 1 }} exit={{ opacity: 0, scale: 0.95 }}>
        <Modal
          title={editingRequisition ? "Edit Purchase Requisition" : "Add Purchase Requisition"}
          open={isModalOpen}
          onOk={handleFormSubmit}
          onCancel={() => {
            setIsModalOpen(false);
            form.resetFields();
          }}
          destroyOnHidden
          className="w-full max-w-[90vw] sm:max-w-[600px] md:max-w-[800px] lg:max-w-[1000px] top-4"
        >
          <Form form={form} layout="vertical" className="px-2 sm:px-4">
            <Row gutter={[16, 16]}>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="indentNo"
                  label="Indent No"
                  rules={[{ required: true, message: "Indent No is required" }]}
                >
                  <Input />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="executive"
                  label="Executive"
                  rules={[{ required: true, message: "Executive is required" }]}
                >
                  <Input />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="plantDept"
                  label="Plant/Dept"
                  rules={[{ required: true, message: "Plant/Dept is required" }]}
                >
                  <Select options={plantDeptOptions.map((v) => ({ label: v, value: v }))} />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="indentDate"
                  label="Indent Date"
                  rules={[{ required: true, message: "Indent Date is required" }]}
                >
                  <DatePicker className="w-full" />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="materialCenter"
                  label="Material Center"
                  rules={[{ required: true, message: "Material Center is required" }]}
                >
                  <Select options={materialCenterOptions.map((v) => ({ label: v, value: v }))} />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="requisitionNo"
                  label="Requisition No"
                  rules={[{ required: true, message: "Requisition No is required" }]}
                >
                  <Input />
                </Form.Item>
              </Col>
              <Col xs={24}>
                <Form.Item name="anyOther" label="Any Other">
                  <Input.TextArea rows={3} />
                </Form.Item>
              </Col>
              <Col xs={24}>
                <Form.Item
                  name="status"
                  label="Status"
                  rules={[{ required: true, message: "Status is required" }]}
                >
                  <Select options={statusOptions.map((v) => ({ label: v, value: v }))} />
                </Form.Item>
              </Col>
              <Col xs={24}>
                <Form.List name="items">
                  {(fields, { add, remove }) => (
                    <>
                      <div className="flex justify-between items-center mb-2">
                        <h3 className="text-lg font-semibold">Items</h3>
                        <Button
                          type="dashed"
                          onClick={() => add()}
                          icon={<PlusOutlined />}
                          className="mb-2"
                        >
                          Add Item
                        </Button>
                      </div>
                      <Table
                        dataSource={fields}
                        columns={[
                          {
                            title: "Description",
                            render: (_, field) => (
                              <Form.Item
                                name={[field.name, "description"]}
                                rules={[{ required: true, message: "Description is required" }]}
                                noStyle
                              >
                                <Input />
                              </Form.Item>
                            ),
                          },
                          {
                            title: "Quantity Required",
                            render: (_, field) => (
                              <Form.Item
                                name={[field.name, "quantityRequired"]}
                                rules={[{ required: true, message: "Quantity is required" }]}
                                noStyle
                              >
                                <Input type="number" min={0} />
                              </Form.Item>
                            ),
                          },
                          {
                            title: "Stock in Hand",
                            render: (_, field) => (
                              <Form.Item
                                name={[field.name, "stockInHand"]}
                                rules={[{ required: true, message: "Stock is required" }]}
                                noStyle
                              >
                                <Input type="number" min={0} />
                              </Form.Item>
                            ),
                          },
                          {
                            title: "Ordered Quantity",
                            render: (_, field) => (
                              <Form.Item
                                name={[field.name, "orderedQuantity"]}
                                rules={[{ required: true, message: "Ordered quantity is required" }]}
                                noStyle
                              >
                                <Input type="number" min={0} />
                              </Form.Item>
                            ),
                          },
                          {
                            title: "Unit",
                            render: (_, field) => (
                              <Form.Item
                                name={[field.name, "unit"]}
                                rules={[{ required: true, message: "Unit is required" }]}
                                noStyle
                              >
                                <Input />
                              </Form.Item>
                            ),
                          },
                          {
                            title: "Action",
                            render: (_, __, index) => (
                              <Button
                                icon={<DeleteOutlined />}
                                onClick={() => remove(index)}
                                danger
                                size="small"
                              />
                            ),
                          },
                        ]}
                        pagination={false}
                        rowKey={(field) => field.key}
                        className="mb-4"
                      />
                    </>
                  )}
                </Form.List>
              </Col>
            </Row>
          </Form>
        </Modal>
      </motion.div>

      <Modal
        title="Purchase Requisition Details"
        open={!!viewingRequisition}
        onCancel={() => setViewingRequisition(null)}
        footer={null}
        className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
      >
        {viewingRequisition && (
          <div className="space-y-2 text-sm sm:text-base">
            <p><strong>Indent No:</strong> {viewingRequisition.indentNo}</p>
            <p><strong>Executive:</strong> <span className="text-green-600">{viewingRequisition.executive}</span></p>
            <p><strong>Plant/Dept:</strong> {viewingRequisition.plantDept}</p>
            <p><strong>Indent Date:</strong> {dayjs(viewingRequisition.indentDate).format("YYYY-MM-DD")}</p>
            <p><strong>Material Center:</strong> {viewingRequisition.materialCenter}</p>
            <p><strong>Requisition No:</strong> {viewingRequisition.requisitionNo}</p>
            <p><strong>Any Other:</strong> {viewingRequisition.anyOther}</p>
            <p><strong>Status:</strong> {viewingRequisition.status}</p>
            <p><strong>Created:</strong> {dayjs(viewingRequisition.created_at).format("YYYY-MM-DD")}</p>
            <h3 className="text-lg font-semibold mt-4">Items</h3>
            <Table
              columns={itemColumns}
              dataSource={viewingRequisition.items}
              pagination={false}
              rowKey={(record, index) => `${index}`}
              className="mt-2"
            />
          </div>
        )}
      </Modal>
    </div>
  );
}