'use client';

import { LineChart, Line, ResponsiveContainer, AreaChart, Area } from 'recharts';
import { ReactNode } from 'react';
import clsx from 'clsx';
import CountUp from 'react-countup';


interface WidgetCardProps {
  title: string;
  value: number;
  icon?: ReactNode;
  trendData?: { value: number }[];
  color?: 'green' | 'blue' | 'red' | 'orange' | 'gray' | 'gold';
  description?: string;
  className?: string;
  percentage?: number; // e.g., +12% growth
}

const getColor = (color: string) => {
  switch (color) {
    case 'green':
      return {
        text: 'text-green-600',
        bg: 'bg-green-100 dark:bg-green-900/40',
        stroke: '#22c55e',
        badge: 'bg-green-200 text-green-800',
      };
    case 'blue':
      return {
        text: 'text-blue-600',
        bg: 'bg-blue-100 dark:bg-blue-900/40',
        stroke: '#3b82f6',
        badge: 'bg-blue-200 text-blue-800',
      };
    case 'red':
      return {
        text: 'text-red-600',
        bg: 'bg-red-100 dark:bg-red-900/40',
        stroke: '#ef4444',
        badge: 'bg-red-200 text-red-800',
      };
    case 'orange':
      return {
        text: 'text-orange-600',
        bg: 'bg-orange-100 dark:bg-orange-900/40',
        stroke: '#f97316',
        badge: 'bg-orange-200 text-orange-800',
      };
    default:
      return {
        text: 'text-gray-700 dark:text-gray-200',
        bg: 'bg-gray-100 dark:bg-gray-800/60',
        stroke: '#6b7280',
        badge: 'bg-gray-200 text-gray-800',
      };
  }
};

export default function WidgetCard({
  title,
  value,
  icon,
  trendData,
  color = 'green',
  description,
  className,
  percentage,
}: WidgetCardProps) {
  const colors = getColor(color);

  return (
    <div
  className={clsx(
    'rounded-xl p-4 bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700 flex flex-col justify-between',
    'transition-all duration-300 ease-in-out transform hover:scale-[1.02] hover:shadow-lg',
    className
  )}
>

      <div className="flex items-start justify-between mb-2">
        <div>
          <h4 className="text-sm font-medium text-gray-500 dark:text-gray-400">{title}</h4>
          <p className={`text-3xl font-bold ${colors.text}`}>
  <CountUp end={value} duration={1.5} separator="," />
</p>
        </div>
        {icon && (
          <div className={`p-2 rounded-full ${colors.bg} ${colors.text}`}>
            {icon}
          </div>
        )}
      </div>

      {description && (
        <p className="text-xs text-gray-500 dark:text-gray-400 mb-2">{description}</p>
      )}

      {percentage !== undefined && (
        <span
          className={`mt-1 inline-block text-xs font-medium px-2 py-1 rounded ${colors.badge}`}
        >
          {percentage > 0 ? '+' : ''}
          {percentage}% from last month
        </span>
      )}
      {trendData && (
        <div className="h-12 mt-2">
          <ResponsiveContainer width="100%" height="100%">
  <AreaChart data={trendData}>
    <defs>
      <linearGradient id="colorTrend" x1="0" y1="0" x2="0" y2="1">
        <stop offset="0%" stopColor={colors.stroke} stopOpacity={0.4} />
        <stop offset="100%" stopColor={colors.stroke} stopOpacity={0.05} />
      </linearGradient>
    </defs>
    <Area
      type="monotone"
      dataKey="value"
      stroke={colors.stroke}
      fill="url(#colorTrend)"
      strokeWidth={2}
      dot={false}
    />
  </AreaChart>
</ResponsiveContainer>

        </div>
      )}
    </div>
  );
}
