'use client';

import Link from 'next/link';
import { usePathname, useRouter } from 'next/navigation';
import { useMemo, useState } from 'react';
import {
  HomeIcon,
  PackageIcon,
  WorkflowIcon,
  FileTextIcon,
  ClipboardListIcon,
  LogOutIcon,
  ArrowLeftIcon,
  ArrowRightIcon,
  WarehouseIcon,
  UserCheckIcon,
  FlaskConicalIcon,
  ShieldCheckIcon,
  FileLineChartIcon,
  UserCircleIcon,
  ShoppingCartIcon,
  FilePlusIcon,
  SearchIcon,
} from 'lucide-react';
import { useAuth } from '@/context/AuthContext';
import Image from 'next/image';
import { motion } from 'framer-motion';
import { SidebarData } from '@/data/sidebarData';
import { highlightMatch } from './common/HighLightMatch';
import { useSearch } from '@/context/SearchContext';

const iconMap: Record<string, React.ReactNode> = {
  HomeIcon: <HomeIcon className="w-5 h-5" />,
  PackageIcon: <PackageIcon className="w-5 h-5" />,
  WorkflowIcon: <WorkflowIcon className="w-5 h-5" />,
  FileTextIcon: <FileTextIcon className="w-5 h-5" />,
  ClipboardListIcon: <ClipboardListIcon className="w-5 h-5" />,
  WarehouseIcon: <WarehouseIcon className="w-5 h-5" />,
  SupplierVisitIcon: <UserCheckIcon className="w-5 h-5" />,
  SampleTestIcon: <FlaskConicalIcon className="w-5 h-5" />,
  SupplierApprovalIcon: <ShieldCheckIcon className="w-5 h-5" />,
  SupplierRatingIcon: <FileLineChartIcon className="w-5 h-5" />,
  SupplierActionsIcon: <ClipboardListIcon className="w-5 h-5" />,
  ShoppingCartIcon: <ShoppingCartIcon className="w-5 h-5" />,
  FilePlusIcon: <FilePlusIcon className="w-5 h-5" />,
};

interface SidebarProps {
  onMobileToggle?: () => void;
}

export default function Sidebar({ onMobileToggle }: SidebarProps = {}) {
  const pathname = usePathname();
  const { logout } = useAuth();
  const router = useRouter();
  const [collapsed, setCollapsed] = useState(false);
  const [hovered, setHovered] = useState(false);
  const { searchTerm, setSearchTerm } = useSearch();


  // Determine if sidebar should be expanded based on collapsed state or hover
  const isSidebarExpanded = !collapsed || hovered;

  const handleLogout = () => {
    logout();
    router.push('/login');
    onMobileToggle?.();
  };

  const renderedSidebarItems = useMemo(() =>
    SidebarData.flatMap(section =>
      section.submenuItems
        .filter(({ label }) =>
          label.toLowerCase().includes(searchTerm.toLowerCase())
        )
        .map(({ label, link, icon }) => ({
          key: link,
          label,
          link,
          iconComponent: iconMap[icon],
          isActive: pathname === link,
          section: section.label,
        }))
    ), [pathname, searchTerm]
  );


  return (
   <motion.aside
  animate={{ width: isSidebarExpanded ? 256 : 80 }}
  transition={{ type: 'spring', stiffness: 300, damping: 30 }}
  className={`bg-white dark:bg-gray-800 shadow-md transition-colors duration-300 flex flex-col
    ${isSidebarExpanded ? 'w-64' : 'w-20'}
    h-screen lg:sticky top-0 left-0  overflow-x-hidden  overflow-y-auto scrollbar-thin scrollbar-thumb-rounded scrollbar-thumb-gray-400 dark:scrollbar-thumb-gray-600 hover:scrollbar-thumb-gray-500`}
  onMouseEnter={() => setHovered(true)}
  onMouseLeave={() => setHovered(false)}
>

      {/* Logo and Collapse Button */}
      <div className="sticky top-0 z-10 bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700 flex flex-col items-center px-4 pt-4 pb-4">
        <Image
          src={isSidebarExpanded ? '/image.png' : '/favicon.ico'}
          alt="Logo"
          width={isSidebarExpanded ? 60 : 32}
          height={40}
          priority
          className="h-10 w-auto"
        />
        {/* {isSidebarExpanded && (
          <span className="text-xl font-semibold text-green-700 dark:text-green-300 mt-2">
            QMS
          </span>
        )} */}
        <motion.button
          whileHover={{ scale: 1.3 }}
          whileTap={{ scale: 0.9 }}
          onClick={() => setCollapsed(!collapsed)}
          className="absolute -right-3 top-4 z-50 bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-full p-1 shadow-md hover:bg-gray-100 dark:hover:bg-gray-600"
          title={collapsed ? 'Expand' : 'Collapse'}
        >
          {collapsed ? (
            <ArrowRightIcon className="w-4 h-4 text-gray-600 dark:text-gray-300" />
          ) : (
            <ArrowLeftIcon className="w-4 h-4 text-gray-600 dark:text-gray-300" />
          )}
        </motion.button>
      </div>
      <div className="px-3 pt-3 pb-3">
        <div className="relative">
          <span className="absolute inset-y-0 left-2 flex items-center text-gray-500 dark:text-gray-400">
            <SearchIcon className="w-4 h-4 sm:w-5 sm:h-5" />
          </span>
          <input
            type="text"
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            placeholder="Search..."
            className="w-full pl-8 text-sm px-3 py-1.5 rounded-md bg-gray-100 dark:bg-gray-700 text-gray-900 dark:text-gray-100 placeholder:text-gray-500 dark:placeholder:text-gray-400 focus:outline-none focus:ring-2 focus:ring-green-400"
          />
        </div>
      </div>

      {/* Navigation */}
      <ul className="border-t border-gray-200 dark:border-gray-700 flex-1 space-y-2 pt-2 px-2">
        {SidebarData.map((section) => (
          <li key={section.label}>
            {isSidebarExpanded && (
              <h3 className="px-3 py-2 text-[0.7rem] font-bold text-gray-500 dark:text-gray-400">
                {section.label}
              </h3>
            )}
            <ul className="space-y-1">
              {renderedSidebarItems
                .filter((item) => item.section === section.label)
                .map(({ key, label, link, iconComponent, isActive }) => (
                  <li key={key}>
                    <Link
                      href={link}
                      className={`group flex items-center gap-3 px-3 py-2 rounded-md font-medium transition-all ${isActive
                        ? 'bg-green-100 dark:bg-green-900 text-green-700 dark:text-green-300'
                        : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 hover:text-green-600 dark:hover:text-green-400'
                        }`}
                      title={isSidebarExpanded ? '' : label}
                      onClick={onMobileToggle}
                    >
                      <span className="text-green-600 dark:text-green-400 group-hover:bg-green-100 dark:group-hover:bg-green-900 group-hover:text-green-600 dark:group-hover:text-green-400 group-hover:scale-130 transition-transform duration-200 flex items-center justify-center w-5 h-5">
                        {iconComponent}
                      </span>
                      {isSidebarExpanded && (
                        <span className="text-sm">
                          {highlightMatch(label, searchTerm)}
                        </span>
                      )}
                    </Link>
                  </li>
                ))}
            </ul>
          </li>
        ))}
      </ul>

      {/* User Profile Section */}
      <Link
        href="/profile"
        className="px-3 pt-4 pb-3 border-t border-gray-200 dark:border-gray-700 hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors duration-200 block"
      >
        <div className="flex items-center gap-3">
          <UserCircleIcon className="w-8 h-8 text-gray-600 dark:text-gray-300 group-hover:text-green-600 dark:group-hover:text-green-400" />
          {isSidebarExpanded && (
            <div className="flex flex-col">
              <span className="text-sm font-medium text-gray-700 dark:text-gray-200">Admin</span>
              <span className="text-xs text-gray-500 dark:text-gray-400">admin@example.com</span>
            </div>
          )}
        </div>
      </Link>

      {/* Logout */}
      <div className="px-3 pb-4">
        <motion.button
          whileHover={{ scale: 1.05 }}
          whileTap={{ scale: 0.95 }}
          onClick={handleLogout}
          className="flex items-center gap-3 w-full px-3 py-2 text-red-500 dark:text-red-400 hover:text-red-600 dark:hover:text-red-300 rounded-md"
          title={isSidebarExpanded ? '' : 'Logout'}
        >
          <LogOutIcon className="w-5 h-5" />
          {isSidebarExpanded && <span className="text-sm">Logout</span>}
        </motion.button>
      </div>
    </motion.aside>
  );
}