"use client";

import React, { useCallback, useEffect, useMemo, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import Col from "antd/es/col";
import "antd/es/col/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Row from "antd/es/row";
import "antd/es/row/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  AppstoreOutlined,
  TableOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
  InfoCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
//import WidgetCard from "../components/common/WidgetCard";
//import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import {
  createSupplierVisit,
  deleteSupplierVisitById,
  downloadSupplierVisit,
  getAllSupplierVisits,
  SupplierVisit,
  updateSupplierVisit,
} from "../api/suppliervisit/suppliervisit";
import { getAllSuppliers, Supplier } from "../api/suppliers/suppliers";
//import ConfirmDialog from "../components/common/ConfirmDialog";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import { ClipboardCheck, ClipboardList, UserCheckIcon } from "lucide-react";

// ✅ WidgetCard
const WidgetCard = dynamic(() => import("../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});

// ✅ ColumnVisibilityToggle
const ColumnVisibilityToggle = dynamic(
  () => import("../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);

// ✅ ConfirmDialog
const ConfirmDialog = dynamic(
  () => import("../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

const { RangePicker } = DatePicker;
const visitTypes = ["Initial", "Follow Up", "Routine"] as const;
const statusOptions = ["Active", "Inactive", "BlackListed"] as const;
const riskOptions = ["Low", "Medium", "High"] as const;

export default function SupplierVisitPage() {
  const [form] = Form.useForm();
  const [visits, setVisits] = useState<SupplierVisit[]>([]);
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [loading, setLoading] = useState(false);
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });

  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingVisit, setEditingVisit] = useState<SupplierVisit | null>(null);
  const [viewingVisit, setViewingVisit] = useState<SupplierVisit | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    visitType: undefined,
    riskClassification: undefined,
    status: undefined,
  });
  const [selectedFile, setSelectedFile] = useState<File | null>(null); // State for file
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();

  useEffect(() => {
    const fetchVisits = async () => {
      setLoading(true);
      try {
        const data = await getAllSupplierVisits();
        setVisits(data);
      } catch (error) {
        toast.error("Failed to load supplier visits.");
      } finally {
        setLoading(false);
      }
    };

    const fetchSuppliers = async () => {
      const data = await getAllSuppliers(); // import this from your supplier API
      setSuppliers(data);
    };

    fetchSuppliers();
    fetchVisits();
  }, []);
  const getSupplierDisplay = (id: string) => {
    const supplier = suppliers.find((s) => s.supplier_id === id);
    if (!supplier) return id;

    const name = supplier.company_name;
    const contact = supplier.contact_person || "N/A";
    return `${name} - ${contact} (${id})`;
  };

  const openAddModal = () => {
    form.resetFields();
    setEditingVisit(null);
    setIsModalOpen(true);
  };

  const openEditModal = (visit: SupplierVisit) => {
    form.setFieldsValue({
      supplier: visit.supplierId,
      visitType: visit.visit_type,
      place: visit.place,
      material: visit.material,
      scheduledMonth: visit.scheduled_month
        ? dayjs(visit.scheduled_month, "YYYY-MM-DD")
        : undefined,
      auditorName: visit.auditor_name,
      completionDate: visit.completion_date
        ? dayjs(visit.completion_date, "YYYY-MM-DD")
        : undefined,
      findings: visit.description,
      riskClassification: visit.risk_classification,
      status: visit.Status,
      objective: visit.objective,
      followUpDate: visit.follow_up_date
        ? dayjs(visit.follow_up_date, "YYYY-MM-DD")
        : undefined,
      isConfidential: visit.is_confidential,
    });

    setEditingVisit(visit);
    setSelectedFile(null); // Reset file state for new uploads
    setIsModalOpen(true);
  };

  const handleDelete = useCallback(
    (supplierId: string) => {
      setConfirmDialog({
        open: true,
        title: "Delete Audit",
        message: "Are you sure you want to delete this audit?",
        onConfirm: async () => {
          try {
            await deleteSupplierVisitById(supplierId);
            const refreshed = await getAllSupplierVisits();
            setVisits(refreshed);
            toast.success("Supplier Audits deleted successfully");
          } catch {
            toast.error("Failed to delete audit");
          }
        },
      });
    },
    [setVisits]
  );

  const handleBulkDelete = useCallback(() => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Supplier Audits",
      message: `Are you sure you want to delete ${selectedRows.length} audits?`,
      onConfirm: async () => {
        try {
          await Promise.all(
            selectedRows.map((id) => deleteSupplierVisitById(id))
          );
          const refreshed = await getAllSupplierVisits();
          setVisits(refreshed);
          setSelectedRows([]);
          toast.success("Supplier Audits deleted successfully");
        } catch {
          toast.error("Failed to delete selected supplier audits");
        }
      },
    });
  }, [selectedRows, setVisits, setSelectedRows]);

  // Handle file input change
  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0] || null;
    setSelectedFile(file);
  };

  const handleFormSubmit = useCallback(() => {
    form.validateFields().then(async (values) => {
      try {
        const payload: SupplierVisit = {
          ...values,
          supplierId: values.supplier,
          place: values.place,
          material: values.material,
          auditor_name: values.auditorName,
          scheduled_month: dayjs(values.scheduledMonth).format("YYYY-MM-DD"),
          completion_date: values.completionDate
            ? dayjs(values.completionDate).format("YYYY-MM-DD")
            : "",
          Status: values.status,
          risk_classification: values.riskClassification,
          description: values.findings || "",
          visit_type: values.visitType,
          objective: values.objective,
          follow_up_date: values.followUpDate
            ? dayjs(values.followUpDate).format("YYYY-MM-DD")
            : undefined,
          is_confidential: values.isConfidential || false,
          attacted_file: selectedFile,
        };
        const alreadyExists = visits.some(
          (v) => v.supplierId === values.supplier && !editingVisit
        );

        if (alreadyExists) {
          toast.warning("This supplier already has a Supplier Audit record.");
          return;
        }

        if (editingVisit) {
          await updateSupplierVisit(editingVisit.supplierId, payload);
          toast.success("Supplier Audit updated successfully");
        } else {
          await createSupplierVisit(payload);
          toast.success("Supplier Audit created successfully");
        }
        setIsModalOpen(false);
        setSelectedFile(null);
        form.resetFields();
        const refreshed = await getAllSupplierVisits();
        setVisits(refreshed);
      } catch (error) {
        toast.error("Failed to submit form." + error);
      }
    });
  }, [
    form,
    visits,
    editingVisit,
    selectedFile,
    setVisits,
    setIsModalOpen,
    setSelectedFile,
  ]);

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Initial: "blue",
      "Follow Up": "cyan",
      Routine: "purple",
      Low: "green",
      Medium: "orange",
      High: "red",
      Active: "green",
      Inactive: "gold",
      BlackListed: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Initial: <InfoCircleOutlined />,
      "Follow Up": <CheckCircleOutlined />,
      Routine: <CheckCircleOutlined />,
      Low: <CheckCircleOutlined />,
      Medium: <PauseCircleOutlined />,
      High: <FireOutlined />,
      Active: <CheckCircleOutlined />,
      Inactive: <MinusCircleOutlined />,
      BlackListed: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredVisits = useMemo(() => {
    return visits.filter((visit) => {
      const matchesSearch =
        visit.supplierId.toLowerCase().includes(searchTerm.toLowerCase()) ||
        visit.place.toLowerCase().includes(searchTerm.toLowerCase());
      const matchesDate =
        !dateRange ||
        (visit.created_at &&
          (!dateRange[0] ||
            dayjs(visit.created_at).isSame(dateRange[0]) ||
            dayjs(visit.created_at).isAfter(dateRange[0])) &&
          (!dateRange[1] ||
            dayjs(visit.created_at).isSame(dateRange[1]) ||
            dayjs(visit.created_at).isBefore(dateRange[1])));
      const matchesAdvanced =
        (!advancedFilters.visitType ||
          visit.visit_type === advancedFilters.visitType) &&
        (!advancedFilters.riskClassification ||
          visit.risk_classification === advancedFilters.riskClassification) &&
        (!advancedFilters.status || visit.Status === advancedFilters.status);
      return matchesSearch && matchesDate && matchesAdvanced;
    });
  }, [visits, searchTerm, dateRange, advancedFilters]);

  const paginatedVisits = useMemo(() => {
    return filteredVisits.slice(
      (currentPage - 1) * pageSize,
      currentPage * pageSize
    );
  }, [filteredVisits, currentPage, pageSize]);

  const columns: ColumnsType<SupplierVisit> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked ? filteredVisits.map((v) => v.supplierId.toString()) : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.supplierId.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.supplierId.toString()]
                : prev.filter((id) => id !== record.supplierId.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Supplier",
      dataIndex: "supplierId",
      key: "supplierId",
      sorter: (a, b) =>
        getSupplierDisplay(a.supplierId).localeCompare(
          getSupplierDisplay(b.supplierId)
        ),
      render: (supplierId) => (
        <span className="text-green-600 font-medium">
          {getSupplierDisplay(supplierId)}
        </span>
      ),
      filterSearch: true,
      onFilter: (value, record) =>
        getSupplierDisplay(record.supplierId)
          .toLowerCase()
          .includes((value as string).toLowerCase()),
      filters: Array.from(new Set(visits.map((v) => v.supplierId))).map(
        (id) => ({
          text: getSupplierDisplay(id),
          value: id,
        })
      ),
    },
    {
      title: "Audit Code",
      dataIndex: "visit_code",
      key: "visitCode",
      render: (text) => text || "-",
    },
    {
      title: "Audit Type",
      dataIndex: "visit_type",
      key: "visitType",
      sorter: (a, b) => a.visit_type.localeCompare(b.visit_type),
      render: renderTag,
      filterSearch: true,
      onFilter: (value, record) => record.visit_type === value,
      filters: visitTypes.map((type) => ({ text: type, value: type })),
    },
    {
      title: "Place",
      dataIndex: "place",
      key: "place",
      sorter: (a, b) => a.place.localeCompare(b.place),
      filterSearch: true,
      onFilter: (value, record) =>
        record.place.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(visits.map((v) => v.place))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Objective",
      dataIndex: "objective",
      key: "objective",
      render: (text) => text || "-",
    },
    {
      title: "Material",
      dataIndex: "material",
      key: "material",
      sorter: (a, b) => a.material.localeCompare(b.material),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Scheduled Month",
      dataIndex: "scheduledMonth",
      key: "scheduledMonth",
      sorter: (a, b) => a.scheduled_month.localeCompare(b.scheduled_month),
      render: (text) => dayjs(text).format("YYYY-MM"),
    },
    {
      title: "Follow-up Date",
      dataIndex: "follow_up_date",
      key: "followUpDate",
      render: (value: string) =>
        value ? dayjs(value).format("YYYY-MM-DD") : "-",
    },
    {
      title: "Auditor Name",
      dataIndex: "auditor_name",
      key: "auditorName",
      sorter: (a, b) => a.auditor_name.localeCompare(b.auditor_name),
      filterSearch: true,
      onFilter: (value, record) =>
        record.auditor_name
          .toLowerCase()
          .includes((value as string).toLowerCase()),
      filters: Array.from(new Set(visits.map((v) => v.auditor_name))).map(
        (value) => ({
          text: value,
          value: value,
        })
      ),
    },
    {
      title: "Completion Date",
      dataIndex: "completion_date",
      key: "completionDate",
      sorter: (a, b) =>
        new Date(a.completion_date ?? 0).getTime() -
        new Date(b.completion_date ?? 0).getTime(),
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Findings",
      dataIndex: "description",
      key: "findings",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text || "-"}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Risk",
      dataIndex: "risk_classification",
      key: "riskClassification",
      filters: riskOptions.map((r) => ({ text: r, value: r })),
      onFilter: (value, record) => record.risk_classification === value,
      render: renderTag,
      sorter: (a, b) =>
        a.risk_classification.localeCompare(b.risk_classification),
    },
    {
      title: "Status",
      dataIndex: "Status",
      key: "status",
      filters: statusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.Status === value,
      render: renderTag,
      sorter: (a, b) => a.Status.localeCompare(b.Status),
    },
    {
      title: "Confidential",
      dataIndex: "is_confidential",
      key: "isConfidential",
      render: (val: boolean) => (val ? "Yes" : "No"),
    },
    {
      title: "Attachment",
      key: "attachment",
      render: (_, record) =>
        record.attacted_file ? (
          <Tooltip title="Download">
            <Button
              icon={<DownloadOutlined />}
              //type="link"
              onClick={() => downloadSupplierVisit(record.supplierId)}
              size="small"
              className="text-green-600 p-0"
            >
              Download
            </Button>
          </Tooltip>
        ) : (
          "-"
        ),
    },

    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Updated At",
      dataIndex: "updated_at",
      key: "updated_at",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Created By",
      dataIndex: "created_by_name",
      key: "created_by_name",
      //render: (text) => <span>{dayjs(text).format("YYYY-MM-DD")}</span>,
      sorter: (a, b) =>
        (a.created_by_name || "").localeCompare(b.created_by_name || ""),
    },
    {
      title: "Updated By",
      dataIndex: "updated_by_name",
      key: "updated_by_name",
      //render: (text) => <span>{dayjs(text).format("YYYY-MM-DD")}</span>,
      sorter: (a, b) =>
        (a.updated_by_name || "").localeCompare(b.updated_by_name || ""),
    },

    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingVisit(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.supplierId)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "supplierId", label: "Supplier" },
    { key: "visitCode", label: "Audit Code" },
    { key: "visitType", label: "Audit Type" },
    { key: "place", label: "Place" },
    { key: "objective", label: "Objective" },
    { key: "material", label: "Material" },
    { key: "scheduledMonth", label: "Scheduled Month" },
    { key: "followUpDate", label: "Follow-up Date" },
    { key: "auditorName", label: "Auditor Name" },
    { key: "completionDate", label: "Completion Date" },
    { key: "findings", label: "Description" },
    { key: "riskClassification", label: "Risk" },
    { key: "status", label: "Status" },
    { key: "isConfidential", label: "Confidential" },
    { key: "attachment", label: "Attachment" },
    { key: "created_at", label: "Created_At" },
    { key: "updated_at", label: "Updated_At" },
    { key: "created_by_name", label: "created_by_name" },
    { key: "updated_by_name", label: "updated_by_name" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const VisitCard = React.memo(function VisitCard({
    visit,
  }: {
    visit: SupplierVisit;
  }) {
    return (
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
          <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
            {visit.supplierId}
          </h3>
          <Tag color="blue">{visit.visit_type}</Tag>
        </div>

        {/* Core visit info */}
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
          <strong>Place:</strong> {visit.place || "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
          <strong>Material:</strong> {visit.material || "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
          <strong>Objective:</strong> {visit.objective || "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
          <strong>Description:</strong> {visit.description || "-"}
        </p>

        {/* Dates */}
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
          <strong>Scheduled Month:</strong>{" "}
          {visit.scheduled_month
            ? dayjs(visit.scheduled_month).format("YYYY-MM")
            : "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
          <strong>Completion Date:</strong>{" "}
          {visit.completion_date
            ? dayjs(visit.completion_date).format("YYYY-MM-DD")
            : "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
          <strong>Follow-up Date:</strong>{" "}
          {visit.follow_up_date
            ? dayjs(visit.follow_up_date).format("YYYY-MM-DD")
            : "-"}
        </p>

        {/* Auditor & status */}
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
          <strong>Auditor:</strong> {visit.auditor_name || "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
          <strong>Risk:</strong> {renderTag(visit.risk_classification)}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
          <strong>Status:</strong> {renderTag(visit.Status)}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
          <strong>Confidential:</strong> {visit.is_confidential ? "Yes" : "No"}
        </p>

        {/* Attachment */}
        {visit.attacted_file && (
          <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 p-0">
            <strong>Attachment:</strong>{" "}
            <Button
              icon={<DownloadOutlined />}
              //type="link"
              className="p-0"
              onClick={() => downloadSupplierVisit(visit.supplierId)}
            >
              Download
            </Button>
          </p>
        )}

        {/* Audit trail */}
        <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
          <strong>Created At:</strong>{" "}
          {visit.created_at
            ? dayjs(visit.created_at).format("DD-MM-YYYY hh:mm A")
            : "-"}
        </p>
        <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
          <strong>Updated At:</strong>{" "}
          {visit.updated_at
            ? dayjs(visit.updated_at).format("DD-MM-YYYY hh:mm A")
            : "-"}
        </p>
        <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
          <strong>Created By:</strong> {visit.created_by_name || "-"}
        </p>
        <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
          <strong>Updated By:</strong> {visit.updated_by_name || "-"}
        </p>

        {/* Action buttons */}
        <div className="mt-2 flex gap-2">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingVisit(visit)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(visit)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(visit.supplierId)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
        </div>
      </div>
    );
  });
  VisitCard.displayName = "VisitCard";

  const total = visits.length;
  const active = visits.filter((v) => v.Status === "Active").length;
  const blackListed = visits.filter((v) => v.Status === "BlackListed").length;
  const highRisk = visits.filter(
    (v) => v.risk_classification === "High"
  ).length;

  // Sample trend data for widgets
  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <UserCheckIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Supplier Audits</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({visits.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage supplier audit schedules and findings.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Supplier Audit",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Audits"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All scheduled supplier audits"
            trendData={trendData}
          />
          <WidgetCard
            title="Active Audits"
            value={active}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Currently active audits"
            trendData={trendData}
          />
          <WidgetCard
            title="BlackListed"
            value={blackListed}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Audits with blacklisted status"
            trendData={trendData}
          />
          <WidgetCard
            title="High Risk"
            value={highRisk}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="High-risk audits"
            trendData={trendData}
          />
        </div>

        <div className="mt-4">
          <Card title="Supplier Audit List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier or Place"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Supplier Audit
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Audit Type"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        visitType: value,
                      }))
                    }
                    allowClear
                    options={visitTypes.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Risk Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        riskClassification: value,
                      }))
                    }
                    allowClear
                    options={riskOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="visit_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.visitType ||
                advancedFilters.riskClassification ||
                advancedFilters.status) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.visitType && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          visitType: undefined,
                        }))
                      }
                    >
                      Audit Type: {advancedFilters.visitType}
                    </Tag>
                  )}
                  {advancedFilters.riskClassification && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          riskClassification: undefined,
                        }))
                      }
                    >
                      Risk Level: {advancedFilters.riskClassification}
                    </Tag>
                  )}
                  {advancedFilters.status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.status}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedVisits.map((visit) => (
                  <VisitCard key={visit.supplierId} visit={visit} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SupplierVisit>
                  columns={filteredColumns}
                  dataSource={paginatedVisits}
                  rowKey="supplierId"
                  scrollX="max-content"
                  className="mt-2 mb-2"
                  loading={loading}
                  pagination={false}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filteredVisits.length)}
                </strong>{" "}
                of <strong>{filteredVisits.length}</strong> audits
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredVisits.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                {editingVisit ? (
                  <ClipboardCheck className="w-5 h-5 text-blue-600" /> // Edit icon
                ) : (
                  <ClipboardList className="w-5 h-5 text-green-600" /> // Add icon
                )}
                <span>
                  {editingVisit ? "Edit Supplier Audit" : "Add Supplier Audit"}
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              setSelectedFile(null);
              form.resetFields();
            }}
            destroyOnHidden
            width="95%" // responsive width
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line (was bodyStyle)
            }}
          >
            <Form form={form} layout="vertical" className="w-full px-4">
              {/* Responsive Grid: 1 → 2 → 3 → 4 columns */}
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {/* ---------- Supplier & Audit Info ---------- */}
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Supplier & Audit Info
                  </h3>
                </div>

                {/* Supplier */}
                <Form.Item
                  label="Supplier"
                  name="supplier"
                  rules={[{ required: true, message: "Supplier is required" }]}
                  className="col-span-2"
                >
                  {editingVisit ? (
                    <div className="ant-input-disabled bg-gray-100 px-3 py-1.5 rounded text-gray-700">
                      {suppliers.find(
                        (s) => s.supplier_id === editingVisit.supplierId
                      )
                        ? `${
                            suppliers.find(
                              (s) => s.supplier_id === editingVisit.supplierId
                            )?.company_name
                          } - ${
                            suppliers.find(
                              (s) => s.supplier_id === editingVisit.supplierId
                            )?.contact_person || "N/A"
                          } (${editingVisit.supplierId})`
                        : editingVisit.supplierId}
                    </div>
                  ) : (
                    <Select
                      showSearch
                      optionFilterProp="children"
                      placeholder="Select Supplier"
                      options={suppliers.map((s) => ({
                        label: `${s.company_name}${
                          s.contact_person ? ` - ${s.contact_person}` : ""
                        } (${s.supplier_id})`,
                        value: s.supplier_id,
                      }))}
                    />
                  )}
                </Form.Item>

                <Form.Item
                  name="visitType"
                  label="Audit Type"
                  rules={[
                    { required: true, message: "Audit Type is required" },
                  ]}
                >
                  <Select
                    options={visitTypes.map((v) => ({ label: v, value: v }))}
                  />
                </Form.Item>

                <Form.Item name="place" label="Place">
                  <Input />
                </Form.Item>

                <Form.Item
                  name="objective"
                  label="Objective"
                  className="col-span-2"
                >
                  <Input.TextArea rows={2} />
                </Form.Item>

                <Form.Item name="material" label="Material">
                  <Input />
                </Form.Item>

                {/* ---------- Dates & Scheduling ---------- */}
                <div className="col-span-full mt-4">
                  <h3 className="text-lg font-semibold mb-2">
                    Dates & Scheduling
                  </h3>
                </div>

                <Form.Item
                  name="scheduledMonth"
                  label="Scheduled Month"
                  rules={[
                    { required: true, message: "Scheduled Month is required" },
                  ]}
                >
                  <DatePicker picker="month" className="w-full" />
                </Form.Item>

                <Form.Item
                  name="followUpDate"
                  label="Follow-up Date"
                  rules={[
                    { required: true, message: "Follow-Up date is required" },
                  ]}
                >
                  <DatePicker className="w-full" />
                </Form.Item>

                <Form.Item name="completionDate" label="Completion Date">
                  <DatePicker className="w-full" />
                </Form.Item>

                <Form.Item name="auditorName" label="Auditor Name">
                  <Input />
                </Form.Item>

                <Form.Item
                  name="isConfidential"
                  label="Confidential?"
                  valuePropName="checked"
                >
                  <Switch />
                </Form.Item>

                {/* ---------- Findings & Risk ---------- */}
                <div className="col-span-full mt-4">
                  <h3 className="text-lg font-semibold mb-2">
                    Findings & Risk Assessment
                  </h3>
                </div>

                <div className="col-span-full">
                  <Form.Item name="findings" label="Findings / Description">
                    <Input.TextArea rows={3} />
                  </Form.Item>
                </div>

                <Form.Item
                  name="riskClassification"
                  label="Risk Classification"
                  rules={[
                    {
                      required: true,
                      message: "Risk Classification is required",
                    },
                  ]}
                >
                  <Select
                    options={riskOptions.map((v) => ({ label: v, value: v }))}
                  />
                </Form.Item>

                <Form.Item
                  name="status"
                  label="Status"
                  rules={[{ required: true, message: "Status is required" }]}
                >
                  <Select
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                </Form.Item>

                {/* ---------- Attachments ---------- */}
                <div className="col-span-full mt-4">
                  <h3 className="text-lg font-semibold mb-2">Attachments</h3>
                </div>

                <div className="col-span-full">
                  <Form.Item name="attacted_file" label="Attachment">
                    <div>
                      {editingVisit && editingVisit.attacted_file && (
                        <div className="mb-2">
                          <span>Current File: </span>
                          <Button
                            type="link"
                            icon={<DownloadOutlined />}
                            onClick={() =>
                              downloadSupplierVisit(editingVisit.supplierId)
                            }
                          >
                            Download Existing File
                          </Button>
                        </div>
                      )}
                      <Input type="file" onChange={handleFileChange} />
                    </div>
                  </Form.Item>
                </div>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Supplier Audit Details"
          open={!!viewingVisit}
          onCancel={() => setViewingVisit(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingVisit && (
            <div className="space-y-2 text-sm sm:text-base">
              <p>
                <strong>Supplier:</strong>{" "}
                <span className="text-green-600">
                  {viewingVisit.supplierId}
                </span>
              </p>
              <p>
                <strong>Audit Code:</strong> {viewingVisit.visit_code}
              </p>
              <p>
                <strong>Audit Type:</strong> {viewingVisit.visit_type}
              </p>
              <p>
                <strong>Place:</strong> {viewingVisit.place || "-"}
              </p>
              <p>
                <strong>Objective:</strong> {viewingVisit.objective || "-"}
              </p>
              <p>
                <strong>Material:</strong> {viewingVisit.material || "-"}
              </p>
              <p>
                <strong>Scheduled Month:</strong>{" "}
                {dayjs(viewingVisit.scheduled_month).format("YYYY-MM")}
              </p>
              <p>
                <strong>Follow-up Date:</strong>{" "}
                {viewingVisit.follow_up_date
                  ? dayjs(viewingVisit.follow_up_date).format("YYYY-MM-DD")
                  : "-"}
              </p>
              <p>
                <strong>Confidential:</strong>{" "}
                {viewingVisit.is_confidential ? "Yes" : "No"}
              </p>
              <p>
                <strong>Auditor Name:</strong>{" "}
                {viewingVisit.auditor_name || "-"}
              </p>
              <p>
                <strong>Completion Date:</strong>{" "}
                {viewingVisit.completion_date
                  ? dayjs(viewingVisit.completion_date).format("YYYY-MM-DD")
                  : "-"}
              </p>
              <p>
                <strong>Findings:</strong> {viewingVisit.description || "-"}
              </p>
              <p>
                <strong>Risk Classification:</strong>{" "}
                {viewingVisit.risk_classification}
              </p>
              <p>
                <strong>Status:</strong> {viewingVisit.Status}
              </p>
              {viewingVisit.attacted_file && (
                <p>
                  <strong>Attachment:</strong>{" "}
                  <Button
                    icon={<DownloadOutlined />}
                    //type="link"
                    className="p-0"
                    onClick={() =>
                      downloadSupplierVisit(viewingVisit.supplierId)
                    }
                  >
                    Download
                  </Button>
                </p>
              )}

              <p>
                <strong>Created_By:</strong> {viewingVisit.created_by_name}
              </p>
              <p>
                <strong>Updated_By:</strong> {viewingVisit.updated_by_name}
              </p>
              <p>
                <strong>Created:</strong>{" "}
                {dayjs(viewingVisit.created_at).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Updated:</strong>{" "}
                {dayjs(viewingVisit.updated_at).format("YYYY-MM-DD")}
              </p>
            </div>
          )}
        </Modal>
      </div>
      <ConfirmDialog
        open={confirmDialog.open}
        title={confirmDialog.title}
        message={confirmDialog.message}
        onClose={() => setConfirmDialog((prev) => ({ ...prev, open: false }))}
        onConfirm={async () => {
          await confirmDialog.onConfirm();
          setConfirmDialog((prev) => ({ ...prev, open: false }));
        }}
      />
    </>
  );
}
