"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  EyeOutlined,
  SearchOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
  FileTextOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
//import WidgetCard from "../components/common/WidgetCard";
//import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useRouter } from "next/navigation";
import { createSupplier, deleteSupplier, getAllSuppliers, Supplier, updateSupplier } from "../api/suppliers/suppliers";
//import ConfirmDialog from "../components/common/ConfirmDialog";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../components/common/GlassCard";
import { highlightMatch } from "../components/common/HighLightMatch";
import { exportSelectedSuppliersExcel, exportSelectedSuppliersPDF, exportSuppliersExcel, exportSuppliersPDF } from "../api/suppliers/exportsuppliers/exportsupplier";
import {  FaFileExcel, FaFileExport, FaFilePdf,  FaTrash } from "react-icons/fa";
import { UserCheck, Users, WarehouseIcon } from "lucide-react";

const WidgetCard = dynamic(() => import("../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});

const ColumnVisibilityToggle = dynamic(() => import("../components/common/ManageColumn"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
});

const ConfirmDialog = dynamic(() => import("../components/common/ConfirmDialog"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
});

const { RangePicker } = DatePicker;
const riskLevelOptions = ["Low", "Medium", "High"] as const;
const statusOptions = ["Active", "Inactive", "BlackListed"] as const;
const approvedStatusOptions = ["Not_Choose", "Approved", "Disapproved"] as const;

const { Option } = Select;
export default function SuppliersPage() {
  const [form] = Form.useForm();
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingSupplier, setEditingSupplier] = useState<Supplier | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => { },
    title: '',
    message: '',
  });
  const [loading, setLoading] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(null);
  const [viewingSupplier, setViewingSupplier] = useState<Supplier | null>(null);
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    riskClassification: undefined,
    status: undefined,
  });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();
  const router = useRouter();

  useEffect(() => {
    const fetchSuppliers = async () => {
      try {
        setLoading(true);
        const data = await getAllSuppliers();
        setSuppliers(data);
      } catch (error) {
        console.error("Failed to fetch suppliers:", error);
        toast.error("Failed to fetch suppliers");
      }
      finally {
        setLoading(false);
      }
    };
    fetchSuppliers();
  }, []);

  const openAddModal = () => {
    setEditingSupplier(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (supplier: Supplier) => {
    setEditingSupplier(supplier);
    form.setFieldsValue({
      companyName: supplier.company_name,
      contactPerson: supplier.contact_person,
      email: supplier.email,
      phone: supplier.phone_number,
      country: supplier.country,
      address: supplier.address,
      companyProfile: supplier.company_profile,
      ownershipDetails: supplier.ownership_details,
      gstTaxId: supplier.gst_tax_id,
      riskClassification: supplier.risk_classification,
      status: supplier.status,
      Suplier_type: supplier.Suplier_type, // New field for supplier type
      fax: supplier.fax,
      license_numbers: supplier.license_numbers,
      critical_information: supplier.critical_information,
      number_of_employees: supplier.number_of_employees,
      nature_of_business: supplier.nature_of_business,
      bank_name: supplier.bank_name,
      bank_branch: supplier.bank_branch,
      account_number: supplier.account_number,
      ifsc_code: supplier.ifsc_code,
      swift_code: supplier.swift_code,
      approved_status: supplier.approved_status,
    });
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Supplier",
      message: "Are you sure you want to delete this supplier?",
      onConfirm: async () => {
        try {
          await deleteSupplier(id);
          const updated = await getAllSuppliers();
          setSuppliers(updated);
          toast.success("Supplier deleted successfully");
        } catch {
          toast.error("Failed to delete supplier");
        }
      },
    });
  };


  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Suppliers",
      message: `Are you sure you want to delete ${selectedRows.length} suppliers?`,
      onConfirm: async () => {
        try {
          await Promise.all(selectedRows.map((id) => deleteSupplier(id)));
          const updated = await getAllSuppliers();
          setSuppliers(updated);
          setSelectedRows([]);
          toast.success("Suppliers deleted successfully");
        } catch {
          toast.error("Failed to delete selected suppliers");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();

      const payload = {
        company_name: values.companyName,
        contact_person: values.contactPerson || null,
        email: values.email || null,
        phone_number: values.phone || null,
        country: values.country || null,
        address: values.address || null,
        company_profile: values.companyProfile || null,
        ownership_details: values.ownershipDetails || null,
        gst_tax_id: values.gstTaxId || null,
        risk_classification: values.riskClassification,
        status: values.status,
        Suplier_type: values.Suplier_type,
        fax: values.fax || null,
        license_numbers: values.license_numbers || null,
        critical_information: values.critical_information || null,
        number_of_employees: values.number_of_employees ? Number(values.number_of_employees) : null,
        nature_of_business: values.nature_of_business || null,
        bank_name: values.bank_name || null,
        bank_branch: values.bank_branch || null,
        account_number: values.account_number || null,
        ifsc_code: values.ifsc_code || null,
        swift_code: values.swift_code || null,
        approved_status: values.approved_status,
      };

      if (editingSupplier) {
        await updateSupplier(editingSupplier.supplier_id!, payload);
        toast.success("Supplier updated successfully");
      } else {
        await createSupplier(payload);
        toast.success("Supplier added successfully");
      }
      const updatedSuppliers = await getAllSuppliers();
      setSuppliers(updatedSuppliers);
      setIsModalOpen(false);
      form.resetFields();
    } catch (err) {
      toast.error("Failed to submit supplier");
    }
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Low: "green",
      Medium: "orange",
      High: "red",
      Active: "green",
      Inactive: "gold",
      BlackListed: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Low: <CheckCircleOutlined />,
      Medium: <PauseCircleOutlined />,
      High: <FireOutlined />,
      Active: <CheckCircleOutlined />,
      Inactive: <MinusCircleOutlined />,
      BlackListed: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredSuppliers = suppliers.filter((supplier) => {
    const matchesSearch =
      supplier.company_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      supplier.supplier_id?.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (supplier.created_at &&
        (!dateRange[0] ||
          dayjs(supplier.created_at).isSame(dateRange[0]) ||
          dayjs(supplier.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(supplier.created_at).isSame(dateRange[1]) ||
          dayjs(supplier.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.riskClassification ||
        supplier.risk_classification === advancedFilters.riskClassification) &&
      (!advancedFilters.status || supplier.status === advancedFilters.status);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedSuppliers = filteredSuppliers.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<Supplier> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredSuppliers.map((s) => s.supplier_id!.toString()) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.supplier_id!.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.supplier_id!.toString()]
                : prev.filter((id) => id !== record.supplier_id!.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Supplier Id",
      dataIndex: "supplier_id",
      key: "supplier_id",
      sorter: (a, b) => a.supplier_id!.localeCompare(b.supplier_id!),
    },
    {
      title: "Company",
      dataIndex: "company_name",
      key: "company_name",
      sorter: (a, b) => a.company_name.localeCompare(b.company_name),
      render: (text, record) => (
        <Button
          type="link"
          className="text-green-600 font-medium p-0"
          onClick={() => router.push(`/suppliers/supplierdetail?id=${record.supplier_id}&name=${encodeURIComponent(record.company_name)}`)}
        >
          {highlightMatch(text, searchTerm)}
        </Button>
      ),
      filterSearch: true,
      onFilter: (value, record) =>
        record.company_name.toLowerCase().includes((value as string).toLowerCase()),
      filters: suppliers.map((s) => ({ text: s.company_name, value: s.company_name })),
    },
    {
      title: "Contact",
      dataIndex: "contact_person",
      key: "contact_person",
      sorter: (a, b) => (a.contact_person || "").localeCompare(b.contact_person || ""),
      filterSearch: true,
      onFilter: (value, record) =>
        (record.contact_person || "").toLowerCase().includes((value as string).toLowerCase()),
      filters: suppliers.map((s) => ({ text: s.contact_person || "", value: s.contact_person || "" })),
    },
    {
      title: "Supplier Type",
      dataIndex: "Suplier_type",
      key: "Suplier_type",
      sorter: (a, b) => (a.Suplier_type || "").localeCompare(b.Suplier_type || ""),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Email",
      dataIndex: "email",
      key: "email",
      sorter: (a, b) => (a.email || "").localeCompare(b.email || ""),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Phone",
      dataIndex: "phone_number",
      key: "phone_number",
      sorter: (a, b) => (a.phone_number || "").localeCompare(b.phone_number || ""),
    },
    {
      title: "Country",
      dataIndex: "country",
      key: "country",
      sorter: (a, b) => (a.country || "").localeCompare(b.country || ""),
      filterSearch: true,
      onFilter: (value, record) =>
        (record.country || "").toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(suppliers.map((s) => s.country || ""))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Address",
      dataIndex: "address",
      key: "address",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Company Profile",
      dataIndex: "company_profile",
      key: "company_profile",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Ownership Details",
      dataIndex: "ownership_details",
      key: "ownership_details",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "GST/Tax ID",
      dataIndex: "gst_tax_id",
      key: "gst_tax_id",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Risk",
      dataIndex: "risk_classification",
      key: "risk_classification",
      filters: riskLevelOptions.map((r) => ({ text: r, value: r })),
      onFilter: (value, record) => record.risk_classification === value,
      render: renderTag,
      sorter: (a, b) => a.risk_classification.localeCompare(b.risk_classification),
    },
    {
      title: "Approved Status",
      dataIndex: "approved_status",
      key: "approved_status",
      filters: approvedStatusOptions.map(s => ({ text: s, value: s })),
      onFilter: (value, record) => record.approved_status === value,
    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      filters: statusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.status === value,
      render: renderTag,
      sorter: (a, b) => a.status.localeCompare(b.status),
    },
    {
      title: "Fax",
      dataIndex: "fax",
      key: "fax",
    },
    {
      title: "License Numbers",
      dataIndex: "license_numbers",
      key: "license_numbers",
    },
    {
      title: "Critical Info",
      dataIndex: "critical_information",
      key: "critical_information",
    },
    {
      title: "Employees",
      dataIndex: "number_of_employees",
      key: "number_of_employees",
    },
    {
      title: "Nature of Business",
      dataIndex: "nature_of_business",
      key: "nature_of_business",
    },
    {
      title: "Bank",
      dataIndex: "bank_name",
      key: "bank_name",
      filters: Array.from(new Set(suppliers.map((s) => s.bank_name || ""))).map((val) => ({ text: val, value: val })),
      onFilter: (value, record) => record.bank_name === value,
      render: (text, record) => (
        <Tooltip
          title={
            <>
              Branch: {record.bank_branch || "-"} <br />
              Account No: {record.account_number || "-"} <br />
              IFSC: {record.ifsc_code || "-"} <br />
              SWIFT: {record.swift_code || "-"}
            </>
          }
        >
          {text || "-"}
        </Tooltip>
      ),
    },
    // {
    //   title: "Bank Information",
    //   key: "bank_name",
    //   render: (_, record) => (
    //     <span>
    //       {record.bank_name || "-"} / {record.bank_branch || "-"} <br />
    //       A/C: {record.account_number || "-"} <br />
    //       IFSC: {record.ifsc_code || "-"}, SWIFT: {record.swift_code || "-"}
    //     </span>
    //   ),
    // },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      sorter: (a, b) => dayjs(a.created_at).unix() - dayjs(b.created_at).unix(),
      render: (value: string) =>
        value ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(value).fromNow()})` : "-",

    },
    {
      title: "Created By",
      dataIndex: "created_by_name",
      key: "created_by_name",
      render: (text: string) => text || "-",
    },
    {
      title: "Updated By",
      dataIndex: "updated_by_name",
      key: "updated_by_name",
      render: (text: string) => text || "-",
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              // onClick={() => setViewingSupplier(record)}
              onClick={() => router.push(`/suppliers/supplierdetail?id=${record.supplier_id}&name=${encodeURIComponent(record.company_name)}`)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.supplier_id || "")}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
          <Tooltip title="Supplier Assessment">
            <Button
              icon={<FileTextOutlined />}
              onClick={() => router.push(`/suppliers/supplierassessment?id=${record.supplier_id}&name=${encodeURIComponent(record.company_name)}`)}
              size="small"
              className="text-purple-600"
            />
          </Tooltip>
          <Tooltip title="Supplier Certification">
            <Button
              icon={<IdcardOutlined />}
              onClick={() => router.push(`/suppliers/supplierscertification?id=${record.supplier_id}&name=${encodeURIComponent(record.company_name)}`)}
              size="small"
              className="text-teal-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "supplier_id", label: "Supplier Id" },
    { key: "company_name", label: "Company" },
    { key: "contact_person", label: "Contact" },
    { key: "Suplier_type", label: "Supplier Type" },
    { key: "email", label: "Email" },
    { key: "phone_number", label: "Phone" },
    { key: "country", label: "Country" },
    { key: "address", label: "Address" },
    { key: "company_profile", label: "Company profile" },
    { key: "ownership_details", label: "Ownership" },
    { key: "gst_tax_id", label: "GST/Tax ID" },
    { key: "risk_classification", label: "Risk" },
    { key: "status", label: "Status" },
    { key: "approved_status", label: "Approved Status" },
    { key: "fax", label: "Fax" },
    { key: "license_numbers", label: "License Numbers" },
    { key: "critical_information", label: "Critical Info" },
    { key: "number_of_employees", label: "Employees" },
    { key: "nature_of_business", label: "Nature of Business" },
    { key: "bank_name", label: "Bank Information" },

    { key: "created_at", label: "Created At" },
    { key: "created_by_name", label: "Created By" },
    { key: "updated_by_name", label: "Updated By" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const SupplierCard = ({ supplier }: { supplier: Supplier }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="success">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full space-y-2">

        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2">
          <h3 className="text-lg font-semibold text-gray-800 dark:text-white">
            {supplier.company_name || "N/A"}
          </h3>
          <Tag color="blue">{supplier.country || "N/A"}</Tag>
        </div>

        {/* Line by line details */}
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Supplier ID:</strong> {supplier.supplier_id || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Type:</strong> {supplier.Suplier_type || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Status:</strong> {renderTag(supplier.status)}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Approved:</strong> {supplier.approved_status}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Risk:</strong> {renderTag(supplier.risk_classification)}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Contact Person:</strong> {supplier.contact_person || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Email:</strong> {supplier.email || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Phone:</strong> {supplier.phone_number || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Fax:</strong> {supplier.fax || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Address:</strong> {supplier.address || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Company Profile:</strong> {supplier.company_profile || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Ownership:</strong> {supplier.ownership_details || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Employees:</strong> {supplier.number_of_employees ?? "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Nature of Business:</strong> {supplier.nature_of_business || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Critical Info:</strong> {supplier.critical_information || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>License Numbers:</strong> {supplier.license_numbers || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Bank Name:</strong> {supplier.bank_name || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Bank Branch:</strong> {supplier.bank_branch || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>Account Number:</strong> {supplier.account_number || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>IFSC Code:</strong> {supplier.ifsc_code || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>SWIFT Code:</strong> {supplier.swift_code || "N/A"}</p>
        <p className="text-sm text-gray-600 dark:text-gray-300"><strong>GST Tax ID:</strong> {supplier.gst_tax_id || "N/A"}</p>

        {/* Footer */}
        <p className="text-xs text-gray-500 dark:text-gray-400 mt-2">
          <strong>Created:</strong>{" "}
          {supplier.created_at ? dayjs(supplier.created_at).format("YYYY-MM-DD") : "N/A"}
        </p>

        {/* Actions */}
        <div className="flex gap-2 pt-2 border-t dark:border-gray-700">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingSupplier(supplier)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(supplier)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => supplier.supplier_id && handleDelete(supplier.supplier_id)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
        </div>
      </div>
    </GlassCard>
  );

  const total = suppliers.length;
  const active = suppliers.filter((s) => s.status === "Active").length;
  const blackListed = suppliers.filter((s) => s.status === "BlackListed").length;
  const highRisk = suppliers.filter((s) => s.risk_classification === "High").length;

  const downloadFile = (data: Blob, fileName: string) => {
    const url = window.URL.createObjectURL(data);
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", fileName);
    document.body.appendChild(link);
    link.click();
    link.remove();
  };

  const handleExport = async (type: "pdf" | "excel") => {
    try {
      let data: Blob;

      if (selectedRows.length > 0) {
        // Export only selected suppliers
        if (type === "pdf") {
          data = await exportSelectedSuppliersPDF(selectedRows as string[]);
          downloadFile(data, "suppliers_selected.pdf");
        } else {
          data = await exportSelectedSuppliersExcel(selectedRows as string[]);
          downloadFile(data, "suppliers_selected.xlsx");
        }
      } else {
        // Export all suppliers
        if (type === "pdf") {
          data = await exportSuppliersPDF();
          downloadFile(data, "suppliers_all.pdf");
        } else {
          data = await exportSuppliersExcel();
          downloadFile(data, "suppliers_all.xlsx");
        }
      }
    } catch (err) {
      console.error("Export failed:", err);
    }
  };


  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
  <WarehouseIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
  <span>Suppliers</span>
  <span className="text-gray-500 text-base sm:text-lg font-medium">
    ({suppliers.length})
  </span>
</h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage supplier master data used throughout the QMS system.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Suppliers",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Suppliers"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All registered suppliers"
          />
          <WidgetCard
            title="Active"
            value={active}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Currently active suppliers"
          />
          <WidgetCard
            title="BlackListed"
            value={blackListed}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Suppliers no longer in use"
          />
          <WidgetCard
            title="High Risk"
            value={highRisk}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="High-risk suppliers"
          />
        </div>

        <div className="mt-4">
          <Card title="Supplier List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier by name or Id"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      {
                        key: "pdf",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFilePdf className="text-red-500" /> Export PDF
                          </span>
                        ),
                      },
                      {
                        key: "excel",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFileExcel className="text-green-500" /> Export Excel
                          </span>
                        ),
                      },
                    ],
                    onClick: ({ key }) => handleExport(key as "pdf" | "excel"),
                  }}
                >
                  <Button type="default" icon={<FaFileExport />}>
                    Export
                  </Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Supplier
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Risk Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, riskClassification: value }))
                    }
                    allowClear
                    options={riskLevelOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="supplier_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    icon={<FaTrash />}
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.riskClassification || advancedFilters.status) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.riskClassification && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          riskClassification: undefined,
                        }))
                      }
                    >
                      Risk Level: {advancedFilters.riskClassification}
                    </Tag>
                  )}
                  {advancedFilters.status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.status}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedSuppliers.map((supplier) => (
                  <SupplierCard key={supplier.supplier_id} supplier={supplier} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<Supplier>
                  columns={filteredColumns}
                  dataSource={paginatedSuppliers}
                  rowKey={(record) => record.supplier_id!}
                  scrollX="max-content"
                  className="mt-2"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing{" "}
                <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>{Math.min(currentPage * pageSize, filteredSuppliers.length)}</strong> of{" "}
                <strong>{filteredSuppliers.length}</strong> suppliers
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredSuppliers.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

;

<motion.div
  initial={{ opacity: 0, scale: 0.95 }}
  animate={{ opacity: 1, scale: 1 }}
  exit={{ opacity: 0, scale: 0.95 }}
>
  <Modal
    title={
      <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
        {/* Both icons shown side by side */}
        <div className="flex items-center gap-1">
          <Users
            className={`w-5 h-5 ${
              !editingSupplier ? "text-green-600" : "text-gray-400"
            }`}
          />
          <UserCheck
            className={`w-5 h-5 ${
              editingSupplier ? "text-blue-600" : "text-gray-400"
            }`}
          />
        </div>
        <span>{editingSupplier ? "Edit Supplier" : "Add Supplier"}</span>
      </div>
    }
    open={isModalOpen}
    onOk={handleFormSubmit}
    onCancel={() => {
      setIsModalOpen(false);
      form.resetFields();
    }}
    destroyOnHidden
    width="95vw" // use vw instead of "%" for modal
    className="!top-5 font-semibold"
    styles={{
    body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line
  }}
  >
    <Form form={form} layout="vertical" className="w-full px-4">
      {/* Responsive grid: 1 col mobile, 2 sm, 3 md, 4 lg */}
      <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
        {/* ---------- General Details ---------- */}
        <div className="col-span-full">
          <h3 className="text-lg font-semibold mb-2">General Details</h3>
        </div>

        <Form.Item
          name="companyName"
          label="Company Name"
          rules={[{ required: true, message: "Company Name is required" }]}
        >
          <Input />
        </Form.Item>

        <Form.Item name="contactPerson" label="Contact Person">
          <Input />
        </Form.Item>

        <Form.Item
          name="email"
          label="Email"
          rules={[{ type: "email", message: "Invalid email format" }]}
        >
          <Input type="email" />
        </Form.Item>

        <Form.Item
          name="phone"
          label="Phone"
          rules={[
            { required: true, message: "Phone number is required" },
            {
              pattern: /^[0-9]{10,15}$/,
              message: "Phone number must be 10 to 15 digits",
            },
          ]}
        >
          <Input />
        </Form.Item>

        <Form.Item
          name="supplier_type"
          label="Supplier Type"
          rules={[{ required: true, message: "Please select a supplier type" }]}
        >
          <Select placeholder="Select supplier type">
            <Option value="Manufacturer">Manufacturer</Option>
            <Option value="Service Provider">Service Provider</Option>
            <Option value="Distributor">Distributor</Option>
          </Select>
        </Form.Item>

        <Form.Item name="country" label="Country">
          <Input />
        </Form.Item>

        <div className="col-span-full">
          <Form.Item name="address" label="Address">
            <Input.TextArea rows={2} />
          </Form.Item>
        </div>

        <div className="col-span-full">
          <Form.Item name="companyProfile" label="Company Profile">
            <Input.TextArea rows={2} />
          </Form.Item>
        </div>

        {/* ---------- Regulatory & Compliance ---------- */}
        <div className="col-span-full mt-4">
          <h3 className="text-lg font-semibold mb-2">Regulatory & Compliance</h3>
        </div>

        <Form.Item name="fax" label="Fax">
          <Input />
        </Form.Item>

        <Form.Item name="license_numbers" label="License Numbers">
          <Input />
        </Form.Item>

        <Form.Item name="gstTaxId" label="GST/Tax ID">
          <Input />
        </Form.Item>

        <Form.Item
          name="riskClassification"
          label="Risk Classification"
          rules={[{ required: true, message: "Classification is required" }]}
        >
          <Select
            options={riskLevelOptions.map((v) => ({ label: v, value: v }))}
          />
        </Form.Item>

        <Form.Item
          name="status"
          label="Status"
          rules={[{ required: true, message: "Status is required" }]}
        >
          <Select
            options={statusOptions.map((v) => ({ label: v, value: v }))}
          />
        </Form.Item>

        <Form.Item
          name="critical_information"
          label="Critical Information"
          rules={[
            {
              required: true,
              message: "Please select Critical or Non Critical",
            },
          ]}
        >
          <Select placeholder="Select Criticality">
            <Option value="Critical">Critical</Option>
            <Option value="Non Critical">Non Critical</Option>
          </Select>
        </Form.Item>

        <Form.Item
          name="approved_status"
          label="Approved Status"
          rules={[{ required: true, message: "Please select approved status" }]}
        >
          <Select
            options={approvedStatusOptions.map((v) => ({
              label: v,
              value: v,
            }))}
          />
        </Form.Item>

        <div className="col-span-full">
          <Form.Item name="ownershipDetails" label="Ownership Details">
            <Input.TextArea rows={2} />
          </Form.Item>
        </div>

        {/* ---------- Business & Financial Information ---------- */}
        <div className="col-span-full mt-4">
          <h3 className="text-lg font-semibold mb-2">
            Business & Financial Information
          </h3>
        </div>

        <Form.Item name="number_of_employees" label="Number of Employees">
          <Input type="number" min={1} />
        </Form.Item>

        <Form.Item name="nature_of_business" label="Nature of Business">
          <Input />
        </Form.Item>

        <Form.Item name="bank_name" label="Bank Name">
          <Input />
        </Form.Item>

        <Form.Item name="bank_branch" label="Bank Branch">
          <Input />
        </Form.Item>

        <Form.Item name="account_number" label="Account Number">
          <Input />
        </Form.Item>

        <Form.Item name="ifsc_code" label="IFSC Code">
          <Input />
        </Form.Item>

        <Form.Item name="swift_code" label="SWIFT Code">
          <Input />
        </Form.Item>
      </div>
    </Form>
  </Modal>
</motion.div>



        <Modal
          title="Supplier Details"
          open={!!viewingSupplier}
          onCancel={() => setViewingSupplier(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingSupplier && (
            <div className="space-y-2 text-sm sm:text-base">
              <p><strong>Supplier Id:</strong> {viewingSupplier.supplier_id}</p>
              <p><strong>Name:</strong> <span className="text-green-600">{viewingSupplier.company_name}</span></p>
              <p><strong>Contact Person:</strong> {viewingSupplier.contact_person || "-"}</p>
              <p><strong>Supplier Type:</strong> {viewingSupplier.Suplier_type || "-"}</p>
              <p><strong>Email:</strong> {viewingSupplier.email || "-"}</p>
              <p><strong>Phone:</strong> {viewingSupplier.phone_number || "-"}</p>
              <p><strong>Country:</strong> {viewingSupplier.country || "-"}</p>
              <p><strong>Address:</strong> {viewingSupplier.address || "-"}</p>
              <p><strong>Company Profile:</strong> {viewingSupplier.company_profile || "-"}</p>
              <p><strong>Ownership Details:</strong> {viewingSupplier.ownership_details || "-"}</p>
              <p><strong>GST/Tax ID:</strong> {viewingSupplier.gst_tax_id || "-"}</p>
              <p><strong>Risk Classification:</strong> {viewingSupplier.risk_classification}</p>
              <p><strong>Status:</strong> {viewingSupplier.status}</p>
              <p><strong>Approved Status:</strong> {viewingSupplier.approved_status}</p>
              <p><strong>Fax:</strong> {viewingSupplier.fax || "-"}</p>
              <p><strong>License Numbers:</strong> {viewingSupplier.license_numbers || "-"}</p>
              <p><strong>Critical Information:</strong> {viewingSupplier.critical_information || "-"}</p>
              <p><strong>Employees:</strong> {viewingSupplier.number_of_employees || "-"}</p>
              <p><strong>Nature of Business:</strong> {viewingSupplier.nature_of_business || "-"}</p>
              <p><strong>Bank Name:</strong> {viewingSupplier.bank_name || "-"}</p>
              <p><strong>Bank Branch:</strong> {viewingSupplier.bank_branch || "-"}</p>
              <p><strong>Account No:</strong> {viewingSupplier.account_number || "-"}</p>
              <p><strong>IFSC Code:</strong> {viewingSupplier.ifsc_code || "-"}</p>
              <p><strong>SWIFT Code:</strong> {viewingSupplier.swift_code || "-"}</p>
              <p><strong>Created By:</strong> {viewingSupplier.created_by_name || "-"}</p>
              <p><strong>Updated By:</strong> {viewingSupplier.updated_by_name || "-"}</p>
              <p><strong>Created At:</strong> {dayjs(viewingSupplier.created_at).format("YYYY-MM-DD")}</p>
            </div>
          )}
        </Modal>
        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}