"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import Upload from "antd/es/upload";
import "antd/es/upload/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  DownloadOutlined,
  WarningOutlined,
  CheckCircleOutlined,
  CloseCircleOutlined,
  FileOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import {
  createSupplierAgreement,
  deleteSupplierAgreement,
  downloadAgreementDocument,
  downloadRenewalDocument,
  downloadTerminationDocument,
  getAllSupplierAgreements,
  SupplierAgreement,
  updateSupplierAgreement,
} from "../api/supplieragreement/supplieragreement";
import { getAllSuppliers, Supplier } from "../api/suppliers/suppliers";
import ConfirmDialog from "../components/common/ConfirmDialog";
import GlassCard from "../components/common/GlassCard";
import { FileEdit, FilePlus, FileSignatureIcon } from "lucide-react";

const WidgetCard = dynamic(() => import("../components/common/WidgetCard"), {
  ssr: false,
});

const ColumnVisibilityToggle = dynamic(
  () => import("../components/common/ManageColumn"),
  {
    ssr: false,
  }
);

const statusOptions = [
  "Active",
  "Expired",
  "Terminated",
  "Pending Renewal",
] as const;
const agreementTypeOptions = [
  "Supply Agreement",
  "NDA",
  "Service Level Agreement",
  "Partnership Agreement",
] as const;

export default function SupplierAgreementPage() {
  const [form] = Form.useForm();
  const [agreements, setAgreements] = useState<SupplierAgreement[]>([]);
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingAgreement, setEditingAgreement] =
    useState<SupplierAgreement | null>(null);
  const [viewingAgreement, setViewingAgreement] =
    useState<SupplierAgreement | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    Status: undefined,
    aggreement_type: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [loading, setLoading] = useState(false);
  const [agreementFileList, setAgreementFileList] = useState<
    import("antd/es/upload/interface").UploadFile[]
  >([]);
  const [renewalFileList, setRenewalFileList] = useState<
    import("antd/es/upload/interface").UploadFile[]
  >([]);
  const [terminationFileList, setTerminationFileList] = useState<
    import("antd/es/upload/interface").UploadFile[]
  >([]);
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    title: "",
    message: "",
    onConfirm: () => {},
  });

  const isMobile = useIsMobile();

  // Fetch suppliers and agreements on mount
  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        const [supplierData, agreementData] = await Promise.all([
          getAllSuppliers(),
          getAllSupplierAgreements(),
        ]);
        setSuppliers(supplierData);
        setAgreements(agreementData);
      } catch (error) {
        toast.error("Failed to fetch data");
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  const openAddModal = () => {
    form.resetFields();
    setEditingAgreement(null);
    setAgreementFileList([]);
    setRenewalFileList([]);
    setTerminationFileList([]);
    setIsModalOpen(true);
  };

  const openEditModal = (agreement: SupplierAgreement) => {
    form.setFieldsValue({
      ...agreement,
      agreement_date: agreement.agreement_date
        ? dayjs(agreement.agreement_date)
        : undefined,
      expiry_date: agreement.expiry_date
        ? dayjs(agreement.expiry_date)
        : undefined,
      renewal_date: agreement.renewal_date
        ? dayjs(agreement.renewal_date)
        : undefined,
      termination_date: agreement.termination_date
        ? dayjs(agreement.termination_date)
        : undefined,
      signature_date: agreement.signature_date
        ? dayjs(agreement.signature_date)
        : undefined,
    });
    setEditingAgreement(agreement);
    setAgreementFileList(
      agreement.agreement_file && typeof agreement.agreement_file === "string"
        ? [
            {
              uid: "-1",
              name: agreement.agreement_file.split("/").pop() || "agreement",
              status: "done",
              url: agreement.agreement_file,
            },
          ]
        : []
    );
    setRenewalFileList(
      agreement.renewal_file && typeof agreement.renewal_file === "string"
        ? [
            {
              uid: "-2",
              name: agreement.renewal_file.split("/").pop() || "renewal",
              status: "done",
              url: agreement.renewal_file,
            },
          ]
        : []
    );
    setTerminationFileList(
      agreement.termination_file &&
        typeof agreement.termination_file === "string"
        ? [
            {
              uid: "-3",
              name:
                agreement.termination_file.split("/").pop() || "termination",
              status: "done",
              url: agreement.termination_file,
            },
          ]
        : []
    );
    setIsModalOpen(true);
  };

  const handleDelete = (supplier_id: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Agreement",
      message: "Are you sure you want to delete this agreement?",
      onConfirm: async () => {
        try {
          await deleteSupplierAgreement(supplier_id);
          toast.success("Supplier Agreement deleted successfully");
          setAgreements((prev) =>
            prev.filter((a) => a.supplier_id !== supplier_id)
          );
        } catch (error) {
          toast.error("Failed to delete supplier agreement");
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;

    setConfirmDialog({
      open: true,
      title: "Delete Supplier Agreements",
      message: `Are you sure you want to delete ${selectedRows.length} agreements?`,
      onConfirm: async () => {
        try {
          for (const supplier_id of selectedRows) {
            await deleteSupplierAgreement(supplier_id);
          }
          toast.success("Supplier Agreements deleted successfully");
          setSelectedRows([]);
          setAgreements((prev) =>
            prev.filter((a) => !selectedRows.includes(a.supplier_id))
          );
        } catch (error) {
          toast.error("Failed to delete agreements");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();
      const data: Partial<SupplierAgreement> = {
        ...values,
        agreement_date: dayjs(values.agreement_date).format("YYYY-MM-DD"),
        expiry_date: dayjs(values.expiry_date).format("YYYY-MM-DD"),
        renewal_date: dayjs(values.renewal_date).format("YYYY-MM-DD"),
        termination_date: dayjs(values.termination_date).format("YYYY-MM-DD"),
        signature_date: dayjs(values.signature_date).format("YYYY-MM-DD"),
        // Only include file fields if they are new uploads (File objects)
        ...(agreementFileList[0]?.originFileObj && {
          agreement_file: agreementFileList[0].originFileObj,
        }),
        ...(renewalFileList[0]?.originFileObj && {
          renewal_file: renewalFileList[0].originFileObj,
        }),
        ...(terminationFileList[0]?.originFileObj && {
          termination_file: terminationFileList[0].originFileObj,
        }),
      };

      if (editingAgreement) {
        await updateSupplierAgreement(editingAgreement.supplier_id, data);
        toast.success("Supplier Agreement updated successfully");
      } else {
        await createSupplierAgreement(data as SupplierAgreement);
        toast.success("Supplier Agreement added successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
      setAgreementFileList([]);
      setRenewalFileList([]);
      setTerminationFileList([]);
      const agreementData = await getAllSupplierAgreements();
      setAgreements(agreementData);
    } catch (error) {
      toast.error("Failed to save agreement");
    }
  };

  const handleDownloadAgreement = async (supplier_id: string) => {
    try {
      await downloadAgreementDocument(supplier_id);
      toast.success("Supplier Agreement document downloaded");
    } catch (error) {
      toast.error("Failed to download Supplier Agreement document");
    }
  };

  const handleDownloadRenewal = async (supplier_id: string) => {
    try {
      await downloadRenewalDocument(supplier_id);
      toast.success("Supplier Renewal document downloaded");
    } catch (error) {
      toast.error("Failed to download Supplier Renewal document");
    }
  };

  const handleDownloadTermination = async (supplier_id: string) => {
    try {
      await downloadTerminationDocument(supplier_id);
      toast.success("Supplier Termination document downloaded");
    } catch (error) {
      toast.error("Failed to download Supplier Termination document");
    }
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Active: "green",
      Expired: "red",
      Terminated: "red",
      "Pending Renewal": "gold",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Active: <CheckCircleOutlined />,
      Expired: <CloseCircleOutlined />,
      Terminated: <CloseCircleOutlined />,
      "Pending Renewal": <WarningOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const getSupplierDisplay = (supplier_id: string) => {
    const supplier = suppliers.find((s) => s.supplier_id === supplier_id);
    return supplier
      ? `${supplier.company_name} (${supplier.contact_person || "N/A"})`
      : supplier_id;
  };

  const filteredAgreements = agreements.filter((agreement) => {
    const supplier = suppliers.find(
      (s) => s.supplier_id === agreement.supplier_id
    );
    const matchesSearch =
      (supplier?.company_name
        .toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      (supplier?.contact_person
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      agreement.aggreement_type
        .toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      agreement.supplier_id.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (agreement.created_at &&
        (!dateRange[0] ||
          dayjs(agreement.created_at).isSame(dateRange[0]) ||
          dayjs(agreement.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(agreement.created_at).isSame(dateRange[1]) ||
          dayjs(agreement.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.Status ||
        agreement.Status === advancedFilters.Status) &&
      (!advancedFilters.aggreement_type ||
        agreement.aggreement_type === advancedFilters.aggreement_type);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedAgreements = filteredAgreements.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<SupplierAgreement> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked ? filteredAgreements.map((a) => a.supplier_id) : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.supplier_id)}
          onChange={(e) => {
            const rowId = record.supplier_id;
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, rowId]
                : prev.filter((id) => id !== rowId)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Supplier",
      dataIndex: "supplier_id",
      key: "supplier_id",
      sorter: (a, b) => {
        const supplierA = suppliers.find(
          (s) => s.supplier_id === a.supplier_id
        );
        const supplierB = suppliers.find(
          (s) => s.supplier_id === b.supplier_id
        );
        return (supplierA?.company_name || a.supplier_id).localeCompare(
          supplierB?.company_name || b.supplier_id
        );
      },
      render: (supplier_id) => {
        const supplier = suppliers.find((s) => s.supplier_id === supplier_id);
        return (
          <span className="text-green-600 font-medium">
            {supplier
              ? `${supplier.company_name} (${supplier.contact_person || "N/A"})`
              : supplier_id}
          </span>
        );
      },
    },
    {
      title: "Agreement Type",
      dataIndex: "aggreement_type",
      key: "aggreement_type",
      sorter: (a, b) => a.aggreement_type.localeCompare(b.aggreement_type),
      filterSearch: true,
      onFilter: (value, record) =>
        record.aggreement_type
          .toLowerCase()
          .includes((value as string).toLowerCase()),
      filters: Array.from(
        new Set(agreements.map((a) => a.aggreement_type))
      ).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Agreement Date",
      dataIndex: "agreement_date",
      key: "agreement_date",
      sorter: (a, b) =>
        dayjs(a.agreement_date).unix() - dayjs(b.agreement_date).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Expiry Date",
      dataIndex: "expiry_date",
      key: "expiry_date",
      sorter: (a, b) =>
        dayjs(a.expiry_date).unix() - dayjs(b.expiry_date).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Renewal Date",
      dataIndex: "renewal_date",
      key: "renewal_date",
      sorter: (a, b) =>
        dayjs(a.renewal_date).unix() - dayjs(b.renewal_date).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Termination Date",
      dataIndex: "termination_date",
      key: "termination_date",
      sorter: (a, b) =>
        dayjs(a.termination_date).unix() - dayjs(b.termination_date).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Signature Date",
      dataIndex: "signature_date",
      key: "signature_date",
      sorter: (a, b) =>
        dayjs(a.signature_date).unix() - dayjs(b.signature_date).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Signature",
      dataIndex: "signature",
      key: "signature",
      sorter: (a, b) => a.signature.localeCompare(b.signature),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Notice Period",
      dataIndex: "notice_period",
      key: "notice_period",
      sorter: (a, b) => a.notice_period.localeCompare(b.notice_period),
    },
    {
      title: "Status",
      dataIndex: "Status",
      key: "Status",
      filters: statusOptions.map((status) => ({ text: status, value: status })),
      onFilter: (value, record) => record.Status === value,
      render: renderTag,
      sorter: (a, b) => a.Status.localeCompare(b.Status),
    },
    {
      title: "Agreement File",
      key: "agreement_file",
      render: (_, record) =>
        record.agreement_file ? (
          <Button
            icon={<DownloadOutlined />}
            onClick={() => handleDownloadAgreement(record.supplier_id)}
            size="small"
            className="text-green-600"
          >
            Download
          </Button>
        ) : (
          "N/A"
        ),
    },
    {
      title: "Renewal File",
      key: "renewal_file",
      render: (_, record) =>
        record.renewal_file ? (
          <Button
            icon={<DownloadOutlined />}
            onClick={() => handleDownloadRenewal(record.supplier_id)}
            size="small"
            className="text-green-600"
          >
            Download
          </Button>
        ) : (
          "N/A"
        ),
    },
    {
      title: "Termination File",
      key: "termination_file",
      render: (_, record) =>
        record.termination_file ? (
          <Button
            icon={<DownloadOutlined />}
            onClick={() => handleDownloadTermination(record.supplier_id)}
            size="small"
            className="text-green-600"
          >
            Download
          </Button>
        ) : (
          "N/A"
        ),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingAgreement(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.supplier_id)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = columns.map((col) => ({
    key: col.key as string,
    label: col.title as string,
  }));

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const AgreementCard = ({ agreement }: { agreement: SupplierAgreement }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="success">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
          <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
            {getSupplierDisplay(agreement.supplier_id)} -{" "}
            {agreement.aggreement_type}
          </h3>
          <Tag color="blue">{agreement.Status}</Tag>
        </div>

        {/* Main Info (Single Column) */}
        <div className="grid grid-cols-1 gap-y-2 text-sm">
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Agreement Date:</strong>{" "}
            {dayjs(agreement.agreement_date).format("YYYY-MM-DD")}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Expiry Date:</strong>{" "}
            {dayjs(agreement.expiry_date).format("YYYY-MM-DD")}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Renewal Date:</strong>{" "}
            {agreement.renewal_date
              ? dayjs(agreement.renewal_date).format("YYYY-MM-DD")
              : "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Termination Date:</strong>{" "}
            {agreement.termination_date
              ? dayjs(agreement.termination_date).format("YYYY-MM-DD")
              : "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Signature Date:</strong>{" "}
            {dayjs(agreement.signature_date).format("YYYY-MM-DD")}
          </p>
          <p className="text-gray-600 dark:text-gray-300 break-words">
            <strong>Signature:</strong> {agreement.signature || "N/A"}
          </p>
          <p className="text-gray-500 dark:text-gray-400 text-xs break-words">
            <strong>Notice Period:</strong> {agreement.notice_period}
          </p>

          {/* Files */}
          {agreement.agreement_file && (
            <p className="text-gray-600 dark:text-gray-300 break-words">
              <strong>Agreement File:</strong>{" "}
              {typeof agreement.agreement_file === "string"
                ? agreement.agreement_file
                : agreement.agreement_file.name}
              <Button
                type="link"
                size="small"
                icon={<DownloadOutlined />}
                onClick={() => handleDownloadAgreement(agreement.supplier_id)}
                className="pl-1"
              >
                Download
              </Button>
            </p>
          )}

          {agreement.renewal_file && (
            <p className="text-gray-600 dark:text-gray-300 break-words">
              <strong>Renewal File:</strong>{" "}
              {typeof agreement.renewal_file === "string"
                ? agreement.renewal_file
                : agreement.renewal_file.name}
              <Button
                type="link"
                size="small"
                icon={<DownloadOutlined />}
                onClick={() => handleDownloadRenewal(agreement.supplier_id)}
                className="pl-1"
              >
                Download
              </Button>
            </p>
          )}

          {agreement.termination_file && (
            <p className="text-gray-600 dark:text-gray-300 break-words">
              <strong>Termination File:</strong>{" "}
              {typeof agreement.termination_file === "string"
                ? agreement.termination_file
                : agreement.termination_file.name}
              <Button
                type="link"
                size="small"
                icon={<DownloadOutlined />}
                onClick={() => handleDownloadTermination(agreement.supplier_id)}
                className="pl-1"
              >
                Download
              </Button>
            </p>
          )}
        </div>

        {/* Footer System Info */}
        <p className="text-gray-500 dark:text-gray-400 text-xs mt-2">
          <strong>Created:</strong>{" "}
          {agreement.created_at
            ? dayjs(agreement.created_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-gray-500 dark:text-gray-400 text-xs">
          <strong>Updated:</strong>{" "}
          {agreement.updated_at
            ? dayjs(agreement.updated_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>

        {/* Actions */}
        <div className="mt-2 flex gap-2">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingAgreement(agreement)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(agreement)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(agreement.supplier_id)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
        </div>
      </div>
    </GlassCard>
  );

  AgreementCard.displayName = "AgreementCard";

  const total = agreements.length;
  const active = agreements.filter((a) => a.Status === "Active").length;
  const expired = agreements.filter((a) => a.Status === "Expired").length;
  const terminated = agreements.filter((a) => a.Status === "Terminated").length;

  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <FileSignatureIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Supplier Agreements</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({agreements.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage supplier agreements.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Supplier Agreements",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Agreements"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All supplier agreements"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,128,255,0.2)] dark:shadow-[0_0_10px_rgba(0,128,255,0.3)]"
          />
          <WidgetCard
            title="Active Agreements"
            value={active}
            icon={<CheckCircleOutlined />}
            color="green"
            percentage={5}
            description="Currently active agreements"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,255,0,0.2)] dark:shadow-[0_0_10px_rgba(0,255,0,0.3)]"
          />
          <WidgetCard
            title="Expired Agreements"
            value={expired}
            icon={<WarningOutlined />}
            color="orange"
            percentage={8}
            description="Expired agreements"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(255,165,0,0.2)] dark:shadow-[0_0_10px_rgba(255,165,0,0.3)]"
          />
          <WidgetCard
            title="Terminated Agreements"
            value={terminated}
            icon={<CloseCircleOutlined />}
            color="red"
            percentage={3}
            description="Terminated agreements"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(255,0,0,0.2)] dark:shadow-[0_0_10px_rgba(255,0,0,0.3)]"
          />
        </div>

        <div className="mt-4">
          <Card title="Supplier Agreement List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier or Agreement Type"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Agreement
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-2 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, Status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Agreement Type"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        aggreement_type: value,
                      }))
                    }
                    allowClear
                    options={agreementTypeOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="supplier_agreement_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <DatePicker.RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.Status || advancedFilters.aggreement_type) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.Status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          Status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.Status}
                    </Tag>
                  )}
                  {advancedFilters.aggreement_type && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          aggreement_type: undefined,
                        }))
                      }
                    >
                      Agreement Type: {advancedFilters.aggreement_type}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedAgreements.map((agreement) => (
                  <AgreementCard
                    key={agreement.supplier_id}
                    agreement={agreement}
                  />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SupplierAgreement>
                  columns={filteredColumns}
                  dataSource={paginatedAgreements}
                  rowKey={(record) => record.supplier_id}
                  scrollX="max-content"
                  className="mt-4"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filteredAgreements.length)}
                </strong>{" "}
                of <strong>{filteredAgreements.length}</strong> agreements
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredAgreements.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                {editingAgreement ? (
                  <>
                    <FileEdit className="w-5 h-5 text-blue-600" />
                    <span className="font-semibold">
                      Edit Supplier Agreement
                    </span>
                  </>
                ) : (
                  <>
                    <FilePlus className="w-5 h-5 text-green-600" />
                    <span className="font-semibold">
                      Add Supplier Agreement
                    </span>
                  </>
                )}
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
              setAgreementFileList([]);
              setRenewalFileList([]);
              setTerminationFileList([]);
            }}
            destroyOnHidden
            width="95%" // responsive width
            className="!top-5 "
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line
            }}
          >
            <Form
              form={form}
              layout="vertical"
              className="w-full px-4 space-y-10"
            >
              {/* Supplier Section */}
              <div className="  font-semibold dark:bg-gray-900">
                <h3 className="flex items-center gap-2 text-lg font-semibold text-gray-800 dark:text-gray-200 mb-6">
                  Supplier Details
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                  <Form.Item
                    name="supplier_id"
                    label="Supplier"
                    rules={[
                      { required: true, message: "Supplier is required" },
                    ]}
                  >
                    <Select
                      showSearch
                      optionFilterProp="label"
                      options={suppliers.map((s) => ({
                        label: `${s.company_name} (${
                          s.contact_person || "N/A"
                        })`,
                        value: s.supplier_id,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="aggreement_type"
                    label="Agreement Type"
                    rules={[
                      { required: true, message: "Agreement Type is required" },
                    ]}
                  >
                    <Select
                      options={agreementTypeOptions.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>
                </div>
              </div>

              {/* Dates Section */}
              <div className=" font-semibold dark:bg-gray-900">
                <h3 className="flex items-center gap-2 text-lg font-semibold text-gray-800 dark:text-gray-200 mb-6">
                  Dates
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                  {[
                    { name: "agreement_date", label: "Agreement Date" },
                    { name: "expiry_date", label: "Expiry Date" },
                    { name: "renewal_date", label: "Renewal Date" },
                    { name: "termination_date", label: "Termination Date" },
                    { name: "signature_date", label: "Signature Date" },
                  ].map((field) => (
                    <Form.Item
                      key={field.name}
                      name={field.name}
                      label={field.label}
                      rules={[
                        {
                          required: true,
                          message: `${field.label} is required`,
                        },
                      ]}
                    >
                      <DatePicker className="w-full" />
                    </Form.Item>
                  ))}
                </div>
              </div>

              {/* Signature & Status Section */}
              <div className=" font-semibold dark:bg-gray-900">
                <h3 className="flex items-center gap-2 text-lg font-semibold text-gray-800 dark:text-gray-200 mb-6">
                  Agreement Details
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                  <Form.Item
                    name="signature"
                    label="Signature"
                    rules={[
                      { required: true, message: "Signature is required" },
                    ]}
                  >
                    <Input />
                  </Form.Item>

                  <Form.Item
                    name="notice_period"
                    label="Notice Period"
                    rules={[
                      { required: true, message: "Notice Period is required" },
                    ]}
                  >
                    <Input />
                  </Form.Item>

                  <Form.Item
                    name="Status"
                    label="Status"
                    rules={[{ required: true, message: "Status is required" }]}
                  >
                    <Select
                      options={statusOptions.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>
                </div>
              </div>

              {/* Files Section */}
              <div className=" font-semibold dark:bg-gray-900">
                <h3 className="flex items-center gap-2 text-lg font-semibold text-gray-800 dark:text-gray-200 mb-6">
                  Files
                </h3>
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                  <Form.Item label="Agreement File">
                    <Upload
                      fileList={agreementFileList}
                      onChange={({ fileList: f }) => setAgreementFileList(f)}
                      beforeUpload={() => false}
                      maxCount={1}
                    >
                      <Button icon={<FileOutlined />}>Upload Agreement</Button>
                    </Upload>
                  </Form.Item>

                  <Form.Item label="Renewal File">
                    <Upload
                      fileList={renewalFileList}
                      onChange={({ fileList: f }) => setRenewalFileList(f)}
                      beforeUpload={() => false}
                      maxCount={1}
                    >
                      <Button icon={<FileOutlined />}>Upload Renewal</Button>
                    </Upload>
                  </Form.Item>

                  <Form.Item label="Termination File" className="lg:col-span-2">
                    <Upload
                      fileList={terminationFileList}
                      onChange={({ fileList: f }) => setTerminationFileList(f)}
                      beforeUpload={() => false}
                      maxCount={1}
                    >
                      <Button icon={<FileOutlined />}>
                        Upload Termination
                      </Button>
                    </Upload>
                  </Form.Item>
                </div>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Supplier Agreement Details"
          open={!!viewingAgreement}
          onCancel={() => setViewingAgreement(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingAgreement && (
            <div className="space-y-2 text-sm sm:text-base">
              <p>
                <strong>Supplier:</strong>{" "}
                <span className="text-green-600">
                  {getSupplierDisplay(viewingAgreement.supplier_id)}
                </span>
              </p>
              <p>
                <strong>Agreement Type:</strong>{" "}
                {viewingAgreement.aggreement_type}
              </p>
              <p>
                <strong>Agreement Date:</strong>{" "}
                {dayjs(viewingAgreement.agreement_date).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Expiry Date:</strong>{" "}
                {dayjs(viewingAgreement.expiry_date).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Renewal Date:</strong>{" "}
                {dayjs(viewingAgreement.renewal_date).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Termination Date:</strong>{" "}
                {dayjs(viewingAgreement.termination_date).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Signature Date:</strong>{" "}
                {dayjs(viewingAgreement.signature_date).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Signature:</strong> {viewingAgreement.signature}
              </p>
              <p>
                <strong>Notice Period:</strong> {viewingAgreement.notice_period}
              </p>
              <p>
                <strong>Status:</strong> {viewingAgreement.Status}
              </p>
              <p>
                <strong>Agreement File:</strong>{" "}
                {viewingAgreement.agreement_file ? (
                  <a
                    href={viewingAgreement.agreement_file as string}
                    target="_blank"
                    rel="noopener noreferrer"
                  >
                    View Agreement
                  </a>
                ) : (
                  "None"
                )}
              </p>
              <p>
                <strong>Renewal File:</strong>{" "}
                {viewingAgreement.renewal_file ? (
                  <a
                    href={viewingAgreement.renewal_file as string}
                    target="_blank"
                    rel="noopener noreferrer"
                  >
                    View Renewal
                  </a>
                ) : (
                  "None"
                )}
              </p>
              <p>
                <strong>Termination File:</strong>{" "}
                {viewingAgreement.termination_file ? (
                  <a
                    href={viewingAgreement.termination_file as string}
                    target="_blank"
                    rel="noopener noreferrer"
                  >
                    View Termination
                  </a>
                ) : (
                  "None"
                )}
              </p>
              <p>
                <strong>Created:</strong>{" "}
                {dayjs(viewingAgreement.created_at).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Updated:</strong>{" "}
                {viewingAgreement.updated_at
                  ? dayjs(viewingAgreement.updated_at).format("YYYY-MM-DD")
                  : "N/A"}
              </p>
            </div>
          )}
        </Modal>
        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
