"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import Upload from "antd/es/upload";
import "antd/es/upload/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  DownloadOutlined,
  WarningOutlined,
  CheckCircleOutlined,
  CloseCircleOutlined,
  FileOutlined,
  ShoppingCartOutlined,
  SendOutlined,
  PlusCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "@/app/components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import {
  PurchaseRequisition,
  getPurchaseRequisitions,
  createPurchaseRequisition,
  updatePurchaseRequisition,
  deletePurchaseRequisition,
  downloadPurchaseRequisitionDocument,
} from "@/app/api/purchaserequisition/purchaserequisition";
import { getAllProducts, Product } from "@/app/api/products/product";

import ConfirmDialog from "@/app/components/common/ConfirmDialog";
import GlassCard from "@/app/components/common/GlassCard";
import { FilePlusIcon } from "lucide-react";
const WidgetCard = dynamic(() => import("@/app/components/common/WidgetCard"), {
  ssr: false,
});

const ColumnVisibilityToggle = dynamic(
  () => import("@/app/components/common/ManageColumn"),
  {
    ssr: false,
  }
);

const statusOptions = ["Pending", "Approved", "Rejected"] as const;
const priorityOptions = ["Low", "Medium", "High", "Urgent"] as const;
const departmentOptions = [
  "HR",
  "IT",
  "Finance",
  "Operations",
  "Marketing",
  "Sales",
] as const;

export default function PurchaseRequisitionPage() {
  const [form] = Form.useForm();
  const [purchaseRequisitions, setPurchaseRequisitions] = useState<
    PurchaseRequisition[]
  >([]);
  const [products, setProducts] = useState<Product[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingPurchaseRequisition, setEditingPurchaseRequisition] =
    useState<PurchaseRequisition | null>(null);
  const [viewingPurchaseRequisition, setViewingPurchaseRequisition] =
    useState<PurchaseRequisition | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    Status: undefined,
    priority: undefined,
    department: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [loading, setLoading] = useState(false);
  const [supportedFileList, setSupportedFileList] = useState<
    import("antd/es/upload/interface").UploadFile[]
  >([]);
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    title: "",
    message: "",
    onConfirm: () => {},
  });

  const isMobile = useIsMobile();

  // Fetch data on mount
  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);

      try {
        // Always fetch products first since they're essential for the form
        const productData = await getAllProducts();
        setProducts(productData || []);

        // Then fetch purchase requisitions
        try {
          const requisitionData = await getPurchaseRequisitions();
          setPurchaseRequisitions(requisitionData || []);
        } catch (requisitionError) {
          console.error(
            "Failed to fetch purchase requisitions:",
            requisitionError
          );
          toast.error("Failed to fetch purchase requisitions");
          setPurchaseRequisitions([]);
        }
      } catch (productError) {
        console.error("Failed to fetch products:", productError);
        toast.error(
          "Failed to fetch products - form functionality may be limited"
        );
        setProducts([]);

        // Still try to fetch purchase requisitions even if products fail
        try {
          const requisitionData = await getPurchaseRequisitions();
          setPurchaseRequisitions(requisitionData || []);
        } catch (requisitionError) {
          console.error(
            "Failed to fetch purchase requisitions:",
            requisitionError
          );
          toast.error("Failed to fetch purchase requisitions");
          setPurchaseRequisitions([]);
        }
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, []);

  const openAddModal = () => {
    form.resetFields();
    setEditingPurchaseRequisition(null);
    setSupportedFileList([]);
    setIsModalOpen(true);
  };

  const openEditModal = (purchaseRequisition: PurchaseRequisition) => {
    form.setFieldsValue({
      ...purchaseRequisition,
      purchase_requisition_date: purchaseRequisition.purchase_requisition_date
        ? dayjs(purchaseRequisition.purchase_requisition_date)
        : undefined,
      expected_date: purchaseRequisition.expected_date
        ? dayjs(purchaseRequisition.expected_date)
        : undefined,
    });
    setEditingPurchaseRequisition(purchaseRequisition);
    setSupportedFileList(
      purchaseRequisition.supported_file &&
        typeof purchaseRequisition.supported_file === "string"
        ? [
            {
              uid: "-1",
              name:
                purchaseRequisition.supported_file.split("/").pop() ||
                "document",
              status: "done",
              url: purchaseRequisition.supported_file,
            },
          ]
        : []
    );
    setIsModalOpen(true);
  };

  const handleDelete = (purchase_requisition_id: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Purchase Requisition",
      message: "Are you sure you want to delete this purchase requisition?",
      onConfirm: async () => {
        try {
          await deletePurchaseRequisition(purchase_requisition_id);
          toast.success("Purchase Requisition deleted successfully");
          setPurchaseRequisitions((prev) =>
            prev.filter(
              (pr) => pr.purchase_requisition_id !== purchase_requisition_id
            )
          );
        } catch (error) {
          toast.error("Failed to delete Purchase Requisition");
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;

    setConfirmDialog({
      open: true,
      title: "Delete Purchase Requisitions",
      message: `Are you sure you want to delete ${selectedRows.length} purchase requisitions?`,
      onConfirm: async () => {
        try {
          for (const purchase_requisition_id of selectedRows) {
            await deletePurchaseRequisition(purchase_requisition_id);
          }
          toast.success("Purchase Requisitions deleted successfully");
          setSelectedRows([]);
          setPurchaseRequisitions((prev) =>
            prev.filter(
              (pr) => !selectedRows.includes(pr.purchase_requisition_id!)
            )
          );
        } catch (error) {
          toast.error("Failed to delete Purchase Requisitions");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();
      const data: PurchaseRequisition = {
        ...values,
        purchase_requisition_date: values.purchase_requisition_date
          ? dayjs(values.purchase_requisition_date).format("YYYY-MM-DD")
          : null,
        expected_date: values.expected_date
          ? dayjs(values.expected_date).format("YYYY-MM-DD")
          : null,
        ...(supportedFileList[0]?.originFileObj && {
          supported_file: supportedFileList[0].originFileObj,
        }),
      };

      if (editingPurchaseRequisition) {
        await updatePurchaseRequisition(
          editingPurchaseRequisition.purchase_requisition_id!,
          data
        );
        toast.success("Purchase Requisition updated successfully");
      } else {
        await createPurchaseRequisition(data);
        toast.success("Purchase Requisition created successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
      setSupportedFileList([]);
      const requisitionData = await getPurchaseRequisitions();
      setPurchaseRequisitions(requisitionData);
    } catch (error) {
      toast.error("Failed to save purchase requisition");
    }
  };

  const handleDownloadDocument = async (purchase_requisition_id: string) => {
    try {
      await downloadPurchaseRequisitionDocument(purchase_requisition_id);
      toast.success("Purchase Requisition document downloaded");
    } catch (error) {
      toast.error("Failed to download Purchase Requisition document");
    }
  };

  const renderStatusTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Pending: "gold",
      Approved: "green",
      Rejected: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Pending: <WarningOutlined />,
      Approved: <CheckCircleOutlined />,
      Rejected: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const renderPriorityTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Low: "green",
      Medium: "blue",
      High: "orange",
      Urgent: "red",
    };

    return <Tag color={colorMap[value] || "default"}>{value}</Tag>;
  };

  const getProductDisplay = (product_id: string) => {
    const product = products.find((p) => p.product_id === product_id);
    return product
      ? `${product.product_name} (${product.product_id})`
      : product_id || "N/A";
  };

  const filteredPurchaseRequisitions = purchaseRequisitions.filter((pr) => {
    const product = products.find((p) => p.product_id === pr.product_id);
    const matchesSearch =
      (product?.product_name
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      (pr.purchase_requisition_id
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      (pr.department?.toLowerCase().includes(searchTerm.toLowerCase()) ??
        false) ||
      (pr.description?.toLowerCase().includes(searchTerm.toLowerCase()) ??
        false);

    const matchesDate =
      !dateRange ||
      (pr.created_at &&
        (!dateRange[0] ||
          dayjs(pr.created_at).isSame(dateRange[0]) ||
          dayjs(pr.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(pr.created_at).isSame(dateRange[1]) ||
          dayjs(pr.created_at).isBefore(dateRange[1])));

    const matchesAdvanced =
      (!advancedFilters.Status || pr.Status === advancedFilters.Status) &&
      (!advancedFilters.priority || pr.priority === advancedFilters.priority) &&
      (!advancedFilters.department ||
        pr.department === advancedFilters.department);

    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedPurchaseRequisitions = filteredPurchaseRequisitions.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<PurchaseRequisition> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked
                ? filteredPurchaseRequisitions.map(
                    (pr) => pr.purchase_requisition_id!
                  )
                : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.purchase_requisition_id!)}
          onChange={(e) => {
            const rowId = record.purchase_requisition_id!;
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, rowId]
                : prev.filter((id) => id !== rowId)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "PR ID",
      dataIndex: "purchase_requisition_id",
      key: "purchase_requisition_id",
      sorter: (a, b) =>
        (a.purchase_requisition_id || "").localeCompare(
          b.purchase_requisition_id || ""
        ),
      render: (text) => (
        <span className="text-blue-600 font-medium">{text}</span>
      ),
    },
    {
      title: "PR Date",
      dataIndex: "purchase_requisition_date",
      key: "purchase_requisition_date",
      sorter: (a, b) =>
        dayjs(a.purchase_requisition_date || "").unix() -
        dayjs(b.purchase_requisition_date || "").unix(),
      render: (text) => (text ? dayjs(text).format("YYYY-MM-DD") : "N/A"),
    },
    // {
    //   title: "Product ID",
    //   dataIndex: "product_id",
    //   key: "product_id",
    //   render: (text) => (text ? `${text}` : "N/A"),
    // },
    // {
    //   title: "Product Name",
    //   dataIndex: "product_name",
    //   key: "product_name",
    //   render: (text) => (text ? `${text}` : "N/A"),
    // },
    {
      title: "Product",
      dataIndex: "product_id",
      key: "product_id",
      sorter: (a, b) => {
        const productA = products.find((p) => p.product_id === a.product_id);
        const productB = products.find((p) => p.product_id === b.product_id);
        return (productA?.product_name || a.product_id || "").localeCompare(
          productB?.product_name || b.product_id || ""
        );
      },
      render: (product_id) => (
        <span className="text-green-600 font-medium">
          {product_id ? getProductDisplay(product_id) : "N/A"}
        </span>
      ),
    },
    {
      title: "Quantity",
      dataIndex: "quantity",
      key: "quantity",
      sorter: (a, b) => (a.quantity || 0) - (b.quantity || 0),
      render: (text, record) =>
        text ? `${text} ${record.unit_of_measure || ""}` : "N/A",
    },
    {
      title: "Department",
      dataIndex: "department",
      key: "department",
      filters: departmentOptions.map((dept) => ({ text: dept, value: dept })),
      onFilter: (value, record) => record.department === value,
      sorter: (a, b) => (a.department || "").localeCompare(b.department || ""),
    },
    {
      title: "Description",
      dataIndex: "description",
      key: "description",
      render: (text) => (
        <div className="max-w-xs truncate" title={text}>
          {text || "N/A"}
        </div>
      ),
    },
    {
      title: "Unit of Measure",
      dataIndex: "unit_of_measure",
      key: "unit_of_measure",
      render: (text) => text || "N/A",
    },
    {
      title: "Expected Date",
      dataIndex: "expected_date",
      key: "expected_date",
      sorter: (a, b) =>
        dayjs(a.expected_date || "").unix() -
        dayjs(b.expected_date || "").unix(),
      render: (text) => (text ? dayjs(text).format("YYYY-MM-DD") : "N/A"),
    },
    {
      title: "Estimated Cost",
      dataIndex: "estimated_cost",
      key: "estimated_cost",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Justification",
      dataIndex: "justification",
      key: "justification",
      render: (text) => (
        <div className="max-w-xs truncate" title={text}>
          {text || "N/A"}
        </div>
      ),
    },
    {
      title: "Priority",
      dataIndex: "priority",
      key: "priority",
      filters: priorityOptions.map((priority) => ({
        text: priority,
        value: priority,
      })),
      onFilter: (value, record) => record.priority === value,
      render: renderPriorityTag,
      sorter: (a, b) => (a.priority || "").localeCompare(b.priority || ""),
    },
    {
      title: "Status",
      dataIndex: "Status",
      key: "Status",
      filters: statusOptions.map((status) => ({ text: status, value: status })),
      onFilter: (value, record) => record?.Status === value,
      render: renderStatusTag,
      sorter: (a, b) => (a.Status || "").localeCompare(b.Status || ""),
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      render: (text) => (text ? dayjs(text).format("YYYY-MM-DD HH:mm") : "N/A"),
    },
    {
      title: "Updated At",
      dataIndex: "updated_at",
      key: "updated_at",
      render: (text) => (text ? dayjs(text).format("YYYY-MM-DD HH:mm") : "N/A"),
    },
    {
      title: "Approved At",
      dataIndex: "approved_at",
      key: "approved_at",
      render: (text) => (text ? dayjs(text).format("YYYY-MM-DD HH:mm") : "N/A"),
    },
    {
      title: "Requested By",
      dataIndex: "requested_by_name",
      key: "requested_by_name",
      render: (text, record) =>
        text ? `${text} (${record.requested_by_id || ""})` : "N/A",
    },
    {
      title: "Approved By",
      dataIndex: "approved_by_name",
      key: "approved_by_name",
      render: (text, record) =>
        text ? `${text} (${record.approved_by_id || ""})` : "N/A",
    },
    {
      title: "Document",
      key: "document",
      render: (_, record) =>
        record.supported_file ? (
          <Button
            icon={<DownloadOutlined />}
            onClick={() =>
              handleDownloadDocument(record.purchase_requisition_id!)
            }
            size="small"
            className="text-green-600"
          >
            Download
          </Button>
        ) : (
          "N/A"
        ),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingPurchaseRequisition(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.purchase_requisition_id!)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = columns.map((col) => ({
    key: col.key as string,
    label: col.title as string,
  }));

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const PurchaseRequisitionCard = ({
    purchaseRequisition,
  }: {
    purchaseRequisition: PurchaseRequisition;
  }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="primary">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
          <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white">
            PR: {purchaseRequisition.purchase_requisition_id || "N/A"}
          </h3>
          <div className="flex gap-1">
            {renderStatusTag(purchaseRequisition.Status || "")}
            {renderPriorityTag(purchaseRequisition.priority || "")}
          </div>
        </div>

        {/* Main Info (Single Column) */}
        <div className="grid grid-cols-1 gap-y-2 text-sm">
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Product:</strong>{" "}
            {purchaseRequisition.product_name || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Department:</strong>{" "}
            {purchaseRequisition.department || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Quantity:</strong>{" "}
            {purchaseRequisition.quantity != null
              ? `${purchaseRequisition.quantity} ${
                  purchaseRequisition.unit_of_measure || ""
                }`
              : "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Estimated Cost:</strong>{" "}
            {purchaseRequisition.estimated_cost || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Expected Date:</strong>{" "}
            {purchaseRequisition.expected_date
              ? dayjs(purchaseRequisition.expected_date).format("YYYY-MM-DD")
              : "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Description:</strong>{" "}
            {purchaseRequisition.description || "N/A"}
          </p>
          {purchaseRequisition.supported_file && (
            <div className="text-gray-600 dark:text-gray-300 break-words">
              <strong>Attachment:</strong>{" "}
              <span className="break-words">
                {typeof purchaseRequisition.supported_file === "string"
                  ? purchaseRequisition.supported_file
                  : purchaseRequisition.supported_file.name}
              </span>
            </div>
          )}
        </div>

        {/* Actions */}
        <div className="mt-2 flex gap-2">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingPurchaseRequisition(purchaseRequisition)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(purchaseRequisition)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() =>
                handleDelete(purchaseRequisition.purchase_requisition_id!)
              }
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
          {purchaseRequisition.supported_file && (
            <Tooltip title="Download">
              <Button
                icon={<DownloadOutlined />}
                onClick={() =>
                  handleDownloadDocument(
                    purchaseRequisition.purchase_requisition_id!
                  )
                }
                size="small"
                className="text-green-600"
              >
                Download
              </Button>
            </Tooltip>
          )}
        </div>

        {/* Created/Updated */}
        <p className="text-gray-500 dark:text-gray-400 text-xs mt-2">
          <strong>Created:</strong>{" "}
          {purchaseRequisition.created_at
            ? dayjs(purchaseRequisition.created_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-gray-500 dark:text-gray-400 text-xs">
          <strong>Updated:</strong>{" "}
          {purchaseRequisition.updated_at
            ? dayjs(purchaseRequisition.updated_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
      </div>
    </GlassCard>
  );

  PurchaseRequisitionCard.displayName = "PurchaseRequisitionCard";

  const total = purchaseRequisitions.length;
  const pending = purchaseRequisitions.filter(
    (pr) => pr.Status === "Pending"
  ).length;
  const approved = purchaseRequisitions.filter(
    (pr) => pr.Status === "Approved"
  ).length;
  const highPriority = purchaseRequisitions.filter(
    (pr) => pr.priority === "High" || pr.priority === "Urgent"
  ).length;

  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <FilePlusIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Purchase Requisitions</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({purchaseRequisitions.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage purchase requisitions and track procurement requests.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Procurement",
            },
            {
              title: "Purchase Requisitions",
            },
          ]}
        />

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Requisitions"
            value={total}
            icon={<ShoppingCartOutlined />}
            color="blue"
            percentage={12}
            description="All requisitions"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,128,255,0.2)] dark:shadow-[0_0_10px_rgba(0,128,255,0.3)]"
          />
          <WidgetCard
            title="Pending Approval"
            value={pending}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="Awaiting approval"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(255,165,0,0.2)] dark:shadow-[0_0_10px_rgba(255,165,0,0.3)]"
          />
          <WidgetCard
            title="Approved"
            value={approved}
            icon={<CheckCircleOutlined />}
            color="green"
            percentage={8}
            description="Approved requisitions"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,255,0,0.2)] dark:shadow-[0_0_10px_rgba(0,255,0,0.3)]"
          />
          <WidgetCard
            title="High Priority"
            value={highPriority}
            icon={<SendOutlined />}
            color="red"
            percentage={3}
            description="High/Urgent priority"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(255,0,0,0.2)] dark:shadow-[0_0_10px_rgba(255,0,0,0.3)]"
          />
        </div>

        <div className="mt-4">
          <Card
            title="Purchase Requisition List"
            className="p-0 mb-6 shadow-sm"
          >
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search PR ID, Product, Department, or Description"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Purchase Requisition
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, Status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Priority"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        priority: value,
                      }))
                    }
                    allowClear
                    options={priorityOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                  <Select
                    placeholder="Department"
                    className="w-full max-w-[200px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        department: value,
                      }))
                    }
                    allowClear
                    options={departmentOptions.map((dept) => ({
                      label: dept,
                      value: dept,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="purchase_requisition_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <DatePicker.RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.Status ||
                advancedFilters.priority ||
                advancedFilters.department) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.Status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          Status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.Status}
                    </Tag>
                  )}
                  {advancedFilters.priority && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          priority: undefined,
                        }))
                      }
                    >
                      Priority: {advancedFilters.priority}
                    </Tag>
                  )}
                  {advancedFilters.department && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          department: undefined,
                        }))
                      }
                    >
                      Department: {advancedFilters.department}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedPurchaseRequisitions.map((purchaseRequisition) => (
                  <PurchaseRequisitionCard
                    key={purchaseRequisition.purchase_requisition_id}
                    purchaseRequisition={purchaseRequisition}
                  />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<PurchaseRequisition>
                  columns={filteredColumns}
                  dataSource={paginatedPurchaseRequisitions}
                  rowKey={(record) => record.purchase_requisition_id!}
                  scrollX="max-content"
                  className="mt-4"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(
                    currentPage * pageSize,
                    filteredPurchaseRequisitions.length
                  )}
                </strong>{" "}
                of <strong>{filteredPurchaseRequisitions.length}</strong>{" "}
                purchase requisitions
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredPurchaseRequisitions.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <span className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                {editingPurchaseRequisition ? (
                  <>
                    <EditOutlined className="text-blue-500" />
                    Edit Purchase Requisition
                  </>
                ) : (
                  <>
                    <PlusCircleOutlined className="text-green-500" />
                    Add Purchase Requisition
                  </>
                )}
              </span>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
              setSupportedFileList([]);
            }}
            destroyOnHidden
            width="95%" // responsive width
            className="!top-5"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line
            }}
          >
            <Form
              form={form}
              layout="vertical"
              className="w-full px-4 space-y-6"
            >
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                {/* --- Requisition Info --- */}
                <div className="col-span-full pb-2 mb-2 text-lg font-semibold text-gray-700">
                  Requisition Information
                </div>

                <Form.Item
                  name="purchase_requisition_date"
                  label="Purchase Requisition Date"
                  rules={[
                    {
                      required: true,
                      message: "Purchase Requisition Date is required",
                    },
                  ]}
                >
                  <DatePicker className="w-full" />
                </Form.Item>

                <Form.Item
                  name="department"
                  label="Department"
                  rules={[
                    { required: true, message: "Department is required" },
                  ]}
                >
                  <Select
                    placeholder="Select department"
                    options={departmentOptions.map((dept) => ({
                      label: dept,
                      value: dept,
                    }))}
                  />
                </Form.Item>

                <Form.Item
                  name="priority"
                  label="Priority"
                  rules={[{ required: true, message: "Priority is required" }]}
                >
                  <Select
                    options={priorityOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>

                <Form.Item
                  name="Status"
                  label="Status"
                  rules={[{ required: true, message: "Status is required" }]}
                >
                  <Select
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                </Form.Item>

                {/* --- Product Details --- */}
                <div className="col-span-full pb-2 mb-2 text-lg font-semibold text-gray-700">
                  Product Details
                </div>

                <Form.Item
                  name="product_id"
                  label="Product"
                  rules={[{ required: true, message: "Product is required" }]}
                >
                  <Select
                    showSearch
                    placeholder="Select a product"
                    optionFilterProp="label"
                    options={products.map((product) => ({
                      label: `${product.product_name ?? "Unnamed"} (${
                        product.product_id
                      })`,
                      value: product.product_id,
                    }))}
                    onChange={(value) => {
                      const selectedProduct = products.find(
                        (p) => p.product_id === value
                      );
                      if (selectedProduct) {
                        form.setFieldsValue({
                          product_name: selectedProduct.product_name,
                        });
                      }
                    }}
                  />
                </Form.Item>

                <Form.Item name="product_name" label="Product Name">
                  <Input
                    placeholder="Product name will be auto-filled"
                    readOnly
                  />
                </Form.Item>

                <Form.Item
                  name="quantity"
                  label="Quantity"
                  rules={[{ required: true, message: "Quantity is required" }]}
                >
                  <Input type="number" placeholder="Enter quantity" />
                </Form.Item>

                <Form.Item name="unit_of_measure" label="Unit of Measure">
                  <Input placeholder="e.g., pieces, kg, liters" />
                </Form.Item>

                {/* --- Logistics & Cost --- */}
                <div className="col-span-full pb-2 mb-2 text-lg font-semibold text-gray-700">
                  Logistics & Cost
                </div>

                <Form.Item name="expected_date" label="Expected Date">
                  <DatePicker className="w-full" />
                </Form.Item>

                <Form.Item name="estimated_cost" label="Estimated Cost">
                  <Input placeholder="Enter estimated cost" />
                </Form.Item>

                {/* --- Additional Info --- */}
                <div className="col-span-full pb-2 mb-2 text-lg font-semibold text-gray-700">
                  Additional Information
                </div>

                <div className="col-span-full">
                  <Form.Item name="description" label="Description">
                    <Input.TextArea
                      rows={3}
                      placeholder="Enter description of the requested item/service"
                    />
                  </Form.Item>
                </div>

                <div className="col-span-full">
                  <Form.Item name="justification" label="Justification">
                    <Input.TextArea
                      rows={3}
                      placeholder="Enter justification for this purchase request"
                    />
                  </Form.Item>
                </div>

                <div className="col-span-full lg:col-span-2">
                  <Form.Item label="Supporting Document">
                    <Upload
                      fileList={supportedFileList}
                      onChange={({ fileList: newFileList }) =>
                        setSupportedFileList(newFileList)
                      }
                      beforeUpload={() => false}
                      maxCount={1}
                    >
                      <Button icon={<FileOutlined />}>Upload Document</Button>
                    </Upload>
                  </Form.Item>
                </div>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Purchase Requisition Details"
          open={!!viewingPurchaseRequisition}
          onCancel={() => setViewingPurchaseRequisition(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[600px] top-4"
        >
          {viewingPurchaseRequisition && (
            <div className="space-y-2 text-sm sm:text-base">
              <p>
                <strong>PR ID:</strong>{" "}
                <span className="text-blue-600">
                  {viewingPurchaseRequisition.purchase_requisition_id}
                </span>
              </p>
              <p>
                <strong>PR Date:</strong>{" "}
                {viewingPurchaseRequisition.purchase_requisition_date
                  ? dayjs(
                      viewingPurchaseRequisition.purchase_requisition_date
                    ).format("YYYY-MM-DD")
                  : "N/A"}
              </p>
              <p>
                <strong>Product ID:</strong>{" "}
                {viewingPurchaseRequisition.product_id || "N/A"}
              </p>
              <p>
                <strong>Product Name:</strong>{" "}
                {viewingPurchaseRequisition.product_name || "N/A"}
              </p>
              <p>
                <strong>Product:</strong>{" "}
                <span className="text-green-600">
                  {viewingPurchaseRequisition.product_id
                    ? getProductDisplay(viewingPurchaseRequisition.product_id)
                    : "N/A"}
                </span>
              </p>
              <p>
                <strong>Quantity:</strong>{" "}
                {viewingPurchaseRequisition.quantity || "N/A"}{" "}
                {viewingPurchaseRequisition.unit_of_measure || ""}
              </p>
              <p>
                <strong>Department:</strong>{" "}
                {viewingPurchaseRequisition.department || "N/A"}
              </p>
              <p>
                <strong>Unit of Measure:</strong>{" "}
                {viewingPurchaseRequisition.unit_of_measure || "N/A"}
              </p>
              <p>
                <strong>Expected Date:</strong>{" "}
                {viewingPurchaseRequisition.expected_date
                  ? dayjs(viewingPurchaseRequisition.expected_date).format(
                      "YYYY-MM-DD"
                    )
                  : "N/A"}
              </p>
              <p>
                <strong>Estimated Cost:</strong>{" "}
                {viewingPurchaseRequisition.estimated_cost || "N/A"}
              </p>
              <p>
                <strong>Description:</strong>{" "}
                {viewingPurchaseRequisition.description || "N/A"}
              </p>
              <p>
                <strong>Justification:</strong>{" "}
                {viewingPurchaseRequisition.justification || "N/A"}
              </p>
              <p>
                <strong>Priority:</strong>{" "}
                {renderPriorityTag(viewingPurchaseRequisition?.priority || "")}
              </p>
              <p>
                <strong>Status:</strong>{" "}
                {renderStatusTag(viewingPurchaseRequisition?.Status || "")}
              </p>
              <p>
                <strong>Supporting Document:</strong>{" "}
                {viewingPurchaseRequisition.supported_file ? (
                  <Button
                    icon={<DownloadOutlined />}
                    onClick={() =>
                      handleDownloadDocument(
                        viewingPurchaseRequisition.purchase_requisition_id!
                      )
                    }
                    size="small"
                    className="ml-2"
                  >
                    Download
                  </Button>
                ) : (
                  "None"
                )}
              </p>
              <p>
                <strong>Created:</strong>{" "}
                {viewingPurchaseRequisition.created_at
                  ? dayjs(viewingPurchaseRequisition.created_at).format(
                      "YYYY-MM-DD HH:mm"
                    )
                  : "N/A"}
              </p>
              <p>
                <strong>Updated:</strong>{" "}
                {viewingPurchaseRequisition.updated_at
                  ? dayjs(viewingPurchaseRequisition.updated_at).format(
                      "YYYY-MM-DD HH:mm"
                    )
                  : "N/A"}
              </p>
              <p>
                <strong>Approved At:</strong>{" "}
                {viewingPurchaseRequisition.approved_at
                  ? dayjs(viewingPurchaseRequisition.approved_at).format(
                      "YYYY-MM-DD HH:mm"
                    )
                  : "N/A"}
              </p>
              <p>
                <strong>Requested By:</strong>{" "}
                {viewingPurchaseRequisition.requested_by_name || "N/A"} (
                {viewingPurchaseRequisition.requested_by_id || "N/A"})
              </p>
              <p>
                <strong>Approved By:</strong>{" "}
                {viewingPurchaseRequisition.approved_by_name || "N/A"} (
                {viewingPurchaseRequisition.approved_by_id || "N/A"})
              </p>
            </div>
          )}
        </Modal>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
