"use client";

import React, { useCallback, useEffect, useMemo, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import Spin from "antd/es/spin";
import "antd/es/spin/style";

import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  EyeOutlined,
  SearchOutlined,
  DownloadOutlined,
  InfoCircleOutlined,
  ExclamationCircleOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
import { toast } from "sonner";
import { motion } from "framer-motion";
//import WidgetCard from "../components/common/WidgetCard";
//import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import { useRouter } from "next/navigation";
import {
  createProduct,
  deleteProduct,
  downloadProductDocument,
  getAllProducts,
  Product,
  updateProduct,
} from "../api/products/product";
import { getAllSuppliers, Supplier } from "../api/suppliers/suppliers";
//import ConfirmDialog from "../components/common/ConfirmDialog";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import { PackageCheck, PackagePlus } from "lucide-react";
import { Package } from "lucide-react";
// ✅ WidgetCard
const WidgetCard = dynamic(() => import("../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});

// ✅ ColumnVisibilityToggle
const ColumnVisibilityToggle = dynamic(
  () => import("../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);

// ✅ ConfirmDialog
const ConfirmDialog = dynamic(
  () => import("../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

const { RangePicker } = DatePicker;
const classificationOptions = [
  { label: "Class I", value: "class 1" },
  { label: "Class II", value: "class 2" },
  { label: "Class III", value: "class 3" },
];
const riskLevelOptions = ["Low", "Medium", "High"] as const;
const statusOptions = ["Active", "Inactive", "Discontinued"] as const;

export default function ProductsPage() {
  const [products, setProducts] = useState<Product[]>([]);
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingProduct, setEditingProduct] = useState<Product | null>(null);
  const [form] = Form.useForm();
  const [searchTerm, setSearchTerm] = useState("");
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    classification: undefined,
    riskLevel: undefined,
    status: undefined,
  });
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [viewingProduct, setViewingProduct] = useState<Product | null>(null);
  const [loading, setLoading] = useState(false);
  const isMobile = useIsMobile();
  const router = useRouter();
  const [confirmDialog, setConfirmDialog] = useState<{
    open: boolean;
    title: string;
    message: string;
    onConfirm: () => void;
  } | null>(null);

  // const handleView = (product: Product) => {
  //   setViewingProduct(product);
  // };
  const handleView = useCallback(
    (product: Product) => {
      router.push(`/products/${product.product_id}`);
    },
    [router]
  );

  const fetchProducts = useCallback(async () => {
    try {
      setLoading(true);
      const data = await getAllProducts();
      setProducts(data);
    } catch (error) {
      toast.error("Failed to fetch products" + error);
    } finally {
      setLoading(false);
    }
  }, []);

  const fetchSuppliers = useCallback(async () => {
    try {
      const data = await getAllSuppliers();
      setSuppliers(data);
    } catch (error) {
      toast.error("Failed to load suppliers" + error);
    }
  }, []);

  useEffect(() => {
    fetchProducts();
    fetchSuppliers();
  }, [fetchProducts, fetchSuppliers]); // now safe

  const openAddModal = () => {
    setEditingProduct(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = useCallback(
    (product: Product) => {
      setEditingProduct(product);
      form.setFieldsValue({
        ...product,
        manufacturing_date: product.manufacturing_date
          ? dayjs(product.manufacturing_date)
          : undefined,
        expiry_date: product.expiry_date
          ? dayjs(product.expiry_date)
          : undefined,
      });
      setIsModalOpen(true);
      setSelectedFile(null);
    },
    [form]
  );

  const handleDelete = useCallback((id: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Product",
      message:
        "Are you sure you want to delete this product and it will delete all related data?",
      onConfirm: async () => {
        try {
          await deleteProduct(id);
          const updated = await getAllProducts();
          setProducts(updated);
          toast.success("Product deleted successfully");
        } catch {
          toast.error("Failed to delete product");
        }
      },
    });
  }, []);

  const handleBulkDelete = useCallback(() => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Products",
      message: `Are you sure you want to delete ${selectedRows.length} products and it will delete all related data?`,
      onConfirm: async () => {
        try {
          await Promise.all(selectedRows.map((id) => deleteProduct(id)));
          const updated = await getAllProducts();
          setProducts(updated);
          setSelectedRows([]);
          toast.success("Products deleted successfully");
        } catch {
          toast.error("Failed to delete selected products");
        }
      },
    });
  }, [selectedRows, setProducts, setSelectedRows]);

  const handleFileChange = useCallback(
    (e: React.ChangeEvent<HTMLInputElement>) => {
      const file = e.target.files?.[0] || null;
      setSelectedFile(file);
    },
    []
  );

  const handleFormSubmit = useCallback(async () => {
    try {
      const values = await form.validateFields();
      if (!values.product_code || !values.product_name) {
        toast.error("Product code and name are required");
        return;
      }
      const capitalizedProductName =
        values.product_name.charAt(0).toUpperCase() +
        values.product_name.slice(1);

      const payload = {
        ...values,
        product_name: capitalizedProductName,
        manufacturing_date: dayjs(values.manufacturing_date).format(
          "YYYY-MM-DD"
        ),
        expiry_date: dayjs(values.expiry_date).format("YYYY-MM-DD"),
        product_file: selectedFile,
      };

      if (editingProduct) {
        await updateProduct(editingProduct.product_id!, payload);
        toast.success("Product updated successfully");
      } else {
        await createProduct(payload);
        toast.success("Product added successfully");
      }
      fetchProducts();
      setIsModalOpen(false);
      setSelectedFile(null);
      form.resetFields();
    } catch (error) {
      console.error("Form submission failed:", error);
      toast.error("Failed to submit product form");
    }
  }, [form, editingProduct, selectedFile, fetchProducts, setIsModalOpen]);

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      "Class I": "blue",
      "Class II": "volcano",
      "Class III": "purple",
      Low: "green",
      Medium: "orange",
      High: "red",
      Active: "green",
      Inactive: "gold",
      Discontinued: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      "Class I": <InfoCircleOutlined />,
      "Class II": <WarningOutlined />,
      "Class III": <ExclamationCircleOutlined />,
      Low: <CheckCircleOutlined />,
      Medium: <PauseCircleOutlined />,
      High: <FireOutlined />,
      Active: <CheckCircleOutlined />,
      Inactive: <MinusCircleOutlined />,
      Discontinued: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredProducts = useMemo(() => {
    return products.filter((product) => {
      const matchesSearch =
        product.product_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        product.product_code.toLowerCase().includes(searchTerm.toLowerCase());
      const matchesDate = (() => {
        if (!dateRange || !Array.isArray(dateRange)) return true;
        const [start, end] = dateRange;
        const created = dayjs(product.created_at);
        const afterStart = start
          ? created.isSame(start, "day") || created.isAfter(start, "day")
          : true;
        const beforeEnd = end
          ? created.isSame(end, "day") || created.isBefore(end, "day")
          : true;
        return afterStart && beforeEnd;
      })();
      const matchesAdvanced =
        (!advancedFilters.classification ||
          product.classification === advancedFilters.classification) &&
        (!advancedFilters.riskLevel ||
          product.risk_level === advancedFilters.riskLevel) &&
        (!advancedFilters.status || product.status === advancedFilters.status);
      return matchesSearch && matchesDate && matchesAdvanced;
    });
  }, [products, searchTerm, dateRange, advancedFilters]);

  const paginatedProducts = useMemo(() => {
    return filteredProducts.slice(
      (currentPage - 1) * pageSize,
      currentPage * pageSize
    );
  }, [filteredProducts, currentPage, pageSize]);

  const columns = useMemo<ColumnsType<Product>>(
    () => [
      {
        title: (
          <input
            type="checkbox"
            onChange={(e) => {
              const checked = e.target.checked;
              setSelectedRows(
                checked
                  ? filteredProducts.map((p) => p.product_id!.toString())
                  : []
              );
            }}
            className="w-4 h-4"
          />
        ),
        key: "selection",
        render: (_, record) => (
          <input
            type="checkbox"
            checked={selectedRows.includes(record.product_id!.toString())}
            onChange={(e) => {
              setSelectedRows((prev) =>
                e.target.checked
                  ? [...prev, record.product_id!.toString()]
                  : prev.filter((id) => id !== record.product_id!.toString())
              );
            }}
            className="w-4 h-4"
          />
        ),
        width: 44,
      },
      {
        title: "Product Id",
        dataIndex: "product_id",
        key: "product_id",
        render: (text) => <span className="text-gray-500">{text}</span>,
      },
      {
        title: "Code",
        dataIndex: "product_code",
        key: "productCode",
        sorter: (a, b) => a.product_code.localeCompare(b.product_code),
      },
      {
        title: "Name",
        dataIndex: "product_name",
        key: "productName",
        sorter: (a, b) => a.product_name.localeCompare(b.product_name),
        render: (_, record) => (
          <span className="text-green-600 font-medium">
            {record.product_name}{" "}
            <span className="text-gray-500 text-xs">
              ({record.supplier_name || record.supplier_id})
            </span>
          </span>
        ),
        filterSearch: true,
        onFilter: (value, record) =>
          record.product_name
            .toLowerCase()
            .includes((value as string).toLowerCase()),
        filters: products.map((p) => ({
          text: p.product_name,
          value: p.product_name,
        })),
      },
      {
        title: "Category",
        dataIndex: "category",
        key: "category",
      },
      {
        title: "Classification",
        dataIndex: "classification",
        key: "classification",
        render: renderTag,
        filters: Array.from(new Set(products.map((p) => p.classification))).map(
          (value) => ({
            text: value,
            value: value,
          })
        ),
        onFilter: (value, record) => record.classification === value,
      },
      {
        title: "Risk Level",
        dataIndex: "risk_level",
        key: "riskLevel",
        render: renderTag,
        filters: Array.from(new Set(products.map((p) => p.risk_level))).map(
          (value) => ({
            text: value,
            value: value,
          })
        ),
        onFilter: (value, record) => record.risk_level === value,
      },
      {
        title: "Status",
        dataIndex: "status",
        key: "status",
        render: renderTag,
        filters: Array.from(new Set(products.map((p) => p.status))).map(
          (value) => ({
            text: value,
            value: value,
          })
        ),
        onFilter: (value, record) => record.status === value,
      },
      {
        title: "Description",
        dataIndex: "description",
        key: "description",
        render: (text) => (
          <Tooltip title={text}>
            <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
              {text}
            </span>
          </Tooltip>
        ),
      },
      {
        title: "Specifications",
        dataIndex: "specfication",
        key: "specifications",
        render: (text) => (
          <Tooltip title={text}>
            <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
              {text}
            </span>
          </Tooltip>
        ),
      },
      {
        title: "Manufacturing Date",
        dataIndex: "manufacturing_date",
        key: "manufacturing_date",
        render: (text) => (text ? dayjs(text).format("YYYY-MM-DD") : "-"),
        sorter: (a, b) =>
          dayjs(a.manufacturing_date).unix() -
          dayjs(b.manufacturing_date).unix(),
      },
      {
        title: "Expiry Date",
        dataIndex: "expiry_date",
        key: "expiry_date",
        render: (text) => (text ? dayjs(text).format("YYYY-MM-DD") : "-"),
        sorter: (a, b) =>
          dayjs(a.expiry_date).unix() - dayjs(b.expiry_date).unix(),
      },

      {
        title: "Batch No",
        dataIndex: "batch_no",
        key: "batch_no",
      },
      {
        title: "Reorder Level",
        dataIndex: "reorder_level",
        key: "reorder_level",
      },
      {
        title: "Current Stock",
        dataIndex: "current_stock",
        key: "current_stock",
      },
      {
        title: "Price",
        dataIndex: "price",
        key: "price",
      },
      {
        title: "Product File",
        dataIndex: "product_file",
        key: "product_file",
        render: (_, record) =>
          record.product_file &&
          typeof record.product_file === "string" &&
          record.product_file.trim() !== "" ? (
            <Tooltip title="Download">
              <Button
                type="link"
                icon={<DownloadOutlined />}
                onClick={() => downloadProductDocument(record.product_id!)}
                size="small"
                className="text-green-600 p-0"
              >
                Download
              </Button>
            </Tooltip>
          ) : (
            "-"
          ),
      },
      {
        title: "Created At",
        dataIndex: "created_at",
        key: "created_at",
        render: (value: string) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                value
              ).fromNow()})`
            : "-",
      },
      {
        title: "Updated At",
        dataIndex: "updated_at",
        key: "updated_at",
        render: (value: string) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                value
              ).fromNow()})`
            : "-",
      },
      {
        title: "Created By",
        dataIndex: "created_by_name",
        key: "created_by",
        render: (text) => <span className="text-gray-500">{text}</span>,
      },
      {
        title: "Updated By",
        dataIndex: "updated_by_name",
        key: "updated_by",
        render: (text) => <span className="text-gray-500">{text}</span>,
      },
      {
        title: "Actions",
        key: "actions",
        fixed: "right",
        render: (_, record) => (
          <Space size="small">
            <Tooltip title="View">
              <Button
                icon={<EyeOutlined />}
                onClick={() => handleView(record)}
                size="small"
                className="text-blue-600"
              />
            </Tooltip>
            <Tooltip title="Edit">
              <Button
                icon={<EditOutlined />}
                onClick={() => openEditModal(record)}
                size="small"
                type="primary"
                className="bg-blue-600 hover:bg-blue-700"
              />
            </Tooltip>
            <Tooltip title="Delete">
              <Button
                icon={<DeleteOutlined />}
                onClick={() => handleDelete(record.product_id!.toString())}
                danger
                size="small"
                className="text-red-600"
              />
            </Tooltip>
          </Space>
        ),
      },
    ],
    [
      products,
      selectedRows,
      filteredProducts,
      handleView,
      openEditModal,
      handleDelete,
    ]
  );

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "productCode", label: "Code" },
    { key: "productName", label: "Name" },
    { key: "category", label: "Category" },
    { key: "classification", label: "Classification" },
    { key: "riskLevel", label: "Risk Level" },
    { key: "manufacturing_date", label: "Manufacturing Date" },
    { key: "expiry_date", label: "Expiry Date" },
    { key: "created_at", label: "Created_at" },
    { key: "updated_at", label: "Updated_at" },
    { key: "status", label: "Status" },
    { key: "description", label: "Description" },
    { key: "specifications", label: "Specifications" },
    { key: "batch_no", label: "Batch No" },
    { key: "reorder_level", label: "Reorder Level" },
    { key: "current_stock", label: "Current Stock" },
    { key: "price", label: "Price" },
    { key: "product_file", label: "Product File" },
    { key: "created_by", label: "Created By" },
    { key: "updated_by", label: "Updated By" },
    { key: "actions", label: "Action" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const ProductCard = React.memo(({ product }: { product: Product }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      {/* Header */}
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
          {product.product_name}
        </h3>
        <Tag color="blue">{product.category}</Tag>
      </div>

      {/* Details */}
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
        <strong>Code:</strong> {product.product_code}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Status:</strong> {renderTag(product.status)}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Classification:</strong> {renderTag(product.classification)}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Risk Level:</strong> {product.risk_level}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
        <strong>Description:</strong> {product.description}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Batch No:</strong> {product.batch_no}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Reorder Level:</strong> {product.reorder_level}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Current Stock:</strong> {product.current_stock}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Price:</strong> {product.price}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
        <strong>Specification:</strong> {product.specfication}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Supplier ID:</strong> {product.supplier_id}
      </p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
        <strong>Manufacturing Date:</strong>{" "}
        {dayjs(product.manufacturing_date).format("YYYY-MM-DD")}
      </p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
        <strong>Expiry Date:</strong>{" "}
        {dayjs(product.expiry_date).format("YYYY-MM-DD")}
      </p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
        <strong>Created:</strong>{" "}
        {dayjs(product.created_at).format("YYYY-MM-DD")}
      </p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
        <strong>Updated:</strong>{" "}
        {product.updated_at
          ? dayjs(product.updated_at).format("YYYY-MM-DD")
          : "-"}
      </p>

      {/* File Download */}
      {product.product_file && typeof product.product_file === "string" && (
        <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
          <strong>File: </strong>
          <Button
            type="link"
            size="small"
            className="text-green-600 p-0"
            icon={<DownloadOutlined />}
            onClick={() => downloadProductDocument(product.product_id!)}
          >
            Download
          </Button>
        </p>
      )}

      {/* Actions */}
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button
            icon={<EyeOutlined />}
            onClick={() => handleView(product)}
            size="small"
            className="text-blue-600"
          />
        </Tooltip>
        <Tooltip title="Edit">
          <Button
            icon={<EditOutlined />}
            onClick={() => openEditModal(product)}
            size="small"
            type="primary"
            className="bg-blue-600 hover:bg-blue-700"
          />
        </Tooltip>
        <Tooltip title="Delete">
          <Button
            icon={<DeleteOutlined />}
            onClick={() => handleDelete(product.product_id!.toString())}
            size="small"
            danger
            className="text-red-600"
          />
        </Tooltip>
      </div>
    </div>
  ));
  ProductCard.displayName = "ProductCard";

  const total = products.length;
  const active = products.filter((p) => p.status === "Active").length;
  const discontinued = products.filter(
    (p) => p.status === "Discontinued"
  ).length;
  const highRisk = products.filter((p) => p.risk_level === "High").length;

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <Package className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Products</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({products.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage product master data used throughout the QMS system.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Products",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Products"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All registered medical products"
            className="shadow-[0_0_10px_rgba(0,255,0,0.2)] dark:shadow-[0_0_10px_rgba(0,255,0,0.3)]"
          />
          <WidgetCard
            title="Active"
            value={active}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Currently active in production"
            className="shadow-[0_0_10px_rgba(0,0,255,0.2)] dark:shadow-[0_0_10px_rgba(0,0,255,0.3)]"
          />
          <WidgetCard
            title="Discontinued"
            value={discontinued}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Products no longer in use"
            className="shadow-[0_0_10px_rgba(0,128,255,0.2)] dark:shadow-[0_0_10px_rgba(0,128,255,0.3)]"
          />
          <WidgetCard
            title="High Risk"
            value={highRisk}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="Class III or high-risk items"
            className="shadow-[0_0_10px_rgba(255,165,0,0.2)] dark:shadow-[0_0_10px_rgba(255,165,0,0.3)]"
          />
        </div>

        <div className="mt-4">
          <Card title="Product List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Product by name or code"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                    // onClick: ({ key }) => handleExport(key),
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Product
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Classification"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        classification: value,
                      }))
                    }
                    allowClear
                    options={classificationOptions}
                  />
                  <Select
                    placeholder="Risk Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        riskLevel: value,
                      }))
                    }
                    allowClear
                    options={riskLevelOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="product_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.classification ||
                advancedFilters.riskLevel ||
                advancedFilters.status) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.classification && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          classification: undefined,
                        }))
                      }
                    >
                      Classification: {advancedFilters.classification}
                    </Tag>
                  )}
                  {advancedFilters.riskLevel && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          riskLevel: undefined,
                        }))
                      }
                    >
                      Risk Level: {advancedFilters.riskLevel}
                    </Tag>
                  )}
                  {advancedFilters.status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.status}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedProducts.map((product) => (
                  <ProductCard key={product.product_id} product={product} />
                ))}
              </div>
            ) : (
              <div className=" overflow-auto">
                <DataTable<Product>
                  columns={filteredColumns}
                  dataSource={paginatedProducts}
                  bordered
                  rowKey="product_id"
                  scrollX="max-content"
                  loading={loading}
                  className=" mt-2"
                  pagination={false}
                  //        pagination={{
                  //   current: currentPage,
                  //   pageSize,
                  //   total: filteredProducts.length,
                  //   onChange: (page, size) => {
                  //     setCurrentPage(page);
                  //     setPageSize(size);
                  //   },
                  //   showSizeChanger: true,
                  //   pageSizeOptions: ['10', '30', '50', '100'],
                  //   responsive: true,
                  // }}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filteredProducts.length)}
                </strong>{" "}
                of <strong>{filteredProducts.length}</strong> products
              </div>

              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredProducts.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                {editingProduct ? (
                  <PackageCheck className="w-5 h-5 text-blue-600" /> // Edit
                ) : (
                  <PackagePlus className="w-5 h-5 text-green-600" /> // Add
                )}
                <span>{editingProduct ? "Edit Product" : "Add Product"}</span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            width="95%" // Almost full width
            className="!top-5 font-semibold" // Small top margin
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line
            }}
          >
            <Form
              form={form}
              preserve={false}
              layout="vertical"
              className="w-full"
            >
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {/* ---------- Primary Information ---------- */}
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    General Details
                  </h3>
                </div>

                <Form.Item
                  name="product_code"
                  label="Product Code"
                  rules={[
                    { required: true, message: "Product code is required" },
                  ]}
                >
                  <Input />
                </Form.Item>

                <Form.Item
                  name="product_name"
                  label="Product Name"
                  rules={[
                    { required: true, message: "Product name is required" },
                  ]}
                >
                  <Input />
                </Form.Item>

                <Form.Item
                  name="supplier_id"
                  label="Supplier"
                  rules={[{ required: true, message: "Supplier is required" }]}
                >
                  <Select
                    showSearch
                    optionFilterProp="children"
                    placeholder="Select Supplier"
                    options={suppliers.map((s) => ({
                      label: `${s.company_name} (${s.contact_person})`,
                      value: s.supplier_id,
                    }))}
                  />
                </Form.Item>

                <Form.Item
                  name="description"
                  label="Description"
                  rules={[
                    { required: true, message: "Description is required" },
                  ]}
                >
                  <Input.TextArea rows={1} />
                </Form.Item>

                <Form.Item name="category" label="Category">
                  <Input />
                </Form.Item>

                <Form.Item
                  name="manufacturing_date"
                  label="Manufacturing Date"
                  rules={[
                    {
                      required: true,
                      message: "Manufacturing date is required",
                    },
                  ]}
                >
                  <DatePicker className="w-full" />
                </Form.Item>

                <Form.Item
                  name="expiry_date"
                  label="Expiry Date"
                  rules={[
                    { required: true, message: "Expiry date is required" },
                  ]}
                >
                  <DatePicker className="w-full" />
                </Form.Item>

                {/* ---------- Secondary Information ---------- */}
                <div className="col-span-full mt-4">
                  <h3 className="text-lg font-semibold mb-2">
                    Specifications & Classification
                  </h3>
                </div>

                <Form.Item
                  name="classification"
                  label="Classification"
                  rules={[
                    { required: true, message: "Classification is required" },
                  ]}
                >
                  <Select options={classificationOptions} />
                </Form.Item>

                <Form.Item
                  name="risk_level"
                  label="Risk Level"
                  rules={[
                    { required: true, message: "Risk level is required" },
                  ]}
                >
                  <Select
                    options={riskLevelOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>

                <Form.Item
                  name="status"
                  label="Status"
                  rules={[{ required: true, message: "Status is required" }]}
                >
                  <Select
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                </Form.Item>

                <Form.Item name="specfication" label="Specifications">
                  <Input.TextArea rows={1} />
                </Form.Item>

                {/* ---------- Inventory Information ---------- */}
                <div className="col-span-full mt-4">
                  <h3 className="text-lg font-semibold mb-2">
                    Stock & Pricing
                  </h3>
                </div>

                <Form.Item name="batch_no" label="Batch No">
                  <Input />
                </Form.Item>

                <Form.Item name="reorder_level" label="Reorder Level">
                  <Input />
                </Form.Item>

                <Form.Item name="current_stock" label="Current Stock">
                  <Input />
                </Form.Item>

                <Form.Item name="price" label="Price">
                  <Input />
                </Form.Item>

                {/* ---------- File Upload (Full Width) ---------- */}
                <div className="col-span-full">
                  <Form.Item name="product_file" label="Product File">
                    {editingProduct &&
                      editingProduct.product_file &&
                      typeof editingProduct.product_file === "string" &&
                      editingProduct.product_file.trim() !== "" && (
                        <div className="mb-2">
                          <span>Current File: </span>
                          <Button
                            type="link"
                            icon={<DownloadOutlined />}
                            onClick={() =>
                              downloadProductDocument(
                                editingProduct.product_id!
                              )
                            }
                          >
                            Download Existing File
                          </Button>
                        </div>
                      )}
                    <Input type="file" onChange={handleFileChange} />
                  </Form.Item>
                </div>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Product Details"
          open={!!viewingProduct}
          onCancel={() => setViewingProduct(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingProduct && (
            <div className="space-y-2 text-sm sm:text-base">
              <p>
                <strong>Code:</strong> {viewingProduct.product_code}
              </p>
              <p>
                <strong>Name:</strong>{" "}
                <span className="text-green-600">
                  {viewingProduct.product_name}
                </span>
              </p>
              <p>
                <strong>Category:</strong> {viewingProduct.category}
              </p>

              <p>
                <strong>Specifications:</strong> {viewingProduct.specfication}
              </p>
              <p>
                <strong>Classification:</strong> {viewingProduct.classification}
              </p>
              <p>
                <strong>Risk Level:</strong> {viewingProduct.risk_level}
              </p>
              <p>
                <strong>Status:</strong> {viewingProduct.status}
              </p>
              <p>
                <strong>Batch No:</strong> {viewingProduct?.batch_no}
              </p>
              <p>
                <strong>Reorder Level:</strong> {viewingProduct?.reorder_level}
              </p>
              <p>
                <strong>Current Stock:</strong> {viewingProduct?.current_stock}
              </p>
              <p>
                <strong>Price:</strong> {viewingProduct?.price}
              </p>

              {viewingProduct?.product_file && (
                <p>
                  <strong>Attachment: </strong>
                  <Button
                    type="link"
                    icon={<DownloadOutlined />}
                    onClick={() =>
                      downloadProductDocument(viewingProduct.product_id!)
                    }
                    className="text-green-600 p-0"
                  >
                    Download
                  </Button>
                </p>
              )}

              <p>
                <strong>Created:</strong>{" "}
                {dayjs(viewingProduct.created_at).format("YYYY-MM-DD")}
              </p>
            </div>
          )}
        </Modal>
        {confirmDialog && (
          <ConfirmDialog
            open={confirmDialog.open}
            title={confirmDialog.title}
            message={confirmDialog.message}
            onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
            onConfirm={confirmDialog.onConfirm}
          />
        )}
        {/* {loading && (
        <div className="fixed inset-0 flex items-center justify-center bg-white bg-opacity-50 z-50">
          <Spin size="large" />
        </div>
      )} */}
      </div>
    </>
  );
}
