"use client";

import { useState, useRef, useEffect } from "react";
import Image from "next/image";
import { toast } from "sonner";
import {
  getManageUserById,
  updateManageUser,
  ManageUser,
} from "../../api/manageusers/manageusers";
import { useAuth } from "@/context/AuthContext";

export default function ProfilePage() {
  const { user, loading: authLoading } = useAuth();
  const [formData, setFormData] = useState<ManageUser>({
    First_name: "",
    Last_name: "",
    email: "",
    phone_number: "",
    password: "",
    Gender: "",
    DOB: "",
    role: "",
    department: "",
    manager: "",
    join_date: "",
    location: "",
    address: "",
    country: "",
    emp_status: "",
  });

  const [avatarUrl, setAvatarUrl] = useState("/avatar.jpg");
  const fileInputRef = useRef<HTMLInputElement>(null);
  const [loading, setLoading] = useState(true);

  // ✅ Fetch user profile when user and token are ready
  useEffect(() => {
    if (!user) return;

    const token = localStorage.getItem("authToken");
    if (!token) {
      console.warn("⚠️ No auth token found — skipping profile fetch");
      return;
    }

    const fetchProfile = async () => {
      try {
        setLoading(true);
        console.log("🔹 Fetching profile for user:", user.user_id);

        // ✅ Pass token to API
        const data = await getManageUserById(user.user_id, token);
        console.log("✅ getManageUserById response:", data);

        if (data) {
          setFormData({
            ...data,
            First_name: data.First_name || "",
            Last_name: data.Last_name || "",
          });

          if (data.photo && typeof data.photo === "string") {
            setAvatarUrl(data.photo);
          }
        } else {
          toast.error("No profile data found for this user");
        }
      } catch (err) {
        console.error("❌ Fetch profile error:", err);
        toast.error("Failed to fetch profile");
      } finally {
        setLoading(false);
      }
    };

    fetchProfile();
  }, [user]);

  const handleChange = (key: keyof ManageUser, value: string) => {
    setFormData((prev) => ({ ...prev, [key]: value }));
  };

  const handleAvatarChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      const url = URL.createObjectURL(file);
      setAvatarUrl(url);
      setFormData((prev) => ({ ...prev, photo: file }));
      toast.success("Avatar updated (preview only)");
    }
  };

  const handleSave = async () => {
    if (!user) return;

    try {
      // 🔍 Debug logs before API call
      const token = localStorage.getItem("authToken");
      console.log("🔹 Updating user:", user.user_id);
      console.log("🔹 Auth token being sent:", token);

      const response = await updateManageUser(user.user_id, formData);

      // 🔍 Debug: see API response
      console.log("✅ updateManageUser response:", response);

      toast.success("Profile updated successfully!");
    } catch (err: any) {
      console.error("❌ Update profile error:", err);
      toast.error(err?.response?.data?.message || "Failed to update profile");
    }
  };

  if (!user || loading || authLoading) {
    return <p className="text-center mt-10">Loading...</p>;
  }

  return (
    <main className="min-h-screen bg-gray-100 dark:bg-[#0f172a] transition-all duration-300">
      <div className="max-w-7xl mx-auto px-3 sm:px-6 lg:px-8 py-6 space-y-10">
        {/* Header */}
        <header className="flex flex-col sm:flex-row sm:items-end sm:justify-between gap-2 border-b pb-5">
          <div>
            <h1 className="text-3xl font-bold text-gray-800 dark:text-white">
              My Profile
            </h1>
            <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
              View and update your personal and account details
            </p>
          </div>
        </header>

        {/* Avatar + Info */}
        <section className="flex flex-col sm:flex-row items-center sm:items-start gap-6 bg-white dark:bg-gray-800 p-4 sm:p-6 rounded-2xl shadow-md">
          <div className="relative">
            <Image
              src={avatarUrl}
              alt="Avatar"
              width={120}
              height={120}
              className="rounded-full border-4 border-gray-300 dark:border-gray-600 object-cover"
            />
            <button
              onClick={() => fileInputRef.current?.click()}
              className="absolute bottom-0 right-0 bg-green-600 text-white px-2 py-1 text-xs rounded-full shadow hover:bg-green-700"
            >
              Change
            </button>
            <input
              type="file"
              ref={fileInputRef}
              accept="image/*"
              onChange={handleAvatarChange}
              className="hidden"
            />
          </div>

          <div className="text-center sm:text-left space-y-1">
            <h2 className="text-2xl font-semibold text-gray-800 dark:text-white">
              {formData.First_name} {formData.Last_name}
            </h2>
            <p className="text-gray-500 dark:text-gray-300 text-sm">
              {formData.role}
            </p>
            <p className="text-gray-400 text-xs">Employee ID: {user.user_id}</p>
          </div>
        </section>

        {/* Responsive grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
          <ProfileCard title="Basic Information">
            <EditableField
              label="First Name"
              value={formData.First_name || ""}
              onChange={(v) => handleChange("First_name", v)}
            />
            <EditableField
              label="Last Name"
              value={formData.Last_name || ""}
              onChange={(v) => handleChange("Last_name", v)}
            />
            <EditableField
              label="Email"
              value={formData.email || ""}
              onChange={(v) => handleChange("email", v)}
            />
            <EditableField
              label="Phone"
              value={formData.phone_number || ""}
              onChange={(v) => handleChange("phone_number", v)}
            />
            <EditableField
              label="Role"
              value={formData.role || ""}
              onChange={(v) => handleChange("role", v)}
            />
          </ProfileCard>

          <ProfileCard title="Personal Details">
            <EditableField
              label="Gender"
              value={formData.Gender || ""}
              onChange={(v) => handleChange("Gender", v)}
            />
            <EditableField
              label="Date of Birth"
              type="date"
              value={formData.DOB || ""}
              onChange={(v) => handleChange("DOB", v)}
            />
            <EditableField
              label="Address"
              value={formData.address || ""}
              onChange={(v) => handleChange("address", v)}
            />
            <EditableField
              label="Country"
              value={formData.country || ""}
              onChange={(v) => handleChange("country", v)}
            />
          </ProfileCard>

          <ProfileCard title="Work Information">
            <EditableField
              label="Department"
              value={formData.department || ""}
              onChange={(v) => handleChange("department", v)}
            />
            <EditableField
              label="Manager"
              value={formData.manager || ""}
              onChange={(v) => handleChange("manager", v)}
            />
            <EditableField
              label="Joined Date"
              type="date"
              value={formData.join_date || ""}
              onChange={(v) => handleChange("join_date", v)}
            />
            <EditableField
              label="Location"
              value={formData.location || ""}
              onChange={(v) => handleChange("location", v)}
            />
          </ProfileCard>

          <ProfileCard title="System Info">
            <FieldStatic
              label="Last Login"
              value={formData.last_activity || ""}
            />
            <FieldStatic label="Status" value={formData.emp_status || ""} />
            <FieldStatic label="Access Level" value={formData.role || ""} />
          </ProfileCard>
        </div>

        <div className="flex justify-end">
          <button
            onClick={handleSave}
            className="bg-green-600 hover:bg-green-700 text-white font-medium px-6 py-2 rounded-lg shadow transition"
          >
            Save Changes
          </button>
        </div>
      </div>
    </main>
  );
}

function ProfileCard({
  title,
  children,
}: {
  title: string;
  children: React.ReactNode;
}) {
  return (
    <div className="bg-white dark:bg-gray-800 rounded-2xl p-6 shadow space-y-4">
      <h3 className="text-xl font-semibold text-gray-800 dark:text-white mb-3 border-b border-gray-200 dark:border-gray-700 pb-2">
        {title}
      </h3>
      <div className="space-y-4">{children}</div>
    </div>
  );
}

function EditableField({
  label,
  value,
  onChange,
  type = "text",
}: {
  label: string;
  value: string;
  onChange: (v: string) => void;
  type?: string;
}) {
  return (
    <div>
      <label className="block text-sm font-medium text-gray-600 dark:text-gray-300 mb-1">
        {label}
      </label>
      <input
        type={type}
        value={value}
        onChange={(e) => onChange(e.target.value)}
        className="w-full px-4 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:outline-none focus:ring-2 focus:ring-green-500 transition"
      />
    </div>
  );
}

function FieldStatic({ label, value }: { label: string; value: string }) {
  return (
    <div>
      <label className="block text-sm font-medium text-gray-600 dark:text-gray-300 mb-1">
        {label}
      </label>
      <div className="px-4 py-2 text-sm rounded-lg bg-gray-100 dark:bg-gray-700 text-gray-800 dark:text-white break-words">
        {value || "—"}
      </div>
    </div>
  );
}
