'use client';

import Image from 'next/image';
import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { Mail } from 'lucide-react';
import { toast } from 'sonner';

const forgotSchema = z.object({
    email: z.string().min(1, 'Email is required').email('Invalid email address'),
});

type ForgotData = z.infer<typeof forgotSchema>;

export default function ForgotPasswordPage() {
    const [loading, setLoading] = useState(false);

    const {
        register,
        handleSubmit,
        formState: { errors },
    } = useForm<ForgotData>({
        resolver: zodResolver(forgotSchema),
    });

    const onSubmit = async (data: ForgotData) => {
        setLoading(true);
        try {
            const response = await fetch('/api/login/forgotpassword', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data),
            });

            const result = await response.json();
            if (response.ok) {
                toast.success(result.message);
            } else {
                toast.error(result.message || 'Failed to send reset link');
            }
        } catch {
            toast.error('Something went wrong');
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen flex flex-col md:flex-row bg-gradient-to-r from-green-100 to-white dark:from-gray-900 dark:to-gray-800">
            {/* Left full-height image */}
            <div className="hidden md:block relative w-full md:w-1/2 h-screen animate-fade-in-left">
                <Image
                    src="/login.jpg"
                    alt="Illustration"
                    fill
                    className="object-cover"
                    priority
                />
            </div>
            {/* Right forgot form */}
            <div className="w-full md:w-1/2 flex items-center justify-center px-4 py-10 min-h-screen md:min-h-0">
                <div className="w-full max-w-md bg-white dark:bg-gray-900 px-8 py-10 rounded-2xl shadow-xl border border-gray-200 dark:border-gray-700 animate-fade-in-up">
                    <Image
                        src="/image.png"
                        alt="IATT Logo"
                        width={120}
                        height={40}
                        priority
                        className="mx-auto mb-6"
                    />
                    <h2 className="text-xl font-bold mb-6 text-center text-gray-900 dark:text-gray-100">
                        Forgot Password
                    </h2>

                    <form onSubmit={handleSubmit(onSubmit)} className="space-y-5">
                        {/* Email */}
                        <div>
                            <label
                                htmlFor="email"
                                className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                            >
                                Registered Email
                            </label>
                            <div className="relative">
                                <span className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 dark:text-gray-500">
                                    <Mail className="w-4 h-4" />
                                </span>
                                <input
                                    id="email"
                                    type="email"
                                    placeholder="Enter your email"
                                    disabled={loading}
                                    {...register('email')}
                                    className="w-full pl-10 p-2 bg-gray-50 dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500 dark:focus:ring-green-400"
                                />
                            </div>
                            {errors.email && (
                                <p className="text-red-500 dark:text-red-400 text-sm mt-1">
                                    {errors.email.message}
                                </p>
                            )}
                        </div>

                        {/* Submit */}
                        <button
                            type="submit"
                            disabled={loading}
                            className="w-full bg-green-600 dark:bg-green-700 text-white py-2 rounded-lg hover:bg-green-700 dark:hover:bg-green-600 flex items-center justify-center disabled:opacity-50"
                        >
                            {loading && (
                                <span className="animate-spin border-2 border-white dark:border-gray-200 border-t-transparent rounded-full h-4 w-4 mr-2"></span>
                            )}
                            {loading ? 'Sending...' : 'Send Reset Link'}
                        </button>

                        {/* Back to login */}
                        <div className="text-sm text-center mt-4">
                            <a
                                href="/login"
                                className="text-green-600 dark:text-green-400 hover:underline"
                            >
                                Back to Login
                            </a>
                        </div>
                        {/* Footer links */}
                        <div className="text-xs text-center text-gray-400 dark:text-gray-500 mt-4">
                            <a href="#" className="hover:underline">
                                Privacy Policy
                            </a>{' '}
                            ·{' '}
                            <a href="#" className="hover:underline">
                                Terms of Service
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    );
}