"use client";

import React, { useEffect, useState } from "react";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  SearchOutlined,
  StarFilled,
  HomeOutlined,
  AppstoreOutlined,
  TableOutlined,
} from "@ant-design/icons";

import { MessageCircle, CheckCircle } from "lucide-react";
import { FaFileExport, FaFilePdf, FaFileExcel, FaTrash } from "react-icons/fa";

import Button from "antd/es/button";
import Form from "antd/es/form";
import Input from "antd/es/input";
import Modal from "antd/es/modal";
import Select from "antd/es/select";
import Tag from "antd/es/tag";
import Breadcrumb from "antd/es/breadcrumb";
import Card from "antd/es/card";
import DatePicker from "antd/es/date-picker";
import Switch from "antd/es/switch";
import Pagination from "antd/es/pagination";
import Dropdown from "antd/es/dropdown";
import Space from "antd/es/space";
import Tooltip from "antd/es/tooltip";
import type { ColumnsType } from "antd/es/table";
import type { Dayjs } from "dayjs";

import DataTable from "../../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../../components/common/GlassCard";
import { highlightMatch } from "../../components/common/HighLightMatch";

import {
  createTrainingFeedback,
  getAllTrainingFeedbacks,
  updateTrainingFeedback,
  updateFeedbackRatings,
  deleteTrainingFeedback,
  TrainingFeedbackData,
} from "@/app/api/trainingfeedback/trainingfeedback";

import {
  getAllTrainingSessions,
  TrainingSessionRecord,
} from "@/app/api/trainingsession/trainingsession";

import {
  getAllManageUsers,
  ManageUser,
} from "@/app/api/manageusers/manageusers";

const WidgetCard = dynamic(() => import("../../components/common/WidgetCard"), {
  ssr: false,
});
const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  { ssr: false }
);
const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  { ssr: false }
);

const { RangePicker } = DatePicker;

const ratingOptions = [
  "Excellent",
  "Very Good",
  "Good",
  "Average",
  "Poor",
] as const;
const durationOptions = ["Adequate", "Inadequate"] as const;
const booleanOptions = ["True", "False"] as const;

export default function TrainingFeedbackPage() {
  const [form] = Form.useForm();
  const [editForm] = Form.useForm();
  const [ratingsForm] = Form.useForm();

  const [feedbacks, setFeedbacks] = useState<TrainingFeedbackData[]>([]);
  const [sessions, setSessions] = useState<TrainingSessionRecord[]>([]);
  const [manageUsers, setManageUsers] = useState<ManageUser[]>([]);
  const [loading, setLoading] = useState(false);
  const [searchTerm, setSearchTerm] = useState("");
  const [dateRange, setDateRange] = useState<[Dayjs, Dayjs] | null>(null);
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);

  const [isAddModalOpen, setIsAddModalOpen] = useState(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [isRatingsModalOpen, setIsRatingsModalOpen] = useState(false);

  const [editingRecord, setEditingRecord] =
    useState<TrainingFeedbackData | null>(null);
  const [ratingsRecord, setRatingsRecord] =
    useState<TrainingFeedbackData | null>(null);

  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    title: "",
    message: "",
    onConfirm: () => {},
  });

  const isMobile = useIsMobile();

  useEffect(() => {
    fetchData();
  }, []);

  const fetchData = async () => {
    setLoading(true);
    try {
      const [fbData, sessionData, userData] = await Promise.all([
        getAllTrainingFeedbacks(),
        getAllTrainingSessions(),
        getAllManageUsers(),
      ]);

      const sorted = fbData.sort(
        (a, b) =>
          dayjs(b.created_at || "").unix() - dayjs(a.created_at || "").unix()
      );

      setFeedbacks(sorted);
      setSessions(sessionData);
      setManageUsers(userData);
    } catch (err) {
      toast.error("Failed to load data");
    } finally {
      setLoading(false);
    }
  };

  const openAddModal = () => {
    form.resetFields();
    setIsAddModalOpen(true);
  };

  const openEditModal = (record: TrainingFeedbackData) => {
    setEditingRecord(record);
    editForm.setFieldsValue({
      session_code: record.session_code,
      conducted_by: record.conducted_by_id || record.conducted_by,
      reviewed_by: record.reviewed_by_id || record.reviewed_by || "",
      program_attended: record.program_attended,
      venue: record.venue || "",
      date: record.date ? dayjs(record.date) : null,
      comments: record.comments || "",
    });
    setIsEditModalOpen(true);
  };

  const openRatingsModal = (record: TrainingFeedbackData) => {
    setRatingsRecord(record);
    ratingsForm.setFieldsValue({
      training_content_rating: record.training_content_rating,
      training_method_rating: record.training_method_rating,
      trainer_effectiveness: record.trainer_effectiveness,
      duration_adequacy: record.duration_adequacy,
      expectations_satisfied: record.expectations_satisfied ? "True" : "False",
      new_skills_gained: record.new_skills_gained ? "True" : "False",
      applicability_to_job: record.applicability_to_job ? "True" : "False",
      further_training_needed: record.further_training_needed
        ? "True"
        : "False",
    });
    setIsRatingsModalOpen(true);
  };

  const handleAdd = async () => {
    try {
      const values = await form.validateFields();

      const payload = {
        session_code: values.session_code,
        conducted_by: values.conducted_by,
        reviewed_by: values.reviewed_by || null,
        program_attended: values.program_attended,
        venue: values.venue || "",
        date: values.date.format("YYYY-MM-DD"),
        training_content_rating: values.training_content_rating,
        training_method_rating: values.training_method_rating,
        trainer_effectiveness: values.trainer_effectiveness,
        duration_adequacy: values.duration_adequacy,
        expectations_satisfied: values.expectations_satisfied === "True",
        new_skills_gained: values.new_skills_gained === "True",
        applicability_to_job: values.applicability_to_job === "True",
        further_training_needed: values.further_training_needed === "True",
        comments: values.comments || "",
      };

      await createTrainingFeedback(payload);
      toast.success("Feedback submitted successfully");
      await fetchData();
      setIsAddModalOpen(false);
    } catch (err: any) {
      if (err.errorFields) {
        toast.error("Please fill all required fields");
      } else {
        toast.error("Failed to submit feedback");
      }
    }
  };

  const handleEdit = async () => {
    if (!editingRecord) return;
    try {
      const values = await editForm.validateFields();
      await updateTrainingFeedback(editingRecord.feedback_code, {
        conducted_by: values.conducted_by,
        reviewed_by: values.reviewed_by || null,
        program_attended: values.program_attended,
        venue: values.venue || "",
        date: values.date.format("YYYY-MM-DD"),
        comments: values.comments || "",
      });
      toast.success("Updated successfully");
      await fetchData();
      setIsEditModalOpen(false);
    } catch {
      toast.error("Update failed");
    }
  };

  const handleRatingsUpdate = async () => {
    if (!ratingsRecord) return;
    try {
      const values = await ratingsForm.validateFields();
      await updateFeedbackRatings(ratingsRecord.feedback_code, {
        training_content_rating: values.training_content_rating,
        training_method_rating: values.training_method_rating,
        trainer_effectiveness: values.trainer_effectiveness,
        duration_adequacy: values.duration_adequacy,
        expectations_satisfied: values.expectations_satisfied === "True",
        new_skills_gained: values.new_skills_gained === "True",
        applicability_to_job: values.applicability_to_job === "True",
        further_training_needed: values.further_training_needed === "True",
      });
      toast.success("Ratings updated");
      await fetchData();
      setIsRatingsModalOpen(false);
    } catch {
      toast.error("Update failed");
    }
  };

  const handleDelete = (code: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Feedback",
      message: "Are you sure you want to delete this feedback?",
      onConfirm: async () => {
        await deleteTrainingFeedback(code);
        await fetchData();
        toast.success("Deleted successfully");
        setConfirmDialog((prev) => ({ ...prev, open: false }));
      },
    });
  };

  const handleBulkDelete = () => {
    if (!selectedRows.length) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Feedbacks",
      message: `Are you sure you want to delete ${selectedRows.length} feedback(s)?`,
      onConfirm: async () => {
        await Promise.all(selectedRows.map(deleteTrainingFeedback));
        await fetchData();
        setSelectedRows([]);
        toast.success("Feedbacks deleted successfully");
        setConfirmDialog((prev) => ({ ...prev, open: false }));
      },
    });
  };

  const renderRating = (text: string) => {
    const colorMap: Record<string, string> = {
      Excellent: "green",
      "Very Good": "lime",
      Good: "blue",
      Average: "gold",
      Poor: "red",
    };
    return <Tag color={colorMap[text] || "default"}>{text || "-"}</Tag>;
  };

  const renderBoolean = (val: boolean) =>
    val ? <Tag color="green">True</Tag> : <Tag color="red">False</Tag>;

  const filtered = feedbacks.filter((f) => {
    const s = searchTerm.toLowerCase();
    return (
      (f.feedback_code.toLowerCase().includes(s) ||
        f.program_attended.toLowerCase().includes(s) ||
        f.employee_name?.toLowerCase().includes(s) ||
        f.conducted_by?.toLowerCase().includes(s)) &&
      (!dateRange ||
        (f.date &&
          dayjs(f.date).isBetween(dateRange[0], dateRange[1], "day", "[]")))
    );
  });

  const paginated = filtered.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<TrainingFeedbackData> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) =>
            setSelectedRows(
              e.target.checked ? filtered.map((f) => f.feedback_code) : []
            )
          }
          className="w-4 h-4"
        />
      ),
      key: "selection",
      width: 44,
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.feedback_code)}
          onChange={(e) =>
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.feedback_code]
                : prev.filter((id) => id !== record.feedback_code)
            )
          }
          className="w-4 h-4"
        />
      ),
    },
    {
      title: "Code",
      dataIndex: "feedback_code",
      key: "code",
      sorter: (a, b) => a.feedback_code.localeCompare(b.feedback_code),
    },
    {
      title: "Employee",
      dataIndex: "employee_name",
      key: "emp",
      render: (t) => highlightMatch(t || "-", searchTerm),
    },
    { title: "Program", dataIndex: "program_attended", key: "prog" },
    { title: "Conducted By", dataIndex: "conducted_by", key: "cond" },
    { title: "Reviewed By", dataIndex: "reviewed_by", key: "rev" },
    { title: "Venue", dataIndex: "venue", key: "venue" },
    {
      title: "Date",
      dataIndex: "date",
      key: "date",
      render: (v) => (v ? dayjs(v).format("DD-MM-YYYY") : "-"),
    },
    {
      title: "Content",
      dataIndex: "training_content_rating",
      key: "content",
      render: renderRating,
    },
    {
      title: "Method",
      dataIndex: "training_method_rating",
      key: "method",
      render: renderRating,
    },
    {
      title: "Trainer",
      dataIndex: "trainer_effectiveness",
      key: "trainer",
      render: renderRating,
    },
    {
      title: "Duration",
      dataIndex: "duration_adequacy",
      key: "dur",
      render: renderRating,
    },
    {
      title: "Expectations",
      dataIndex: "expectations_satisfied",
      key: "exp",
      render: renderBoolean,
    },
    {
      title: "New Skills",
      dataIndex: "new_skills_gained",
      key: "skill",
      render: renderBoolean,
    },
    {
      title: "Job Applicable",
      dataIndex: "applicability_to_job",
      key: "job",
      render: renderBoolean,
    },
    {
      title: "Further Needed",
      dataIndex: "further_training_needed",
      key: "need",
      render: renderBoolean,
    },
    {
      title: "Comments",
      dataIndex: "comments",
      key: "comments",
      render: (t) => (
        <Tooltip title={t}>
          <span className="truncate block max-w-[150px]">{t || "-"}</span>
        </Tooltip>
      ),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, r) => (
        <Space size="small">
          <Tooltip title="Edit Info">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(r)}
              size="small"
            />
          </Tooltip>
          <Tooltip title="Edit Ratings">
            <Button
              icon={<StarFilled />}
              type="primary"
              onClick={() => openRatingsModal(r)}
              size="small"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              danger
              onClick={() => handleDelete(r.feedback_code)}
              size="small"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "code", label: "Feedback Code" },
    { key: "emp", label: "Employee" },
    { key: "prog", label: "Program" },
    { key: "cond", label: "Conducted By" },
    { key: "rev", label: "Reviewed By" },
    { key: "venue", label: "Venue" },
    { key: "date", label: "Date" },
    { key: "content", label: "Content Rating" },
    { key: "method", label: "Method Rating" },
    { key: "trainer", label: "Trainer Rating" },
    { key: "dur", label: "Duration" },
    { key: "exp", label: "Expectations Met" },
    { key: "skill", label: "New Skills" },
    { key: "job", label: "Job Applicable" },
    { key: "need", label: "Further Needed" },
    { key: "comments", label: "Comments" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <MessageCircle className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Training Feedback</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({feedbacks.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Collect and analyze participant feedback for training sessions.
        </p>

        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            { title: <HomeOutlined />, href: "/hrm" },
            { title: "Training Management" },
            { title: "Feedback" },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total"
            value={feedbacks.length}
            icon={<MessageCircle />}
            color="blue"
          />
          <WidgetCard
            title="Excellent/Very Good"
            value={
              feedbacks.filter((f) =>
                ["Excellent", "Very Good"].includes(f.training_content_rating)
              ).length
            }
            icon={<StarFilled className="text-yellow-500" />}
            color="green"
          />
          <WidgetCard
            title="Expectations Met"
            value={feedbacks.filter((f) => f.expectations_satisfied).length}
            icon={<CheckCircle className="text-green-600" />}
            color="purple"
          />
          <WidgetCard
            title="Further Needed"
            value={feedbacks.filter((f) => f.further_training_needed).length}
            icon={<MessageCircle />}
            color="orange"
          />
        </div>

        <div className="mt-4">
          <Card title="Training Feedback List" className="p-0 mb-6">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search by code, program, employee..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <div className="flex gap-2">
                  <Dropdown
                    menu={{
                      items: [
                        {
                          key: "pdf",
                          label: (
                            <>
                              <FaFilePdf className="text-red-500" /> Export PDF
                            </>
                          ),
                        },
                        {
                          key: "excel",
                          label: (
                            <>
                              <FaFileExcel className="text-green-500" /> Export
                              Excel
                            </>
                          ),
                        },
                      ],
                    }}
                  >
                    <Button icon={<FaFileExport />}>Export</Button>
                  </Dropdown>
                  <Button
                    type="primary"
                    icon={<PlusOutlined />}
                    onClick={openAddModal}
                    className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                  >
                    Add Feedback
                  </Button>
                </div>
              </div>

              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto"></div>
                <div className="w-full flex flex-col lg:flex-row lg:justify-between lg:items-center gap-4">
                  {/* RIGHT SIDE — OTHER ACTIONS */}
                  <div className="flex flex-col lg:flex-row lg:items-center lg:justify-end gap-4 w-full">
                    <div className="flex flex-wrap items-center gap-4 w-full lg:w-auto">
                      <ColumnVisibilityToggle
                        columns={columnKeys}
                        labels={labelsMap}
                        storageKey="feedback_table_visibility"
                        onChange={setVisibleColumnKeys}
                      />

                      <RangePicker
                        value={dateRange}
                        onChange={(dates) =>
                          setDateRange(dates as [Dayjs, Dayjs] | null)
                        }
                        className="w-full sm:w-64 border-gray-300 dark:border-gray-600"
                      />

                      <Button
                        onClick={handleBulkDelete}
                        disabled={selectedRows.length === 0}
                        danger
                        icon={<FaTrash />}
                        className="w-full sm:w-auto"
                      >
                        Bulk Delete
                      </Button>

                      <Switch
                        checkedChildren={<AppstoreOutlined />}
                        unCheckedChildren={<TableOutlined />}
                        checked={isGridView}
                        onChange={setIsGridView}
                        className="hidden sm:block"
                      />
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginated.map((f) => (
                  <GlassCard
                    key={f.feedback_code}
                    gradient
                    blurStrength="lg"
                    hoverLift
                    color="success"
                  >
                    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full space-y-2">
                      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2">
                        <h3 className="text-lg font-semibold text-gray-800 dark:text-white">
                          {f.program_attended || "N/A"}
                        </h3>
                        {renderRating(f.training_content_rating)}
                      </div>
                      <p className="text-sm text-gray-600 dark:text-gray-300">
                        <strong>Code:</strong> {f.feedback_code}
                      </p>
                      <p className="text-sm text-gray-600 dark:text-gray-300">
                        <strong>Employee:</strong> {f.employee_name || "N/A"}
                      </p>
                      <p className="text-sm text-gray-600 dark:text-gray-300">
                        <strong>Date:</strong>{" "}
                        {f.date ? dayjs(f.date).format("DD-MM-YYYY") : "-"}
                      </p>
                      <div className="flex gap-2 pt-2 border-t dark:border-gray-700">
                        <Button
                          size="small"
                          icon={<EditOutlined />}
                          onClick={() => openEditModal(f)}
                        ></Button>
                        <Button
                          size="small"
                          icon={<StarFilled />}
                          type="primary"
                          onClick={() => openRatingsModal(f)}
                        ></Button>
                        <Button
                          size="small"
                          danger
                          icon={<DeleteOutlined />}
                          onClick={() => handleDelete(f.feedback_code)}
                        ></Button>
                      </div>
                    </div>
                  </GlassCard>
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable
                  columns={columns.filter((col) =>
                    visibleColumnKeys.includes(col.key as string)
                  )}
                  dataSource={paginated}
                  rowKey="feedback_code"
                  loading={loading}
                  pagination={false}
                />
              </div>
            )}

            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filtered.length)}
                </strong>{" "}
                of <strong>{filtered.length}</strong> feedbacks
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filtered.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        {/* ADD / EDIT MODAL */}
        {/* ADD / EDIT MODAL */}
        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-300 dark:border-gray-600">
                <MessageCircle className="w-5 h-5 text-green-600" />
                <span>Add Training Feedback</span>
              </div>
            }
            open={isAddModalOpen}
            onOk={handleAdd}
            onCancel={() => {
              setIsAddModalOpen(false);
              setIsEditModalOpen(false);
              setEditingRecord(null);
            }}
            width="95vw"
            className="!top-5 font-semibold"
            styles={{ body: { maxHeight: "80vh", overflowY: "auto" } }}
            destroyOnClose
          >
            <Form
              form={editingRecord ? editForm : form}
              layout="vertical"
              className="w-full px-4"
            >
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {/* Session Code (Only Add Mode) */}
                {!editingRecord && (
                  <Form.Item
                    name="session_code"
                    label="Session Code *"
                    rules={[{ required: true }]}
                  >
                    <Select showSearch placeholder="Select session">
                      {sessions.map((s) => (
                        <Select.Option
                          key={s.session_code}
                          value={s.session_code}
                        >
                          {s.session_code} - {s.topic} (
                          {dayjs(s.session_date).format("DD-MM-YYYY")})
                        </Select.Option>
                      ))}
                    </Select>
                  </Form.Item>
                )}

                {/* Conducted By */}
                <Form.Item
                  name="conducted_by"
                  label="Conducted By *"
                  rules={[{ required: true }]}
                >
                  <Select showSearch placeholder="Select user">
                    {manageUsers.map((u) => (
                      <Select.Option key={u.id} value={u.id}>
                        {u.First_name} {u.Last_name}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                {/* Reviewed By */}
                <Form.Item name="reviewed_by" label="Reviewed By">
                  <Select allowClear showSearch placeholder="Select reviewer">
                    {manageUsers.map((u) => (
                      <Select.Option key={u.id} value={u.id}>
                        {u.First_name} {u.Last_name} ({u.email})
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                {/* Program Attended */}
                <Form.Item
                  name="program_attended"
                  label="Program Attended *"
                  rules={[{ required: true }]}
                >
                  <Input placeholder="Program name" />
                </Form.Item>

                {/* Venue */}
                <Form.Item name="venue" label="Venue">
                  <Input placeholder="Venue" />
                </Form.Item>

                {/* Date */}
                <Form.Item
                  name="date"
                  label="Date *"
                  rules={[{ required: true }]}
                >
                  <DatePicker className="w-full" format="YYYY-MM-DD" />
                </Form.Item>

                {/* 🔥 NEW RATING FIELDS INSIDE SAME MODAL */}

                <Form.Item
                  name="training_content_rating"
                  label="Content Rating *"
                  rules={[{ required: true }]}
                >
                  <Select>
                    {ratingOptions.map((v) => (
                      <Select.Option key={v} value={v}>
                        {v}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item
                  name="training_method_rating"
                  label="Method Rating *"
                  rules={[{ required: true }]}
                >
                  <Select>
                    {ratingOptions.map((v) => (
                      <Select.Option key={v} value={v}>
                        {v}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item
                  name="trainer_effectiveness"
                  label="Trainer Effectiveness *"
                  rules={[{ required: true }]}
                >
                  <Select>
                    {ratingOptions.map((v) => (
                      <Select.Option key={v} value={v}>
                        {v}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item
                  name="duration_adequacy"
                  label="Duration Adequacy *"
                  rules={[{ required: true }]}
                >
                  <Select>
                    {durationOptions.map((v) => (
                      <Select.Option key={v} value={v}>
                        {v}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item
                  name="expectations_satisfied"
                  label="Expectations Satisfied *"
                  rules={[{ required: true }]}
                >
                  <Select>
                    {booleanOptions.map((v) => (
                      <Select.Option key={v} value={v}>
                        {v}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item
                  name="new_skills_gained"
                  label="New Skills Gained *"
                  rules={[{ required: true }]}
                >
                  <Select>
                    {booleanOptions.map((v) => (
                      <Select.Option key={v} value={v}>
                        {v}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item
                  name="applicability_to_job"
                  label="Job Applicability *"
                  rules={[{ required: true }]}
                >
                  <Select>
                    {booleanOptions.map((v) => (
                      <Select.Option key={v} value={v}>
                        {v}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item
                  name="further_training_needed"
                  label="Further Training?"
                >
                  <Select>
                    {booleanOptions.map((v) => (
                      <Select.Option key={v} value={v}>
                        {v}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                {/* Comments */}
                <Form.Item name="comments" label="Comments">
                  <Input.TextArea rows={3} placeholder="Add comments" />
                </Form.Item>
              </div>
            </Form>
          </Modal>
        </motion.div>

        {/* EDIT INFO MODAL (ONLY EDIT FIELDS SHOWN) */}
        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-300 dark:border-gray-600">
                <MessageCircle className="w-5 h-5 text-green-600" />
                <span>Edit Training Feedback</span>
              </div>
            }
            open={isEditModalOpen}
            onOk={handleEdit}
            onCancel={() => {
              setIsEditModalOpen(false);
              setEditingRecord(null);
            }}
            width="90vw"
            className="!top-5 font-semibold"
            styles={{ body: { maxHeight: "80vh", overflowY: "auto" } }}
            destroyOnClose
          >
            <Form form={editForm} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {/* feedback_code READ ONLY */}
                <Form.Item label="Feedback Code *">
                  <Input value={editingRecord?.feedback_code} disabled />
                </Form.Item>

                {/* Conducted By */}
                <Form.Item
                  name="conducted_by"
                  label="Conducted By *"
                  rules={[{ required: true }]}
                >
                  <Select showSearch placeholder="Select user">
                    {manageUsers.map((u) => (
                      <Select.Option key={u.id} value={u.id}>
                        {u.First_name} {u.Last_name}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                {/* Reviewed By */}
                <Form.Item name="reviewed_by" label="Reviewed By">
                  <Select allowClear showSearch placeholder="Select reviewer">
                    {manageUsers.map((u) => (
                      <Select.Option key={u.id} value={u.id}>
                        {u.First_name} {u.Last_name} ({u.email})
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                {/* Program Attended */}
                <Form.Item
                  name="program_attended"
                  label="Program Attended *"
                  rules={[{ required: true }]}
                >
                  <Input />
                </Form.Item>

                {/* Venue */}
                <Form.Item name="venue" label="Venue">
                  <Input />
                </Form.Item>

                {/* Date */}
                <Form.Item
                  name="date"
                  label="Date *"
                  rules={[{ required: true }]}
                >
                  <DatePicker className="w-full" format="YYYY-MM-DD" />
                </Form.Item>

                {/* Comments */}
                <Form.Item name="comments" label="Comments">
                  <Input.TextArea rows={3} placeholder="Enter comments" />
                </Form.Item>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog((prev) => ({ ...prev, open: false }))}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
