"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
  SearchOutlined,
  InfoCircleOutlined,
  StarFilled,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useRouter } from "next/navigation";
import {
  createCompetencyEvaluation,
  deleteCompetencyEvaluation,
  getAllCompetencyEvaluations,
  updateCompetencyEvaluation,
  CompetencyEvaluation,
  updateEvaluationResult,
} from "@/app/api/trainingevaluations/trainingevaluations";
import { getAllEmployees, Employee } from "@/app/api/employee/employee";
import {
  getAllManageUsers,
  ManageUser,
} from "@/app/api/manageusers/manageusers";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../../components/common/GlassCard";
import { highlightMatch } from "../../components/common/HighLightMatch";
import { FaTrash } from "react-icons/fa";
import { CheckSquare, FileCheck } from "lucide-react";
import { LiaUserEditSolid } from "react-icons/lia";

const WidgetCard = dynamic(() => import("../../components/common/WidgetCard"), {
  ssr: false,
});
const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  { ssr: false }
);
const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  { ssr: false }
);

const { RangePicker } = DatePicker;
const result = ["Competent", "Non-Competent"] as const;
const evaluationMethod = ["Test", "Observation", "Interview"] as const;
const statusOptions = ["Approved", "Pending", "Rejected"] as const;
const competencyLevelOptions = [
  "Beginner",
  "Intermediate",
  "Advanced",
] as const;

export default function CompetencyPage() {
  const [form] = Form.useForm();
  const [approvalForm] = Form.useForm();
  const [evaluations, setEvaluations] = useState<CompetencyEvaluation[]>([]);
  const [employees, setEmployees] = useState<Employee[]>([]);
  const [users, setUsers] = useState<ManageUser[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [isApprovalModalOpen, setIsApprovalModalOpen] = useState(false);
  const [editingEvaluation, setEditingEvaluation] =
    useState<CompetencyEvaluation | null>(null);
  const [editingApprovalEvaluation, setEditingApprovalEvaluation] =
    useState<CompetencyEvaluation | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });
  const [loading, setLoading] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    competencyLevel: undefined as string | undefined,
    approvedStatus: undefined as string | undefined,
  });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [forceRender, setForceRender] = useState(0);
  const isMobile = useIsMobile();
  const router = useRouter();

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        const [evaluationData, employeeData, userData] = await Promise.all([
          getAllCompetencyEvaluations(),
          getAllEmployees(),
          getAllManageUsers(),
        ]);
        const sorted = evaluationData.sort(
          (a: any, b: any) =>
            new Date(b.created_at || "").getTime() -
            new Date(a.created_at || "").getTime()
        );
        setEvaluations(sorted);
        setEmployees(employeeData);
        setUsers(userData.filter((u) => !u.is_delete));
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  const openAddModal = () => {
    setEditingEvaluation(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (evaluation: CompetencyEvaluation) => {
    setEditingEvaluation(evaluation);
    form.resetFields();
    form.setFieldsValue({
      evaluationDate: evaluation.evaluation_date
        ? dayjs(evaluation.evaluation_date)
        : null,
      competencyArea: evaluation.competency_area || "",
      evaluationMethod: evaluation.evaluation_method || "",
      score: evaluation.score || "",
      competencyLevel: evaluation.competency_level || "",
      remarks: evaluation.remarks || "",
      nextReviewDate: evaluation.next_review_date
        ? dayjs(evaluation.next_review_date)
        : null,
    });
    setIsModalOpen(true);
  };

  const openApprovalModal = (evaluation: CompetencyEvaluation) => {
    setEditingApprovalEvaluation(evaluation);
    approvalForm.resetFields();
    approvalForm.setFieldsValue({
      approvedStatus: evaluation.approved_status || "Pending",
      result: evaluation.result || undefined,
      comments: evaluation.comments || "",
      rejectionReason:
        evaluation.approved_status === "Rejected"
          ? evaluation.rejection_reason || ""
          : "",
      approvedBy: evaluation.approved_by_name || "",
      approvedDate: evaluation.approved_date
        ? dayjs(evaluation.approved_date)
        : null,
    });
    setForceRender((prev) => prev + 1);
    setIsApprovalModalOpen(true);
  };

  // DELETE SINGLE
  const handleDelete = (evaluationCode: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Competency Evaluation",
      message: "Are you sure you want to delete this evaluation?",
      onConfirm: async () => {
        try {
          await deleteCompetencyEvaluation(evaluationCode);
          setEvaluations((prev) =>
            prev.filter((e) => e.evaluation_code !== evaluationCode)
          );
          // Success toast comes from API file
        } catch {
          // Error toast already shown in API file
        }
      },
    });
  };

  // BULK DELETE
  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;

    setConfirmDialog({
      open: true,
      title: "Delete Multiple Evaluations",
      message: `Delete ${selectedRows.length} selected evaluation(s)?`,
      onConfirm: async () => {
        try {
          await Promise.all(selectedRows.map(deleteCompetencyEvaluation));
          setEvaluations((prev) =>
            prev.filter((e) => !selectedRows.includes(e.evaluation_code!))
          );
          setSelectedRows([]);
          // Success toast from API
        } catch {
          // Error from API
        }
      },
    });
  };

  const fetchData = async () => {
    try {
      const data = await getAllCompetencyEvaluations();
      const sorted = data.sort(
        (a: any, b: any) =>
          new Date(b.created_at || "").getTime() -
          new Date(a.created_at || "").getTime()
      );
      setEvaluations(sorted);
    } catch {
      // Error already shown in API file
    }
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();

      let employeeCode: string;
      if (editingEvaluation) {
        employeeCode = editingEvaluation.employee_code;
      } else {
        const selectedName = values.employeeName?.trim();
        const found = employees.find(
          (e) => `${e.first_name} ${e.last_name || ""}`.trim() === selectedName
        );
        if (!found) {
          toast.error("Please select a valid employee");
          return;
        }
        employeeCode = found.employee_code;
      }

      const payload: any = {
        employee_code: employeeCode,
        evaluation_date: values.evaluationDate.format("YYYY-MM-DD"),
        competency_area: values.competencyArea,
        evaluation_method: values.evaluationMethod || null,
        score: values.score || null,
        competency_level: values.competencyLevel || null,
        remarks: values.remarks || null,
        next_review_date: values.nextReviewDate
          ? values.nextReviewDate.format("YYYY-MM-DD")
          : null,
      };

      if (!editingEvaluation && values.result) {
        payload.result = values.result;
      }

      if (editingEvaluation) {
        await updateCompetencyEvaluation(
          editingEvaluation.evaluation_code!,
          payload
        );
      } else {
        await createCompetencyEvaluation(payload);
      }

      await fetchData();
      setIsModalOpen(false);
      form.resetFields();
      setEditingEvaluation(null);
      // Success toast comes from API file
    } catch (err: any) {
      if (err.errorFields?.length > 0) {
        const missing = err.errorFields.map((f: any) => {
          const map: Record<string, string> = {
            employeeName: "Employee Name",
            evaluationDate: "Evaluation Date",
            competencyArea: "Competency Area",
          };
          return map[f.name[0]] || f.name[0];
        });
        toast.error(
          missing.length > 1
            ? `${missing.join(", ")} are required`
            : `${missing[0]} is required`
        );
      }
      // Backend errors shown in API file
    }
  };

  const handleApprovalSubmit = async () => {
    if (!editingApprovalEvaluation) return;

    try {
      const values = await approvalForm.validateFields();

      const payload: any = {
        approved_status: values.approvedStatus,
        result: values.result,
        comments: values.comments || null,
        rejection_reason:
          values.approvedStatus === "Rejected"
            ? values.rejectionReason || null
            : null,
        approved_by_user_id: values.approvedBy
          ? users.find(
              (u) =>
                `${u.First_name} ${u.Last_name || ""}`.trim() ===
                values.approvedBy
            )?.id || null
          : null,
        approved_date: values.approvedDate
          ? values.approvedDate.format("YYYY-MM-DD")
          : null,
      };

      await updateEvaluationResult(
        editingApprovalEvaluation.evaluation_code!,
        payload
      );
      await fetchData();
      setIsApprovalModalOpen(false);
      approvalForm.resetFields();
      // Success toast comes from API
    } catch {
      // Error shown in API
    }
  };

  const renderTag = (value: string) => {
    const tagConfig = [
      { value: "Approved", color: "green", icon: <CheckCircleOutlined /> },
      { value: "Pending", color: "gold", icon: <MinusCircleOutlined /> },
      { value: "Rejected", color: "red", icon: <CloseCircleOutlined /> },
      { value: "Beginner", color: "blue", icon: <CheckCircleOutlined /> },
      { value: "Intermediate", color: "cyan", icon: <InfoCircleOutlined /> },
      { value: "Advanced", color: "purple", icon: <StarFilled /> },
      { value: "Competent", color: "green", icon: <CheckCircleOutlined /> },
      { value: "Non-Competent", color: "red", icon: <CloseCircleOutlined /> },
    ];

    const config = tagConfig.find((c) => c.value === value);

    return (
      <Tag color={config?.color || "default"} icon={config?.icon || null}>
        {value || "-"}
      </Tag>
    );
  };

  const filteredEvaluations = evaluations.filter((evaluation) => {
    const matchesSearch =
      evaluation.employee_name
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      evaluation.evaluation_code
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      evaluation.competency_area
        .toLowerCase()
        .includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (evaluation.evaluation_date &&
        (!dateRange[0] ||
          dayjs(evaluation.evaluation_date).isSame(dateRange[0], "day") ||
          dayjs(evaluation.evaluation_date).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(evaluation.evaluation_date).isSame(dateRange[1], "day") ||
          dayjs(evaluation.evaluation_date).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.competencyLevel ||
        evaluation.competency_level === advancedFilters.competencyLevel) &&
      (!advancedFilters.approvedStatus ||
        evaluation.approved_status === advancedFilters.approvedStatus);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedEvaluations = filteredEvaluations.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const total = evaluations.length;
  const approved = evaluations.filter(
    (e) => e.approved_status === "Approved"
  ).length;
  const pending = evaluations.filter(
    (e) => e.approved_status === "Pending"
  ).length;
  const advanced = evaluations.filter(
    (e) => e.competency_level === "Advanced"
  ).length;
  const columns: ColumnsType<CompetencyEvaluation> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked
                ? filteredEvaluations
                    .map((e) => e.evaluation_code!)
                    .filter(Boolean)
                : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) =>
        record.evaluation_code ? (
          <input
            type="checkbox"
            checked={selectedRows.includes(record.evaluation_code!)}
            onChange={(e) => {
              if (!record.evaluation_code) return;
              setSelectedRows((prev) =>
                e.target.checked
                  ? [...prev, record.evaluation_code!]
                  : prev.filter((id) => id !== record.evaluation_code!)
              );
            }}
            className="w-4 h-4"
          />
        ) : null,
      width: 44,
    },
    {
      title: "Evaluation Code",
      dataIndex: "evaluation_code",
      key: "evaluation_code",
      sorter: (a, b) =>
        (a.evaluation_code || "").localeCompare(b.evaluation_code || ""),
    },
    {
      title: "Employee Name",
      key: "employee_name",
      sorter: (a, b) =>
        (a.employee_name || "").localeCompare(b.employee_name || ""),
      render: (_, record) => (
        <span>{highlightMatch(record.employee_name || "", searchTerm)}</span>
      ),
    },
    {
      title: "Competency Area",
      dataIndex: "competency_area",
      key: "competency_area",
      sorter: (a, b) =>
        (a.competency_area || "").localeCompare(b.competency_area || ""),
    },
    {
      title: "Evaluation Date",
      dataIndex: "evaluation_date",
      key: "evaluation_date",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY")} (${dayjs(value).fromNow()})`
          : "-",
    },
    {
      title: "Competency Level",
      dataIndex: "competency_level",
      key: "competency_level",
      render: renderTag,
    },
    {
      title: "Score",
      dataIndex: "score",
      key: "score",
      render: (text) => text || "-",
    },
    {
      title: "Approved Status",
      dataIndex: "approved_status",
      key: "approved_status",
      render: renderTag,
    },
    {
      title: "Evaluated By",
      dataIndex: "evaluated_by_name",
      key: "evaluated_by_name",
      render: (text) => text || "-",
    },
    {
      title: "Approved By",
      dataIndex: "approved_by_name",
      key: "approved_by_name",
      render: (text) => text || "-",
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Evaluation Method",
      dataIndex: "evaluation_method",
      key: "evaluation_method",
      render: (text) => text || "-",
    },
    {
      title: "Next Review Date",
      dataIndex: "next_review_date",
      key: "next_review_date",
      render: (value: string) =>
        value ? dayjs(value).format("DD-MM-YYYY") : "-",
    },
    {
      title: "Remarks",
      dataIndex: "remarks",
      key: "remarks",
      render: (text) => text || "-",
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) =>
        record.evaluation_code ? (
          <Space size="small">
            <Tooltip title="Edit">
              <Button
                icon={<EditOutlined />}
                onClick={() => openEditModal(record)}
                size="small"
                type="primary"
                className="bg-blue-600 hover:bg-blue-700"
              />
            </Tooltip>
            <Tooltip title="Edit Approval">
              <Button
                icon={<LiaUserEditSolid />}
                onClick={() => openApprovalModal(record)}
                size="small"
                className="text-purple-600"
              />
            </Tooltip>
            <Tooltip title="Delete">
              <Button
                icon={<DeleteOutlined />}
                onClick={() => handleDelete(record.evaluation_code!)}
                danger
                size="small"
                className="text-red-600"
              />
            </Tooltip>
          </Space>
        ) : null,
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "evaluation_code", label: "Evaluation Code" },
    { key: "employee_name", label: "Employee Name" },
    { key: "competency_area", label: "Competency Area" },
    { key: "evaluation_date", label: "Evaluation Date" },
    { key: "competency_level", label: "Competency Level" },
    { key: "score", label: "Score" },
    { key: "approved_status", label: "Approved Status" },
    { key: "evaluated_by_name", label: "Evaluated By" },
    { key: "approved_by_name", label: "Approved By" },
    { key: "created_at", label: "Created At" },
    { key: "evaluation_method", label: "Evaluation Method" },
    { key: "next_review_date", label: "Next Review Date" },
    { key: "remarks", label: "Remarks" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const EvaluationCard = ({
    evaluation,
  }: {
    evaluation: CompetencyEvaluation;
  }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="success">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full space-y-2">
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2">
          <h3 className="text-lg font-semibold text-gray-800 dark:text-white">
            {evaluation.employee_name || "N/A"}
          </h3>
          <Tag color="blue">{evaluation.competency_area || "N/A"}</Tag>
        </div>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Evaluation Code:</strong>{" "}
          {evaluation.evaluation_code || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Evaluation Date:</strong>{" "}
          {evaluation.evaluation_date
            ? dayjs(evaluation.evaluation_date).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Competency Level:</strong>{" "}
          {renderTag(evaluation.competency_level || "N/A")}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Score:</strong> {evaluation.score || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Approved Status:</strong>{" "}
          {renderTag(evaluation.approved_status || "N/A")}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Evaluated By:</strong> {evaluation.evaluated_by_name || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Approved By:</strong> {evaluation.approved_by_name || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Result:</strong> {evaluation.result || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Evaluation Method:</strong>{" "}
          {evaluation.evaluation_method || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Next Review Date:</strong>{" "}
          {evaluation.next_review_date
            ? dayjs(evaluation.next_review_date).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Remarks:</strong> {evaluation.remarks || "N/A"}
        </p>
        <p className="text-xs text-gray-500 dark:text-gray-400 mt-2">
          <strong>Created:</strong>{" "}
          {evaluation.created_at
            ? dayjs(evaluation.created_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <div className="flex gap-2 pt-2 border-t dark:border-gray-700">
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(evaluation)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() =>
                evaluation.evaluation_code &&
                handleDelete(evaluation.evaluation_code)
              }
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
          <Tooltip title="Edit Approval">
            <Button
              icon={<LiaUserEditSolid />}
              onClick={() => openApprovalModal(evaluation)}
              size="small"
              className="text-purple-600"
            />
          </Tooltip>
        </div>
      </div>
    </GlassCard>
  );

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <FileCheck className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Competency Evaluations</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({evaluations.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage competency evaluations for employees.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm",
            },
            {
              title: "Master Data",
            },
            {
              title: "Competency Evaluations",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Evaluations"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={10}
            description="All competency evaluations"
          />
          <WidgetCard
            title="Approved"
            value={approved}
            icon={<CheckCircleOutlined />}
            color="blue"
            percentage={5}
            description="Approved evaluations"
          />
          <WidgetCard
            title="Pending"
            value={pending}
            icon={<MinusCircleOutlined />}
            color="gold"
            percentage={3}
            description="Pending evaluations"
          />
          <WidgetCard
            title="Advanced Level"
            value={advanced}
            icon={<CheckSquare />}
            color="orange"
            percentage={2}
            description="Advanced-level competencies"
          />
        </div>

        <div className="mt-4">
          <Card
            title="Competency Evaluations List"
            className="p-0 mb-6 shadow-sm"
          >
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search by employee name, code, or competency area"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Evaluation
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Competency Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        competencyLevel: value,
                      }))
                    }
                    allowClear
                    options={competencyLevelOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                  <Select
                    placeholder="Approved Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        approvedStatus: value,
                      }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="competency_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    icon={<FaTrash />}
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedEvaluations.map((evaluation) => (
                  <EvaluationCard
                    key={evaluation.evaluation_code}
                    evaluation={evaluation}
                  />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<CompetencyEvaluation>
                  columns={filteredColumns}
                  dataSource={paginatedEvaluations}
                  rowKey="evaluation_code"
                  scrollX="max-content"
                  className="mt-2"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filteredEvaluations.length)}
                </strong>{" "}
                of <strong>{filteredEvaluations.length}</strong> evaluations
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredEvaluations.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <div className="flex items-center gap-1">
                  <FileCheck
                    className={`w-5 h-5 ${
                      !editingEvaluation ? "text-green-600" : "text-gray-400"
                    }`}
                  />
                  <CheckSquare
                    className={`w-5 h-5 ${
                      editingEvaluation ? "text-blue-600" : "text-gray-400"
                    }`}
                  />
                </div>
                <span>
                  {editingEvaluation ? "Edit Evaluation" : "Add Evaluation"}
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
              setEditingEvaluation(null);
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" },
            }}
          >
            <Form form={form} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Evaluation Details
                  </h3>
                </div>

                {/* EDIT MODE: Show Evaluation Code */}
                {editingEvaluation && (
                  <Form.Item label="Evaluation Code">
                    <Input value={editingEvaluation.evaluation_code} disabled />
                  </Form.Item>
                )}

                {/* ADD MODE: Show Employee Name */}
                {!editingEvaluation && (
                  <Form.Item
                    name="employeeName"
                    label="Employee Name"
                    rules={[
                      { required: true, message: "Employee Name is required" },
                    ]}
                  >
                    <Select
                      showSearch
                      placeholder="Select an employee"
                      optionFilterProp="children"
                      filterOption={(input, option) =>
                        (option?.label?.toString() ?? "")
                          .toLowerCase()
                          .includes(input.toLowerCase())
                      }
                      options={employees.map((employee) => ({
                        label: `${employee.first_name} ${
                          employee.last_name || ""
                        }`.trim(),
                        value: `${employee.first_name} ${
                          employee.last_name || ""
                        }`.trim(),
                      }))}
                    />
                  </Form.Item>
                )}

                <Form.Item
                  name="evaluationDate"
                  label="Evaluation Date"
                  rules={[
                    { required: true, message: "Evaluation Date is required" },
                  ]}
                >
                  <DatePicker format="YYYY-MM-DD" className="w-full" />
                </Form.Item>

                <Form.Item
                  name="competencyArea"
                  label="Competency Area"
                  rules={[
                    { required: true, message: "Competency Area is required" },
                  ]}
                >
                  <Input />
                </Form.Item>

                <Form.Item name="evaluationMethod" label="Evaluation Method">
                  <Select
                    options={evaluationMethod.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>

                <Form.Item name="score" label="Score">
                  <Input type="number" min={0} max={100} />
                </Form.Item>

                <Form.Item name="competencyLevel" label="Competency Level">
                  <Select
                    options={competencyLevelOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>

                {/* RESULT ONLY IN ADD MODAL */}
                {!editingEvaluation && (
                  <Form.Item name="result" label="Result">
                    <Select
                      options={result.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>
                )}

                <Form.Item name="nextReviewDate" label="Next Review Date">
                  <DatePicker format="YYYY-MM-DD" className="w-full" />
                </Form.Item>

                <div className="col-span-full">
                  <Form.Item name="remarks" label="Remarks">
                    <Input.TextArea rows={2} />
                  </Form.Item>
                </div>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <motion.div
          key={forceRender}
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <CheckCircleOutlined className="w-5 h-5 text-purple-600" />
                <span>Edit Approval Status</span>
              </div>
            }
            open={isApprovalModalOpen}
            onOk={handleApprovalSubmit}
            onCancel={() => {
              setIsApprovalModalOpen(false);
              approvalForm.resetFields();
              setEditingApprovalEvaluation(null);
              setForceRender((prev) => prev + 1);
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" },
            }}
          >
            <Form form={approvalForm} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Approval Details
                  </h3>
                </div>
                <Form.Item
                  name="approvedStatus"
                  label="Approval Status"
                  rules={[
                    { required: true, message: "Approval Status is required" },
                  ]}
                >
                  <Select
                    options={statusOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                    onChange={(value) => {
                      const currentComments =
                        approvalForm.getFieldValue("comments") ||
                        editingApprovalEvaluation?.comments ||
                        undefined;
                      const currentRejectionReason =
                        approvalForm.getFieldValue("rejectionReason") ||
                        editingApprovalEvaluation?.rejection_reason ||
                        undefined;
                      const currentResult =
                        approvalForm.getFieldValue("result") ||
                        editingApprovalEvaluation?.result ||
                        undefined;
                      const fields = {
                        approvedStatus: value,
                        comments: currentComments,
                        rejectionReason:
                          value === "Rejected"
                            ? currentRejectionReason
                            : undefined,
                        approvedBy:
                          value === "Approved"
                            ? editingApprovalEvaluation?.approved_by_name ||
                              "Shankar.S"
                            : undefined,
                        approvedDate:
                          value === "Approved"
                            ? editingApprovalEvaluation?.approved_date
                              ? dayjs(editingApprovalEvaluation.approved_date)
                              : dayjs()
                            : undefined,
                        result: currentResult,
                      };
                      approvalForm.setFieldsValue(fields);
                      setForceRender((prev) => prev + 1);
                    }}
                  />
                </Form.Item>
                <Form.Item
                  name="result"
                  label="Result"
                  rules={[{ required: true, message: "Result is required" }]}
                >
                  <Select
                    options={result.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>
                <Form.Item name="comments" label="Comments">
                  <Input.TextArea rows={2} />
                </Form.Item>
                {approvalForm.getFieldValue("approvedStatus") ===
                  "Rejected" && (
                  <Form.Item
                    name="rejectionReason"
                    label="Rejection Reason"
                    rules={[
                      {
                        required: true,
                        message: "Rejection Reason is required",
                      },
                    ]}
                  >
                    <Input.TextArea rows={2} />
                  </Form.Item>
                )}
                {approvalForm.getFieldValue("approvedStatus") ===
                  "Approved" && (
                  <>
                    <Form.Item
                      name="approvedBy"
                      label="Approved By"
                      rules={[
                        { required: true, message: "Approved By is required" },
                      ]}
                    >
                      <Select
                        showSearch
                        placeholder="Select an approver"
                        optionFilterProp="children"
                        filterOption={(input, option) =>
                          (option?.label?.toString() ?? "")
                            .toLowerCase()
                            .includes(input.toLowerCase())
                        }
                        options={users.map((user) => ({
                          label: `${user.First_name} ${
                            user.Last_name || ""
                          }`.trim(),
                          value: `${user.First_name} ${
                            user.Last_name || ""
                          }`.trim(),
                        }))}
                      />
                    </Form.Item>
                    <Form.Item
                      name="approvedDate"
                      label="Approved Date"
                      rules={[
                        {
                          required: true,
                          message: "Approved Date is required",
                        },
                      ]}
                    >
                      <DatePicker format="YYYY-MM-DD" className="w-full" />
                    </Form.Item>
                  </>
                )}
              </div>
            </Form>
          </Modal>
        </motion.div>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
