"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  EyeOutlined,
  SearchOutlined,
  FileTextOutlined,
  ClockCircleOutlined,
  FileDoneOutlined,
  DownloadOutlined,
  UserOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import { MdGeneratingTokens } from "react-icons/md";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useRouter } from "next/navigation";
import {
  createInductionSchedule,
  deleteInductionSchedule,
  getAllInductionSchedules,
  InductionScheduleListItem,
  updateInductionSchedule,
  downloadInductionFile,
  updateSignatures,
} from "@/app/api/induction/induction";
import { getAllEmployees, Employee } from "@/app/api/employee/employee";
import {
  getAllManageUsers,
  ManageUser,
} from "@/app/api/manageusers/manageusers";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../../components/common/GlassCard";
import { highlightMatch } from "../../components/common/HighLightMatch";
import { FaFileExcel, FaFileExport, FaFilePdf, FaTrash } from "react-icons/fa";
import { Calendar, FileCheck } from "lucide-react";
import { LiaUserEditSolid } from "react-icons/lia";

const WidgetCard = dynamic(() => import("../../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});

const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);

const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

const { RangePicker, TimePicker } = DatePicker;

export default function InductionSchedulePage() {
  const [form] = Form.useForm();
  const [approvalForm] = Form.useForm();
  const [schedules, setSchedules] = useState<InductionScheduleListItem[]>([]);
  const [employees, setEmployees] = useState<Employee[]>([]);
  const [users, setUsers] = useState<ManageUser[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [isApprovalModalOpen, setIsApprovalModalOpen] = useState(false);
  const [editingSchedule, setEditingSchedule] =
    useState<InductionScheduleListItem | null>(null);
  const [approvingSchedule, setApprovingSchedule] =
    useState<InductionScheduleListItem | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });
  const [loading, setLoading] = useState(false);
  const [employeesLoading, setEmployeesLoading] = useState(false);
  const [usersLoading, setUsersLoading] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [hodSig, setHodSig] = useState<File | null>(null);
  const isMobile = useIsMobile();
  const router = useRouter();

  const [traineeSig, setTraineeSig] = useState<File | null>(null);
  const [trainerSig, setTrainerSig] = useState<File | null>(null);
  const [hrSig, setHrSig] = useState<File | null>(null);

  useEffect(() => {
    fetchSchedules();
    fetchEmployees();
    fetchUsers();
  }, []);

  const fetchSchedules = async () => {
    try {
      setLoading(true);
      const response = await getAllInductionSchedules();
      const sorted = response.data
        .map((item: any) => ({
          ...item,
          // FINAL FIX: Convert null/empty to real empty string so it never shows "?"
          from_time: item.from_time ? String(item.from_time).trim() : "",
          to_time: item.to_time ? String(item.to_time).trim() : "",
        }))
        .sort(
          (a: any, b: any) =>
            new Date(b.created_at || "").getTime() -
            new Date(a.created_at || "").getTime()
        );
      setSchedules(sorted);
    } catch (error) {
      console.error("Failed to fetch schedules:", error);
      toast.error("Failed to fetch induction schedules");
    } finally {
      setLoading(false);
    }
  };

  const fetchEmployees = async () => {
    try {
      setEmployeesLoading(true);
      const employeesData = await getAllEmployees();
      setEmployees(employeesData);
    } catch (error) {
      console.error("Failed to fetch employees:", error);
      toast.error("Failed to fetch employees");
    } finally {
      setEmployeesLoading(false);
    }
  };

  const fetchUsers = async () => {
    try {
      setUsersLoading(true);
      const userData = await getAllManageUsers();
      setUsers(userData);
    } catch (error) {
      console.error("Failed to fetch users:", error);
      toast.error("Failed to fetch trainers list");
    } finally {
      setUsersLoading(false);
    }
  };

  const openAddModal = () => {
    setEditingSchedule(null);
    form.resetFields();
    setTraineeSig(null);
    setTrainerSig(null);
    setHrSig(null);
    setIsModalOpen(true);
  };

  const openEditModal = (schedule: InductionScheduleListItem) => {
    setEditingSchedule(schedule);
    form.setFieldsValue({
      employee_code: schedule.employee_code,
      trainer_name: schedule.trainer_id,
      date: dayjs(schedule.date),
      duration: schedule.duration,
      scope: schedule.scope,
      topics_covered: schedule.topics_covered,
      // FIXED: Always parse as strict "HH:mm"
      from_time: schedule.from_time
        ? dayjs(schedule.from_time, "HH:mm", true).isValid()
          ? dayjs(schedule.from_time, "HH:mm")
          : null
        : null,
      to_time: schedule.to_time
        ? dayjs(schedule.to_time, "HH:mm", true).isValid()
          ? dayjs(schedule.to_time, "HH:mm")
          : null
        : null,
      hr_signed_date: schedule.hr_signed_date
        ? dayjs(schedule.hr_signed_date)
        : undefined,
    });
    setIsModalOpen(true);
  };

  const openApprovalModal = (schedule: InductionScheduleListItem) => {
    setApprovingSchedule(schedule);
    approvalForm.setFieldsValue({
      hod_remarks: schedule.hod_remarks || "",
      hod_signed_date: schedule.hod_signed_date
        ? dayjs(schedule.hod_signed_date)
        : undefined,
      evaluation_understanding_org:
        schedule.evaluation_understanding_org || false,
      evaluation_understanding_qms:
        schedule.evaluation_understanding_qms || false,
      evaluation_job_requirements:
        schedule.evaluation_job_requirements || false,
      evaluation_internal_customers:
        schedule.evaluation_internal_customers || false,
    });
    setHodSig(null);
    setIsApprovalModalOpen(true);
  };

  const handleDelete = (code: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Schedule",
      message: "Are you sure you want to delete this induction schedule?",
      onConfirm: async () => {
        try {
          await deleteInductionSchedule(code);
          await fetchSchedules();
          toast.success("Schedule deleted successfully");
        } catch {
          toast.error("Failed to delete schedule");
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Schedules",
      message: `Are you sure you want to delete ${selectedRows.length} schedules?`,
      onConfirm: async () => {
        try {
          await Promise.all(
            selectedRows.map((id) => deleteInductionSchedule(id))
          );
          await fetchSchedules();
          setSelectedRows([]);
          toast.success("Schedules deleted successfully");
        } catch {
          toast.error("Failed to delete selected schedules");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();

      if (!editingSchedule && (!traineeSig || !trainerSig || !hrSig)) {
        toast.error("Trainee, Trainer and HR signatures are required");
        return;
      }

      const payload: any = {
        employee_code: values.employee_code,
        trainer_id: values.trainer_name,
        date: values.date.format("YYYY-MM-DD"),
        duration: values.duration,
        scope: values.scope,
        topics_covered: values.topics_covered,
        hr_signed_date: values.hr_signed_date?.format("YYYY-MM-DD") || "",
        from_time: values.from_time ? values.from_time.format("HH:mm") : "",
        to_time: values.to_time ? values.to_time.format("HH:mm") : "",
      };

      if (editingSchedule) {
        if (traineeSig) payload.trainee_signature = traineeSig;
        if (trainerSig) payload.trainer_signature = trainerSig;
        if (hrSig) payload.hr_signature = hrSig;
        await updateInductionSchedule(editingSchedule.schedule_code, payload);
        toast.success("Schedule updated successfully");
      } else {
        payload.trainee_signature = traineeSig!;
        payload.trainer_signature = trainerSig!;
        payload.hr_signature = hrSig!;
        await createInductionSchedule(payload);
        toast.success("Schedule added successfully");
      }

      await fetchSchedules();
      setIsModalOpen(false);
      form.resetFields();
      setTraineeSig(null);
      setTrainerSig(null);
      setHrSig(null);
    } catch (err: any) {
      if (err.errorFields && err.errorFields.length > 0) {
        const missingFields = err.errorFields.map((field: any) => {
          const fieldName = field.name[0];
          const labelMap: Record<string, string> = {
            employee_code: "Employee",
            trainer_name: "Trainer Name",
            date: "Date",
            duration: "Duration (hrs)",
            scope: "Scope",
            topics_covered: "Topics Covered",
          };
          return labelMap[fieldName] || fieldName;
        });

        const message =
          missingFields.length > 1
            ? `${missingFields
                .slice(0, -1)
                .join(", ")} and ${missingFields.slice(-1)} are required`
            : `${missingFields[0]} is required`;

        toast.error(message);
        return;
      }

      toast.error("Failed to save schedule");
    }
  };

  // FIXED: Now properly updates instead of creating new record
  const handleApprovalSubmit = async () => {
    if (!approvingSchedule) return;

    try {
      const values = await approvalForm.validateFields();
      const payload: any = {
        hod_remarks: values.hod_remarks || "",
        hod_signed_date: values.hod_signed_date?.format("YYYY-MM-DD") || null,
        evaluation_understanding_org:
          values.evaluation_understanding_org ?? null,
        evaluation_understanding_qms:
          values.evaluation_understanding_qms ?? null,
        evaluation_job_requirements: values.evaluation_job_requirements ?? null,
        evaluation_internal_customers:
          values.evaluation_internal_customers ?? null,
      };

      if (hodSig) payload.hod_signature = hodSig;

      await updateSignatures(approvingSchedule.schedule_code, payload);
      toast.success("Approval updated successfully");
      await fetchSchedules();
      setIsApprovalModalOpen(false);
      approvalForm.resetFields();
      setHodSig(null);
      setApprovingSchedule(null); // Critical fix
    } catch (err) {
      toast.error("Failed to update approval");
    }
  };

  const filteredSchedules = schedules.filter((s) => {
    const matchesSearch =
      s.schedule_code.toLowerCase().includes(searchTerm.toLowerCase()) ||
      s.employee_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      s.employee_code.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      ((dayjs(s.date).isSame(dateRange[0]) ||
        dayjs(s.date).isAfter(dateRange[0])) &&
        (dayjs(s.date).isSame(dateRange[1]) ||
          dayjs(s.date).isBefore(dateRange[1])));
    return matchesSearch && matchesDate;
  });

  const paginatedSchedules = filteredSchedules.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<InductionScheduleListItem> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked ? filteredSchedules.map((s) => s.schedule_code) : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.schedule_code)}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.schedule_code]
                : prev.filter((id) => id !== record.schedule_code)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Schedule Code",
      dataIndex: "schedule_code",
      key: "schedule_code",
      sorter: (a, b) => a.schedule_code.localeCompare(b.schedule_code),
      render: (text) => highlightMatch(text, searchTerm),
    },
    {
      title: "Employee",
      key: "employee",
      render: (_, record) => (
        <div>
          <p className="font-medium">
            {highlightMatch(record.employee_name, searchTerm)}
          </p>
          {/* <p className="text-xs text-gray-500">{record.employee_code}</p> */}
        </div>
      ),
    },
    {
      title: "Designation",
      dataIndex: "designation_emp",
      key: "designation_emp",
    },
    {
      title: "Department",
      dataIndex: "department_emp",
      key: "department_emp",
    },
    {
      title: "Trainer",
      dataIndex: "trainer_name",
      key: "trainer_name",
    },
    {
      title: "Date",
      dataIndex: "date",
      key: "date",
      render: (date) => dayjs(date).format("DD-MM-YYYY"),
      sorter: (a, b) => dayjs(a.date).unix() - dayjs(b.date).unix(),
    },
    {
      title: "Time",
      key: "time",
      render: (_, record) => (
        <span>
          {record.from_time || "-"} - {record.to_time || "-"}
        </span>
      ),
    },
    {
      title: "Duration",
      dataIndex: "duration",
      key: "duration",
    },
    {
      title: "Scope",
      dataIndex: "scope",
      key: "scope",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[120px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Trainee Sign",
      key: "trainee_signature",
      render: (_, record) =>
        record.trainee_signature ? (
          <Button
            size="small"
            icon={<DownloadOutlined />}
            onClick={() =>
              downloadInductionFile(record.schedule_code, "trainee_sign")
            }
          >
            Download
          </Button>
        ) : (
          <Tag color="red">Pending</Tag>
        ),
    },
    {
      title: "Trainer Sign",
      key: "trainer_signature",
      render: (_, record) =>
        record.trainer_signature ? (
          <Button
            size="small"
            icon={<DownloadOutlined />}
            onClick={() =>
              downloadInductionFile(record.schedule_code, "trainer_sign")
            }
          >
            Download
          </Button>
        ) : (
          <Tag color="orange">Pending</Tag>
        ),
    },
    {
      title: "HR Sign",
      key: "hr_signature",
      render: (_, record) =>
        record.hr_signature ? (
          <Button
            size="small"
            icon={<DownloadOutlined />}
            onClick={() =>
              downloadInductionFile(record.schedule_code, "hr_sign")
            }
          >
            Download
          </Button>
        ) : (
          <Tag color="gold">Pending</Tag>
        ),
    },
    {
      title: "HOD Sign",
      key: "hod_signature",
      render: (_, record) =>
        record.hod_signature ? (
          <Tag
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="green"
          >
            Signed
          </Tag>
        ) : (
          <Tag color="default">Pending</Tag>
        ),
    },
    {
      title: "HOD Remarks",
      dataIndex: "hod_remarks",
      key: "hod_remarks",
      render: (text) => (text ? text : "-"),
    },
    {
      title: "HOD Signed Date",
      dataIndex: "hod_signed_date",
      key: "hod_signed_date",
      render: (text) => (text ? dayjs(text).format("DD-MM-YYYY") : "-"),
    },
    {
      title: "Eval: Org Understanding",
      dataIndex: "evaluation_understanding_org",
      key: "evaluation_understanding_org",
      render: (val) => (val ? "Yes" : "No"),
    },
    {
      title: "Eval: QMS Understanding",
      dataIndex: "evaluation_understanding_qms",
      key: "evaluation_understanding_qms",
      render: (val) => (val ? "Yes" : "No"),
    },
    {
      title: "Eval: Job Req.",
      dataIndex: "evaluation_job_requirements",
      key: "evaluation_job_requirements",
      render: (val) => (val ? "Yes" : "No"),
    },
    {
      title: "Eval: Internal Cust.",
      dataIndex: "evaluation_internal_customers",
      key: "evaluation_internal_customers",
      render: (val) => (val ? "Yes" : "No"),
    },
    {
      title: "Created By",
      dataIndex: "created_by_name",
      key: "created_by_name",
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      render: (value) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Actions",
      key: "actions",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Edit Approval">
            <Button
              icon={<LiaUserEditSolid />}
              onClick={() => openApprovalModal(record)}
              size="small"
              className="text-purple-600"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.schedule_code)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "schedule_code", label: "Schedule Code" },
    { key: "employee", label: "Employee" },
    { key: "designation_emp", label: "Designation" },
    { key: "department_emp", label: "Department" },
    { key: "trainer_name", label: "Trainer" },
    { key: "date", label: "Date" },
    { key: "time", label: "Time" },
    { key: "duration", label: "Duration" },
    { key: "scope", label: "Scope" },
    { key: "trainee_signature", label: "Trainee Sign" },
    { key: "trainer_signature", label: "Trainer Sign" },
    { key: "hr_signature", label: "HR Sign" },
    { key: "hod_signature", label: "HOD Sign" },
    { key: "hod_remarks", label: "HOD Remarks" },
    { key: "hod_signed_date", label: "HOD Signed Date" },
    { key: "evaluation_understanding_org", label: "Eval: Org Understanding" },
    { key: "evaluation_understanding_qms", label: "Eval: QMS Understanding" },
    { key: "evaluation_job_requirements", label: "Eval: Job Req." },
    { key: "evaluation_internal_customers", label: "Eval: Internal Cust." },
    { key: "created_by_name", label: "Created By" },
    { key: "created_at", label: "Created At" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const ScheduleCard = ({
    schedule,
  }: {
    schedule: InductionScheduleListItem;
  }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="primary">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full space-y-2">
        <div className="flex justify-between items-start mb-2">
          <h3 className="text-lg font-semibold text-gray-800 dark:text-white">
            {schedule.employee_name}
          </h3>
          <Tag color="blue">{schedule.schedule_code}</Tag>
        </div>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Code:</strong> {schedule.employee_code}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Dept:</strong> {schedule.department_emp}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Trainer:</strong> {schedule.trainer_name}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Date:</strong> {dayjs(schedule.date).format("DD MMM YYYY")}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Time:</strong> {schedule.from_time || "-"} -
          {schedule.to_time || "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Duration:</strong> {schedule.duration}
        </p>
        <div className="flex gap-1 flex-wrap">
          {schedule.trainee_signature ? (
            <Tag color="green">Trainee Signed</Tag>
          ) : (
            <Tag color="red">Trainee</Tag>
          )}
          {schedule.trainer_signature ? (
            <Tag color="green">Trainer Signed</Tag>
          ) : (
            <Tag color="orange">Trainer</Tag>
          )}
          {schedule.hr_signature ? (
            <Tag color="green">HR Signed</Tag>
          ) : (
            <Tag color="gold">HR</Tag>
          )}
          {schedule.hod_signature ? (
            <Tag color="green">HOD Signed</Tag>
          ) : (
            <Tag color="default">HOD</Tag>
          )}
        </div>
        <div className="flex gap-2 pt-2 border-t dark:border-gray-700">
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(schedule)}
              size="small"
              type="primary"
              className="bg-blue-600"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(schedule.schedule_code)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
          <Tooltip title="Edit Approval">
            <Button
              icon={<LiaUserEditSolid />}
              onClick={() => openApprovalModal(schedule)}
              size="small"
              className="text-purple-600"
            />
          </Tooltip>
        </div>
      </div>
    </GlassCard>
  );

  const total = schedules.length;
  const signedCount = schedules.filter(
    (s) => s.trainee_signature && s.trainer_signature && s.hr_signature
  ).length;
  const pendingCount = schedules.filter(
    (s) => !s.trainee_signature || !s.trainer_signature || !s.hr_signature
  ).length;
  const todayCount = schedules.filter((s) =>
    dayjs().isSame(dayjs(s.date), "day")
  ).length;

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <MdGeneratingTokens className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Induction Schedules</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({schedules.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage employee induction training schedules and signatures.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm",
            },
            { title: "HR" },
            { title: "Induction Schedules" },
          ]}
        />
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Schedules"
            value={total}
            icon={<Calendar />}
            color="blue"
            percentage={12}
            description="All inductions"
          />
          <WidgetCard
            title="Fully Signed"
            value={signedCount}
            icon={<FileDoneOutlined />}
            color="green"
            percentage={8}
            description="Completed"
          />
          <WidgetCard
            title="Pending Sign"
            value={pendingCount}
            icon={<ClockCircleOutlined />}
            color="orange"
            percentage={-3}
            description="Awaiting"
          />
          <WidgetCard
            title="Today"
            value={todayCount}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={5}
            description="Scheduled today"
          />
        </div>

        <div className="mt-4">
          <Card title="Induction Schedule List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search by code, name, or employee"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  menu={{
                    items: [
                      {
                        key: "pdf",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFilePdf className="text-red-500" /> Export PDF
                          </span>
                        ),
                      },
                      {
                        key: "excel",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFileExcel className="text-green-500" /> Export
                            Excel
                          </span>
                        ),
                      },
                    ],
                    onClick: ({ key }) =>
                      toast.info(`Export ${key.toUpperCase()} coming soon!`),
                  }}
                >
                  <Button type="default" icon={<FaFileExport />}>
                    Export
                  </Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Schedule
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 w-full lg:w-auto">
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) => setDateRange(dates as any)}
                    className="w-full border-gray-300 dark:border-gray-600"
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="induction_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    icon={<FaTrash />}
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedSchedules.map((schedule) => (
                  <ScheduleCard
                    key={schedule.schedule_code}
                    schedule={schedule}
                  />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<InductionScheduleListItem>
                  columns={filteredColumns}
                  dataSource={paginatedSchedules}
                  rowKey="schedule_code"
                  scrollX="max-content"
                  className="mt-2"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filteredSchedules.length)}
                </strong>{" "}
                of <strong>{filteredSchedules.length}</strong> schedules
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredSchedules.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <FileCheck
                  className={`w-5 h-5 ${
                    !editingSchedule ? "text-green-600" : "text-gray-400"
                  }`}
                />
                <UserOutlined
                  className={`w-5 h-5 ${
                    editingSchedule ? "text-blue-600" : "text-gray-400"
                  }`}
                />
                <span>
                  {editingSchedule ? "Edit Schedule" : "Add Schedule"}
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
              setTraineeSig(null);
              setTrainerSig(null);
              setHrSig(null);
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{ body: { maxHeight: "80vh", overflowY: "auto" } }}
          >
            <Form form={form} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Schedule Details
                  </h3>
                </div>

                <Form.Item
                  name="employee_code"
                  label="Employee"
                  rules={[{ required: true, message: "Employee is required" }]}
                >
                  <Select
                    showSearch
                    placeholder="Select employee"
                    loading={employeesLoading}
                    allowClear
                  >
                    {employees.map((emp) => (
                      <Select.Option
                        key={emp.employee_code}
                        value={emp.employee_code}
                      >
                        {emp.first_name} {emp.last_name || ""}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item
                  name="trainer_name"
                  label="Trainer Name"
                  rules={[{ required: true, message: "Trainer is required" }]}
                >
                  <Select
                    showSearch
                    placeholder="Select trainer"
                    loading={usersLoading}
                    optionFilterProp="children"
                    allowClear
                  >
                    {users.map((user) => (
                      <Select.Option
                        key={user.id || user.email}
                        value={user.id || user.email}
                      >
                        {user.First_name} {user.Last_name || ""}
                      </Select.Option>
                    ))}
                  </Select>
                </Form.Item>

                <Form.Item
                  name="date"
                  label="Date"
                  rules={[{ required: true, message: "Date is required" }]}
                >
                  <DatePicker className="w-full" />
                </Form.Item>

                <Form.Item
                  name="duration"
                  label="Duration (hrs)"
                  rules={[{ required: true, message: "Duration is required" }]}
                >
                  <Input />
                </Form.Item>

                <Form.Item name="from_time" label="From Time">
                  <TimePicker format="HH:mm" className="w-full" />
                </Form.Item>

                <Form.Item name="to_time" label="To Time">
                  <TimePicker format="HH:mm" className="w-full" />
                </Form.Item>

                <Form.Item
                  name="scope"
                  label="Scope"
                  rules={[{ required: true, message: "Scope is required" }]}
                >
                  <Input.TextArea rows={2} />
                </Form.Item>

                <Form.Item
                  name="topics_covered"
                  label="Topics Covered"
                  rules={[
                    { required: true, message: "Topics Covered is required" },
                  ]}
                  className="col-span-full"
                >
                  <Input.TextArea rows={3} />
                </Form.Item>

                {!editingSchedule && (
                  <>
                    <div className="col-span-full">
                      <h3 className="text-lg font-semibold mb-2">
                        Signatures (Required)
                      </h3>
                    </div>
                    <Form.Item
                      label="Trainee Signature"
                      className="col-span-full"
                    >
                      <Input
                        type="file"
                        onChange={(e) =>
                          setTraineeSig(e.target.files?.[0] || null)
                        }
                      />
                    </Form.Item>
                    <Form.Item
                      label="Trainer Signature"
                      className="col-span-full"
                    >
                      <Input
                        type="file"
                        onChange={(e) =>
                          setTrainerSig(e.target.files?.[0] || null)
                        }
                      />
                    </Form.Item>
                    <Form.Item label="HR Signature" className="col-span-full">
                      <Input
                        type="file"
                        onChange={(e) => setHrSig(e.target.files?.[0] || null)}
                      />
                    </Form.Item>
                    <Form.Item name="hr_signed_date" label="HR Signed Date">
                      <DatePicker className="w-full" />
                    </Form.Item>
                  </>
                )}

                {editingSchedule && (
                  <>
                    <div className="col-span-full">
                      <h3 className="text-lg font-semibold mb-2">
                        Update Signatures
                      </h3>
                    </div>
                    <Form.Item label="Update Trainee Signature">
                      <Input
                        type="file"
                        onChange={(e) =>
                          setTraineeSig(e.target.files?.[0] || null)
                        }
                      />
                      {editingSchedule.trainee_signature && (
                        <p className="text-sm text-gray-500 mt-1">
                          Existing:{" "}
                          {editingSchedule.trainee_signature.split("/").pop()}
                        </p>
                      )}
                    </Form.Item>
                    <Form.Item label="Update Trainer Signature">
                      <Input
                        type="file"
                        onChange={(e) =>
                          setTrainerSig(e.target.files?.[0] || null)
                        }
                      />
                      {editingSchedule.trainer_signature && (
                        <p className="text-sm text-gray-500 mt-1">
                          Existing:{" "}
                          {editingSchedule.trainer_signature.split("/").pop()}
                        </p>
                      )}
                    </Form.Item>
                    <Form.Item label="Update HR Signature">
                      <Input
                        type="file"
                        onChange={(e) => setHrSig(e.target.files?.[0] || null)}
                      />
                      {editingSchedule.hr_signature && (
                        <p className="text-sm text-gray-500 mt-1">
                          Existing:{" "}
                          {editingSchedule.hr_signature.split("/").pop()}
                        </p>
                      )}
                    </Form.Item>
                    <Form.Item
                      name="hr_signed_date"
                      label="Update HR Signed Date"
                    >
                      <DatePicker className="w-full" />
                    </Form.Item>
                  </>
                )}
              </div>
            </Form>
          </Modal>

          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <IdcardOutlined className="w-5 h-5 text-purple-600" />
                <span>Edit Approval</span>
              </div>
            }
            open={isApprovalModalOpen}
            onOk={handleApprovalSubmit}
            onCancel={() => {
              setIsApprovalModalOpen(false);
              approvalForm.resetFields();
              setHodSig(null);
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{ body: { maxHeight: "80vh", overflowY: "auto" } }}
          >
            <Form form={approvalForm} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Approval Details
                  </h3>
                </div>
                <Form.Item name="hod_remarks" label="HOD Remarks">
                  <Input.TextArea rows={3} />
                </Form.Item>
                <Form.Item name="hod_signed_date" label="HOD Signed Date">
                  <DatePicker className="w-full" />
                </Form.Item>
                <Form.Item name="hod_signature" label="HOD Signature">
                  <Input
                    type="file"
                    onChange={(e) => setHodSig(e.target.files?.[0] || null)}
                  />
                </Form.Item>
                <Form.Item
                  name="evaluation_understanding_org"
                  label="Understands Organization"
                >
                  <Select placeholder="Select">
                    <Select.Option value={true}>Yes</Select.Option>
                    <Select.Option value={false}>No</Select.Option>
                  </Select>
                </Form.Item>
                <Form.Item
                  name="evaluation_understanding_qms"
                  label="Understands QMS"
                >
                  <Select placeholder="Select">
                    <Select.Option value={true}>Yes</Select.Option>
                    <Select.Option value={false}>No</Select.Option>
                  </Select>
                </Form.Item>
                <Form.Item
                  name="evaluation_job_requirements"
                  label="Understands Job Requirements"
                >
                  <Select placeholder="Select">
                    <Select.Option value={true}>Yes</Select.Option>
                    <Select.Option value={false}>No</Select.Option>
                  </Select>
                </Form.Item>
                <Form.Item
                  name="evaluation_internal_customers"
                  label="Understands Internal Customers"
                >
                  <Select placeholder="Select">
                    <Select.Option value={true}>Yes</Select.Option>
                    <Select.Option value={false}>No</Select.Option>
                  </Select>
                </Form.Item>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
