"use client";

import React, { useState, useEffect } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import Tabs from "antd/es/tabs";
import "antd/es/tabs/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Descriptions from "antd/es/descriptions";
import "antd/es/descriptions/style";
import Empty from "antd/es/empty";
import "antd/es/empty/style";
import Table from "antd/es/table";
import "antd/es/table/style";
import Collapse from "antd/es/collapse";
import "antd/es/collapse/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import {
  HomeOutlined,
  EyeOutlined,
  EditOutlined,
  DeleteOutlined,
  FileTextOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import { useSearchParams, useRouter } from "next/navigation";
import { motion } from "framer-motion";
import dayjs from "@/data/utils/dayjs";
import { ColumnsType } from "antd/es/table";
import { ArrowBigLeft } from "lucide-react";
import { Employee, getAllEmployees } from "@/app/api/employee/employee";

// Updated types for assessments and certifications
interface EmployeeAssessment {
  id: string;
  assessment_date: string;
  assessment_type: string;
  assessor: string;
  score: number;
  status: string;
  created_at: string;
  created_by_name: string;
}

interface EmployeeCertification {
  id: string;
  employee_id: string;
  certification_type: string;
  certification_number: string;
  issue_date: string;
  expiry_date: string;
  status: string;
  created_at: string;
  created_by_name: string;
}

// Placeholder API functions (replace with your actual implementations)
const getEmployeeAssessmentsByEmployeeId = async (
  employeeId: string
): Promise<EmployeeAssessment[]> => {
  // Simulate fetching assessments
  return [];
};

const getAllEmployeeCertifications = async (): Promise<
  EmployeeCertification[]
> => {
  // Simulate fetching certifications
  return [];
};

const EmployeeDetailPage = () => {
  const searchParams = useSearchParams();
  const router = useRouter();
  const employeeId = searchParams.get("id");
  const employeeName = searchParams.get("name") || "Unknown Employee";
  const [employee, setEmployee] = useState<Employee | null>(null);
  const [assessments, setAssessments] = useState<EmployeeAssessment[]>([]);
  const [certifications, setCertifications] = useState<EmployeeCertification[]>(
    []
  );
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (!employeeId) {
      router.push("/hrm/employeemaster");
      return;
    }

    const fetchData = async () => {
      try {
        const [employees, allCertifications, assessments] = await Promise.all([
          getAllEmployees(),
          getAllEmployeeCertifications(),
          getEmployeeAssessmentsByEmployeeId(employeeId),
        ]);

        const selectedEmployee = employees.find(
          (e: { employee_code: string; }) => e.employee_code === employeeId
        );
        if (!selectedEmployee) throw new Error("Employee not found");

        setEmployee({
          ...selectedEmployee,
          employee_code: selectedEmployee.employee_code ?? "N/A",
          first_name: selectedEmployee.first_name,
          last_name: selectedEmployee.last_name ?? "-",
          designation: selectedEmployee.designation ?? "-",
          department: selectedEmployee.department ?? "-",
          email: selectedEmployee.email ?? "-",
          phone: selectedEmployee.phone ?? "-",
          date_of_birth: selectedEmployee.date_of_birth ?? "-",
          date_of_joining: selectedEmployee.date_of_joining ?? "-",
          qualification: selectedEmployee.qualification ?? "-",
          experience_years: selectedEmployee.experience_years ?? 0,
          employee_type: selectedEmployee.employee_type ?? "-",
          employee_status: selectedEmployee.employee_status ?? "-",
          reporting_manager_id: selectedEmployee.reporting_manager_id ?? "-",
          reporting_manager_name:
            selectedEmployee.reporting_manager_name ?? "-",
          remarks: selectedEmployee.remarks ?? "-",
          created_at: selectedEmployee.created_at ?? "-",
          created_by_name: selectedEmployee.created_by_name ?? "Unknown",
          updated_by_name: selectedEmployee.updated_by_name ?? "-",
        });

        const employeeCertifications = allCertifications.filter(
          (c: { employee_id: string; }) => c.employee_id === employeeId
        );
        setCertifications(employeeCertifications);
        setAssessments(assessments);
      } catch (err) {
        console.error("Failed to load employee details:", err);
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, [employeeId, router]);

  const renderTag = (value: string | number) => {
    const colorMap: { [key: string]: string } = {
      Active: "success",
      Inactive: "warning",
      Terminated: "error",
      Permanent: "blue",
      Contract: "cyan",
      Intern: "purple",
      Trainee: "orange",
      Pass: "success",
      Fail: "error",
      Conditional: "warning",
      Valid: "success",
      Expired: "error",
      Pending: "warning",
    };

    const iconMap: { [key: string]: React.JSX.Element } = {
      Active: <EyeOutlined />,
      Inactive: <EditOutlined />,
      Terminated: <DeleteOutlined />,
      Permanent: <EyeOutlined />,
      Contract: <FileTextOutlined />,
      Intern: <EditOutlined />,
      Trainee: <FileTextOutlined />,
      Pass: <EyeOutlined />,
      Fail: <DeleteOutlined />,
      Conditional: <EditOutlined />,
      Valid: <EyeOutlined />,
      Expired: <DeleteOutlined />,
      Pending: <EditOutlined />,
    };

    const displayValue = String(value); // Convert to string for consistency

    return (
      <Tag
        icon={iconMap[displayValue] || <FileTextOutlined />}
        color={colorMap[displayValue] || "default"}
        style={{
          whiteSpace: "normal",
          wordBreak: "break-word",
          maxWidth: "28rem",
          display: "inline-block",
        }}
      >
        {displayValue}
      </Tag>
    );
  };

  const assessmentColumns: ColumnsType<EmployeeAssessment> = [
    {
      title: "Date",
      dataIndex: "assessment_date",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    { title: "Type", dataIndex: "assessment_type" },
    { title: "Assessor", dataIndex: "assessor" },
    {
      title: "Score (%)",
      dataIndex: "score",
      render: (text) => `${text || 0}%`,
    },
    { title: "Status", dataIndex: "status", render: renderTag },
    {
      title: "Created At",
      dataIndex: "created_at",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    { title: "Created By", dataIndex: "created_by_name" },
  ];

  const certificationColumns: ColumnsType<EmployeeCertification> = [
    { title: "Type", dataIndex: "certification_type" },
    { title: "Number", dataIndex: "certification_number" },
    {
      title: "Issue Date",
      dataIndex: "issue_date",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Expiry Date",
      dataIndex: "expiry_date",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    { title: "Status", dataIndex: "status", render: renderTag },
    {
      title: "Created At",
      dataIndex: "created_at",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    { title: "Created By", dataIndex: "created_by_name" },
  ];

  if (loading || !employee)
    return <div className="text-center py-10">Loading...</div>;

  return (
    <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-6 min-h-screen bg-gradient-to-br from-gray-100 to-white dark:from-gray-900 dark:to-gray-800">
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.6, ease: "easeOut" }}
      >
        <Button
          icon={<ArrowBigLeft className="text-green-600" />}
          onClick={() => router.push("/hrm/employeemaster")}
          className="mb-3 text-green-600 hover:text-green-700 border border-green-600 hover:border-green-700 bg-transparent shadow-none"
        >
          Back to Employees
        </Button>
        <h2 className="text-2xl sm:text-3xl md:text-4xl font-extrabold text-gray-900 dark:text-white mb-3 bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
          Employee Details - {employeeName}
        </h2>
        <p className="text-gray-600 border-b pb-4 dark:text-gray-400 text-base sm:text-lg mb-5">
          Comprehensive overview of {employeeName}.
        </p>
        <Breadcrumb
          className="mb-5 sm:mb-6 text-sm sm:text-base text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm/dashboard",
            },
            { title: "Employees", href: "/hrm/employeemaster" },
            { title: "Details" },
          ]}
        />
      </motion.div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 mt-3">
        {/* Summary Card */}
        <motion.div
          className="lg:col-span-1 flex flex-col"
          initial={{ scale: 0.95, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          transition={{ delay: 0.2, duration: 0.5 }}
        >
          <Card className="shadow-2xl bg-white dark:bg-gray-800 rounded-2xl p-6 border-0 transform hover:-translate-y-2 transition-all duration-300 flex-1">
            <h3 className="text-xl font-semibold text-gray-800 dark:text-white mb-4 border-b-2 border-blue-200 pb-2 flex justify-between items-center">
              Summary
              <div className="flex gap-2">
                {/* <Tooltip title="Assessment">
                  <Button
                    icon={<FileTextOutlined />}
                    onClick={() =>
                      router.push(
                        `/employees/employeeassessment?id=${employeeId}&name=${encodeURIComponent(
                          employeeName
                        )}`
                      )
                    }
                    size="small"
                    className="bg-purple-500 hover:bg-purple-600 text-white"
                  />
                </Tooltip>
                <Tooltip title="Certification">
                  <Button
                    icon={<IdcardOutlined />}
                    onClick={() =>
                      router.push(
                        `/employees/employeecertification?id=${employeeId}&name=${encodeURIComponent(
                          employeeName
                        )}`
                      )
                    }
                    size="small"
                    className="bg-teal-500 hover:bg-teal-600 text-white"
                  />
                </Tooltip> */}
              </div>
            </h3>
            <div className="space-y-4">
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Employee Code:
                </span>
                {renderTag(employee.employee_code ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">Name:</span>
                {renderTag(
                  `${employee.first_name} ${employee.last_name || ""}`
                )}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Designation:
                </span>
                {renderTag(employee.designation ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Department:
                </span>
                {renderTag(employee.department ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">Email:</span>
                {renderTag(employee.email ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">Phone:</span>
                {renderTag(employee.phone ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Date of Birth:
                </span>
                {renderTag(
                  employee.date_of_birth
                    ? dayjs(employee.date_of_birth).format("YYYY-MM-DD")
                    : "N/A"
                )}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Date of Joining:
                </span>
                {renderTag(
                  employee.date_of_joining
                    ? dayjs(employee.date_of_joining).format("YYYY-MM-DD")
                    : "N/A"
                )}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Qualification:
                </span>
                {renderTag(employee.qualification ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Experience:
                </span>
                {renderTag(
                  employee.experience_years
                    ? `${employee.experience_years} years`
                    : "N/A"
                )}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Employee Type:
                </span>
                {renderTag(employee.employee_type ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Status:
                </span>
                {renderTag(employee.employee_status ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Reporting Manager:
                </span>
                {renderTag(employee.reporting_manager_name ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Remarks:
                </span>
                {renderTag(employee.remarks ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-gray-300">
                  Created:
                </span>
                {renderTag(
                  employee.created_at
                    ? dayjs(employee.created_at).format("YYYY-MM-DD")
                    : "N/A"
                )}
              </div>
            </div>
          </Card>
        </motion.div>

        {/* Details and Tabs */}
        <motion.div
          className="lg:col-span-2 flex flex-col"
          initial={{ scale: 0.95, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          transition={{ delay: 0.3, duration: 0.5 }}
        >
          <Card className="shadow-2xl bg-white dark:bg-gray-800 rounded-2xl p-6 border-0 transform hover:-translate-y-2 transition-all duration-300 flex-1">
            <Tabs
              defaultActiveKey="1"
              className="mb-6"
              items={[
                {
                  key: "1",
                  label: "Overview",
                  children: (
                    <div className="space-y-4">
                      <Collapse
                        items={[
                          {
                            key: "1",
                            label: "Basic Information",
                            children: (
                              <Descriptions
                                column={1}
                                size="small"
                                className="text-sm text-gray-700 dark:text-gray-300"
                              >
                                <Descriptions.Item label="Name">
                                  {employee.first_name}{" "}
                                  {employee.last_name || ""}
                                </Descriptions.Item>
                                <Descriptions.Item label="Email">
                                  {employee.email ?? "-"}
                                </Descriptions.Item>
                                <Descriptions.Item label="Phone">
                                  {employee.phone ?? "-"}
                                </Descriptions.Item>
                                <Descriptions.Item label="Department">
                                  {employee.department ?? "-"}
                                </Descriptions.Item>
                                <Descriptions.Item label="Designation">
                                  {employee.designation ?? "-"}
                                </Descriptions.Item>
                                <Descriptions.Item label="Employee Type">
                                  {employee.employee_type ?? "-"}
                                </Descriptions.Item>
                                <Descriptions.Item label="Status">
                                  {employee.employee_status ?? "-"}
                                </Descriptions.Item>
                                <Descriptions.Item label="Created By">
                                  {employee.created_by_name ?? "-"}
                                </Descriptions.Item>
                              </Descriptions>
                            ),
                          },
                          {
                            key: "2",
                            label: "Additional Details",
                            children: (
                              <div className="text-sm text-gray-700 dark:text-gray-300">
                                <p>
                                  <strong>Date of Birth:</strong>{" "}
                                  {employee.date_of_birth
                                    ? dayjs(employee.date_of_birth).format(
                                        "YYYY-MM-DD"
                                      )
                                    : "-"}
                                </p>
                                <p>
                                  <strong>Date of Joining:</strong>{" "}
                                  {employee.date_of_joining
                                    ? dayjs(employee.date_of_joining).format(
                                        "YYYY-MM-DD"
                                      )
                                    : "-"}
                                </p>
                                <p>
                                  <strong>Qualification:</strong>{" "}
                                  {employee.qualification ?? "-"}
                                </p>
                                <p>
                                  <strong>Experience:</strong>{" "}
                                  {employee.experience_years
                                    ? `${employee.experience_years} years`
                                    : "-"}
                                </p>
                                <p>
                                  <strong>Reporting Manager:</strong>{" "}
                                  {employee.reporting_manager_name ?? "-"}
                                </p>
                                <p>
                                  <strong>Remarks:</strong>{" "}
                                  {employee.remarks ?? "-"}
                                </p>
                              </div>
                            ),
                          },
                        ]}
                      />
                    </div>
                  ),
                },
                {
                  key: "2",
                  label: "Assessments",
                  children:
                    assessments.length > 0 ? (
                      <Table
                        columns={assessmentColumns}
                        dataSource={assessments}
                        rowKey="id"
                        pagination={{ pageSize: 5 }}
                        scroll={{ x: "max-content" }}
                        className="mb-4 border border-gray-200 dark:border-gray-700 rounded-lg"
                      />
                    ) : (
                      <Empty
                        description="No assessments available"
                        className="py-10"
                      />
                    ),
                },
                {
                  key: "3",
                  label: "Certifications",
                  children:
                    certifications.length > 0 ? (
                      <Table
                        columns={certificationColumns}
                        dataSource={certifications}
                        rowKey="id"
                        pagination={{ pageSize: 5 }}
                        scroll={{ x: "max-content" }}
                        className="mb-4 border border-gray-200 dark:border-gray-700 rounded-lg"
                      />
                    ) : (
                      <Empty
                        description="No certifications available"
                        className="py-10"
                      />
                    ),
                },
              ]}
            />
          </Card>
        </motion.div>
      </div>
    </div>
  );
};

export default EmployeeDetailPage;
