'use client';

import React, { useEffect, useMemo, useState } from "react";
import Link from "next/link";
import QmsCard from "../components/QmsCard";
import { getAlldisqualifiedSuppliers, getAllSuppliers, Supplier } from "../api/suppliers/suppliers";
import {
  BarChart,
  Bar,
  XAxis,
  YAxis, PieChart, Pie, Cell, Legend, Tooltip, ResponsiveContainer
} from "recharts";
import { getAllProducts, Product } from "../api/products/product";
import { BarChart2, BarChart4, Info, Package, UserPlus } from "lucide-react";
import { LayoutDashboardIcon } from "lucide-react";
const COLORS = ["#22c55e", "#ef4444", "#facc15"];

export default function Dashboard() {
  const [data, setData] = useState<{ name: string; value: number }[]>([]);
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [products, setProducts] = useState<Product[]>([]);

  useEffect(() => {
    const fetchSuppliers = async () => {
      try {
        const allSuppliers: Supplier[] = await getAllSuppliers();
        setSuppliers(allSuppliers);

        const approved = allSuppliers.filter(
          (s) => s.approved_status === "Approved"
        ).length;

        const pending = allSuppliers.filter(
          (s) => s.approved_status === "Not_Choose"
        ).length;

        const disqualified: Supplier[] = await getAlldisqualifiedSuppliers();
        const disapproved = disqualified.length;

        setData([
          { name: "Approved", value: approved },
          { name: "Disapproved", value: disapproved },
          { name: "Pending", value: pending },
        ]);
      } catch (err) {
        console.error("Error fetching suppliers:", err);
      }
    };

    fetchSuppliers();
  }, []);

  useEffect(() => {
    const fetchProducts = async () => {
      try {
        const res = await getAllProducts();
        setProducts(res);
      } catch (err) {
        console.error("Error fetching products:", err);
      }
    };
    fetchProducts();
  }, []);

  // ✅ Compute chart data only when products change
  const chartData = useMemo(() => {
    if (!products.length) return [];

    const riskLevels = ["High", "Medium", "Low"];
    const statusSet = Array.from(new Set(products.map((p) => p.status)));

    return riskLevels.map((risk) => {
      const obj: Record<string, number | string> = { risk };
      statusSet.forEach((status) => {
        obj[status] = products.filter(
          (p) => p.risk_level === risk && p.status === status
        ).length;
      });
      return obj;
    });
  }, [products]);


  return (
    <div className="flex min-h-screen bg-gray-100">
      <main className="flex-1 w-full px-4 py-2 sm:px-5 sm:py-3">
       <h1 className="flex items-center gap-3 text-2xl sm:text-3xl font-bold text-gray-800 mb-2 sm:mb-3">
  <LayoutDashboardIcon className="w-7 h-7 text-green-600 flex-shrink-0" />
  <span>Welcome to Quality Management System</span>
</h1>

        <p className="text-gray-600 text-base sm:text-lg mb-4 sm:mb-6">
          This system is designed to comply with ISO 13485:2016 standards for medical devices.
        </p>

        <div className="grid gap-4 sm:gap-6 grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 animate-fade-in-up">
          <QmsCard
            title="Products"
            description="Manage medical device products and specifications"
            href="/products"
            count={42}
          />
          <QmsCard
            title="Quality Processes"
            description="View and manage SOPs, CAPA, Deviations"
            href="/quality-processes"
            count={18}
          />
          <QmsCard
            title="Audits"
            description="Track internal and external audit logs"
            href="/audits"
            count={9}
          />
          <QmsCard
            title="Documents"
            description="Access QMS-related documents and templates"
            href="/documents"
            count={76}
          />
        </div>
        {/* Charts Section for suppliers approved, disapproved,.... & products risklevel and status */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-4">
          <div className="rounded-2xl shadow-lg border border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-900 p-6 transition-colors duration-300">
            <h2 className="text-lg sm:text-xl font-bold text-gray-800 dark:text-gray-200 mb-4 flex items-center gap-2">
              <BarChart4 className="w-5 h-5 text-green-600 dark:text-green-400" />
              Supplier Status Distribution

              {/* Info icon with tooltip */}
              <div className="relative group">
                <Info className="w-4 h-4 text-gray-400 dark:text-gray-300 cursor-pointer" />
              <div className="absolute left-1/2 transform -translate-x-1/2 -top-8 bg-gray-800 text-white text-xs rounded py-1 px-2 opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50">
                  Shows Approved, Disapproved & Pending suppliers
                </div>
              </div>
            </h2>
            <div className="h-72 sm:h-80">
              <ResponsiveContainer width="100%" height="100%">
                <PieChart>
                  <Pie
                    data={data}
                    cx="50%"
                    cy="50%"
                    innerRadius="50%"
                    outerRadius="80%"
                    paddingAngle={5}
                    dataKey="value"
                    label={({ name, value }) => `${name} (${value})`}
                  >
                    {data.map((entry, index) => (
                      <Cell
                        key={`cell-${index}`}
                        fill={COLORS[index % COLORS.length]}
                        className="transition-all duration-300 hover:opacity-80"
                      />
                    ))}
                  </Pie>
                  <Tooltip
                    contentStyle={{
                      backgroundColor: "#1f2937", // dark gray
                      color: "#f9fafb",
                      borderRadius: "0.5rem",
                      border: "none",
                      boxShadow: "0 2px 10px rgba(0,0,0,0.15)",
                    }}
                    itemStyle={{ color: "#f9fafb" }}
                    labelStyle={{ color: "#f9fafb" }}
                  />
                </PieChart>
              </ResponsiveContainer>
            </div>

            {/* Legend */}
            <div className="flex justify-center gap-4 mt-4 flex-wrap">
              {data.map((item, index) => (
                <div key={item.name} className="flex items-center gap-2">
                  <span
                    className="w-3 h-3 rounded-full"
                    style={{ backgroundColor: COLORS[index] }}
                  />
                  <span className="text-sm text-gray-700 dark:text-gray-300">
                    {item.name}: {item.value}
                  </span>
                </div>
              ))}
            </div>
          </div>
          <div className="rounded-2xl shadow-lg border border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-900 p-6 transition-colors duration-300">
            <h2 className="text-lg sm:text-xl font-bold text-gray-800 dark:text-gray-200 mb-4 flex items-center gap-2">
              <BarChart2 className="w-5 h-5 text-green-600 dark:text-green-400" />
              Products by Risk Level & Status{" "}
              <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                ({chartData.length > 0 ? Object.keys(chartData[0]).filter((k) => k !== "risk").join(" vs ") : "No Status"})
              </span>
            </h2>
            <div className="h-72 sm:h-80">
              <ResponsiveContainer width="100%" height="100%">
                <BarChart
                  data={chartData}
                  margin={{ top: 10, right: 20, left: 0, bottom: 20 }}
                >
                  <XAxis dataKey="risk" stroke="#9ca3af" />
                  <YAxis stroke="#9ca3af" />
                  <Tooltip
                    contentStyle={{
                      backgroundColor: "#1f2937",
                      color: "#f9fafb",
                      borderRadius: "0.5rem",
                      border: "none",
                      boxShadow: "0 2px 10px rgba(0,0,0,0.15)",
                    }}
                  />
                  <Legend />
                  {chartData.length > 0 &&
                    Object.keys(chartData[0])
                      .filter((k) => k !== "risk")
                      .map((status, idx) => (
                        <Bar
                          key={status}
                          dataKey={status}
                          stackId="a"
                          fill={["#22c55e", "#ef4444", "#3b82f6", "#f59e0b"][idx % 4]}
                          radius={[4, 4, 0, 0]}
                        />
                      ))}
                </BarChart>
              </ResponsiveContainer>
            </div>
          </div>
        </div>

        {/* Recently Added Suppliers & Products Section */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mt-6">
          {/* Recently Added Suppliers */}
          <div className="rounded-2xl shadow-lg border border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-900 p-6 transition-colors duration-300">
            <div className="flex items-center mb-4 gap-2">
              <UserPlus className="w-6 h-6 text-green-600 dark:text-green-400" />
              <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200">
                Recently Added Suppliers
              </h3>

              {/* Info button with tooltip */}
              {suppliers.length > 0 && (
                <div className="relative group">
                  <Info className="w-4 h-4 text-gray-400 dark:text-gray-300 cursor-pointer" />
                  <div className="absolute left-1/2 transform -translate-x-1/2 -top-8 bg-gray-800 text-white text-xs rounded py-1 px-2 opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50">
                    This table shows the 5 most recently added suppliers.
                  </div>
                </div>
              )}
            </div>

            <ul className="divide-y divide-gray-200 dark:divide-gray-700 max-h-100">
              {suppliers.length > 0 ? (
                suppliers
                  .slice(0, 5)
                  .map((supplier) => (
                    <li
                      key={supplier.supplier_id}
                      className="py-2 flex justify-between items-center hover:bg-gray-50 dark:hover:bg-gray-800 transition-all rounded-md px-2 animate-fade-in-up"
                      title={`Risk: ${supplier.risk_classification || "N/A"} | Type: ${supplier.Suplier_type}`}
                    >
                      <div className="flex items-center gap-2">
                        {/* Status dot */}
                        <span
                          className="w-2 h-2 rounded-full"
                          style={{
                            backgroundColor:
                              supplier.approved_status === "Approved"
                                ? "#22c55e"
                                : supplier.approved_status === "Disapproved"
                                  ? "#ef4444"
                                  : "#facc15",
                          }}
                        />
                        <div className="flex flex-col">
                          <p className="text-sm font-medium text-gray-800 dark:text-gray-200">{supplier.company_name}</p>
                          <p className="text-xs text-gray-500 dark:text-gray-400">
                            {supplier.contact_person || "N/A"} | {supplier.country || "Unknown"}
                          </p>
                          <p className="text-xs text-gray-400 dark:text-gray-500">
                            Added: {supplier.created_at ? new Date(supplier.created_at).toLocaleDateString() : "Unknown"}
                          </p>
                        </div>
                      </div>

                      <span
                        className={`text-xs font-medium px-2 py-0.5 rounded-full ${supplier.status === "Active"
                          ? "bg-green-100 text-green-700 dark:bg-green-800 dark:text-green-300"
                          : "bg-red-100 text-red-700 dark:bg-red-800 dark:text-red-300"
                          }`}
                      >
                        {supplier.status}
                      </span>
                    </li>
                  ))
              ) : (
                <p className="text-gray-500 dark:text-gray-400 text-sm">No suppliers found.</p>
              )}
            </ul>

            <a
              href="/suppliers"
              className="text-sm text-blue-600 dark:text-blue-400 hover:underline mt-2 block"
            >
              View All
            </a>
          </div>

          {/* Recently Added Products */}
          <div className="rounded-2xl shadow-lg border border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-900 p-6 transition-colors duration-300">
            <div className="flex items-center mb-4 gap-2">
              <Package className="w-6 h-6 text-blue-600 dark:text-blue-400" />
              <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200">
                Recently Added Products
              </h3>

              {/* Info button with tooltip */}
              {products.length > 0 && (
                <div className="relative group">
                  <Info className="w-4 h-4 text-gray-400 dark:text-gray-300 cursor-pointer" />
                  <div className="absolute left-1/2 transform -translate-x-1/2 -top-8 bg-gray-800 text-white text-xs rounded py-1 px-2 opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50">
                    This table shows the 5 most recently added products.
                  </div>
                </div>
              )}
            </div>


            <ul className="divide-y divide-gray-200 dark:divide-gray-700 max-h-100">
              {products.length > 0 ? (
                products
                  .slice(0, 5)
                  .map((product) => (
                    <li
                      key={product.product_id}
                      className="py-2 flex justify-between items-center hover:bg-gray-50 dark:hover:bg-gray-800 transition-all rounded-md px-2 animate-fade-in-up"
                      title={`Risk: ${product.risk_level || "N/A"} | Category: ${product.category || "N/A"}`}
                    >
                      <div className="flex items-center gap-2">
                        {/* Risk dot */}
                        <span
                          className="w-2 h-2 rounded-full"
                          style={{
                            backgroundColor:
                              product.risk_level === "High"
                                ? "#ef4444"
                                : product.risk_level === "Medium"
                                  ? "#facc15"
                                  : "#22c55e",
                          }}
                        />
                        <div className="flex flex-col">
                          <p className="text-sm font-medium text-gray-800 dark:text-gray-200">{product.product_name}</p>
                          <p className="text-xs text-gray-500 dark:text-gray-400">{product.supplier_name || "Unknown Supplier"}</p>
                          <p className="text-xs text-gray-400 dark:text-gray-500">
                            Added: {product.created_at ? new Date(product.created_at).toLocaleDateString() : "Unknown"}
                          </p>
                        </div>
                      </div>

                      <span
                        className={`text-xs font-medium px-2 py-0.5 rounded-full ${product.status === "Active"
                          ? "bg-green-100 text-green-700 dark:bg-green-800 dark:text-green-300"
                          : "bg-red-100 text-red-700 dark:bg-red-800 dark:text-red-300"
                          }`}
                      >
                        {product.status}
                      </span>
                    </li>
                  ))
              ) : (
                <p className="text-gray-500 dark:text-gray-400 text-sm">No products found.</p>
              )}
            </ul>

            <Link
              href="/products"
              className="text-sm text-blue-600 dark:text-blue-400 hover:underline mt-2 block"
            >
              View All
            </Link>
          </div>
        </div>

      </main>
    </div>
  );
}
