"use client";

import { useState, useEffect, useRef } from "react";
import { useRouter } from "next/navigation";
import {
  LogOutIcon,
  MoonIcon,
  SunIcon,
  SearchIcon,
  MenuIcon,
  UserIcon,
  BellIcon,
  SettingsIcon,
  Maximize2Icon,
  Minimize2Icon,
  MoreHorizontal,
} from "lucide-react";
import { motion, AnimatePresence } from "framer-motion";
import { useAuth } from "@/context/AuthContext";
import { useDarkMode } from "@/data/useDarkMode";
import { useSearch } from "@/context/SearchContext";
import { SidebarData } from "@/data/sidebarData";
import Link from "next/link";
import { highlightMatch } from "./common/HighLightMatch";

interface NavbarProps {
  onMobileToggle: () => void;
}

export default function Navbar({ onMobileToggle }: NavbarProps) {
  const [darkMode, setDarkMode] = useDarkMode();

  const { searchTerm, setSearchTerm } = useSearch();
  const [isProfileOpen, setIsProfileOpen] = useState(false);
  const [isNotificationsOpen, setIsNotificationsOpen] = useState(false);
  const [isFullScreen, setIsFullScreen] = useState(false);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const mobileMenuRef = useRef<HTMLDivElement>(null);
  const notificationRef = useRef<HTMLDivElement>(null);
  const profileRef = useRef<HTMLDivElement>(null);
  const router = useRouter();
  const { logout, user } = useAuth();
  const unreadCount = 3; // Replace with dynamic value later
  const [mounted, setMounted] = useState(false);

  useEffect(() => {
    setMounted(true);
  }, []);

  useEffect(() => {
    const handleFullScreenChange = () => {
      setIsFullScreen(!!document.fullscreenElement);
    };
    document.addEventListener("fullscreenchange", handleFullScreenChange);
    return () =>
      document.removeEventListener("fullscreenchange", handleFullScreenChange);
  }, []);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (
        mobileMenuRef.current &&
        !mobileMenuRef.current.contains(event.target as Node)
      ) {
        setIsMobileMenuOpen(false);
      }
    }
    if (isMobileMenuOpen) {
      document.addEventListener("mousedown", handleClickOutside);
    } else {
      document.removeEventListener("mousedown", handleClickOutside);
    }

    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, [isMobileMenuOpen]);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (
        profileRef.current &&
        !profileRef.current.contains(event.target as Node)
      ) {
        setIsProfileOpen(false);
      }

      if (
        notificationRef.current &&
        !notificationRef.current.contains(event.target as Node)
      ) {
        setIsNotificationsOpen(false);
      }
    };

    document.addEventListener("mousedown", handleClickOutside);
    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, []);

  const handleLogout = () => {
    logout();
    router.push("/login");
  };

  const handleProfileClick = () => {
    router.push("/login/profile");
    setIsProfileOpen(false);
  };

  const handleSettingsClick = () => {
    router.push("/login/settings");
    setIsProfileOpen(false);
  };

  const toggleFullScreen = () => {
    if (!document.fullscreenElement) {
      document.documentElement
        .requestFullscreen()
        .catch((err) => console.error("Fullscreen error:", err));
    } else {
      document.exitFullscreen();
    }
  };

  return (
    <nav className="sticky top-0 z-30 w-full px-2 sm:px-4 py-2 sm:py-3 backdrop-blur-lg bg-gradient-to-r from-green-600/90 to-green-700/90 dark:from-gray-900/90 dark:to-gray-800/90 text-white shadow-lg">
      <div className="w-full flex items-center justify-between flex-wrap gap-2 sm:gap-4">
        {/* Left: Logo & Mobile Menu */}
        <div className="flex items-center gap-2 sm:gap-3">
          {mounted && (
            <motion.button
              whileHover={{ scale: 1.05 }}
              whileTap={{ scale: 0.95 }}
              className="lg:hidden p-1 sm:p-2 rounded-lg hover:bg-green-700/50 dark:hover:bg-gray-700/50 transition-colors"
              onClick={onMobileToggle}
            >
              <MenuIcon className="w-5 h-5 sm:w-6 h-6" />
            </motion.button>
          )}
          {mounted && (
            <motion.div
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.5 }}
              className="block md:hidden text-lg sm:text-xl font-extrabold tracking-tight whitespace-nowrap"
            >
              QMS System
            </motion.div>
          )}
          {/* Search bar */}
          <div className="relative flex items-center w-full max-w-[120px] sm:max-w-xs md:max-w-sm lg:max-w-md xl:max-w-lg">
            {mounted && (
              <motion.div
                initial={{ width: "80%" }}
                whileFocus={{ width: "100%" }}
                transition={{ duration: 0.3 }}
                className="relative flex items-center w-full px-2 sm:px-3 py-1 sm:py-2 rounded-full bg-white/10 hover:bg-white/20 transition-all shadow-inner"
              >
                <input
                  type="text"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  placeholder="Search..."
                  className="flex-1 bg-transparent outline-none text-white placeholder-white/60 text-xs sm:text-sm pr-6 sm:pr-8"
                />
                <motion.button
                  whileHover={{ scale: 1.1 }}
                  whileTap={{ scale: 0.9 }}
                  onClick={() => {}}
                  className="absolute right-1 sm:right-2 text-white hover:text-green-200"
                >
                  <SearchIcon className="w-3 h-3 sm:w-4 h-4" />
                </motion.button>
                {searchTerm && (
                  <div className="absolute top-full left-0 mt-1 w-full max-h-64 overflow-y-auto bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-md shadow-lg z-50 overflow-x-auto max-w-xs">
                    <ul className="divide-y divide-gray-200 dark:divide-gray-600 text-sm">
                      {SidebarData.flatMap((section) =>
                        section.submenuItems
                          .filter((item) =>
                            item.label
                              .toLowerCase()
                              .includes(searchTerm.toLowerCase())
                          )
                          .map((item) => (
                            <li key={item.link}>
                              <Link
                                href={item.link}
                                className="block px-4 py-2 hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-900 dark:text-white"
                                onClick={() => setSearchTerm("")}
                              >
                                {highlightMatch(item.label, searchTerm)}
                              </Link>
                            </li>
                          ))
                      )}
                    </ul>
                  </div>
                )}
              </motion.div>
            )}
          </div>
        </div>

        {/* Right: Icons */}
        <div className="hidden lg:flex items-center gap-1 sm:gap-2">
          {/* Notifications Button with Dropdown */}
          <div className="relative" ref={notificationRef}>
            {mounted && (
              <motion.button
                whileHover={{ scale: 1.1 }}
                whileTap={{ scale: 0.9 }}
                onClick={() => setIsNotificationsOpen(!isNotificationsOpen)}
                className="relative group p-1 sm:p-2 rounded-lg hover:bg-white/10 dark:hover:bg-gray-700/50 transition-colors"
              >
                <BellIcon className="w-4 h-4 sm:w-5 h-5 text-white" />
                {unreadCount > 0 && (
                  <span className="absolute -top-1 -right-1 bg-green-500 text-white text-[10px] px-1.5 py-0.5 rounded-full">
                    {unreadCount}
                  </span>
                )}
                <div className="absolute top-full mt-2 px-2 py-1 text-xs bg-green-600/80 dark:bg-gray-800/80 text-white rounded shadow-lg opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none">
                  Notifications
                </div>
              </motion.button>
            )}
            <AnimatePresence>
              {isNotificationsOpen && (
                <motion.div
                  initial={{ opacity: 0, y: -5 }}
                  animate={{ opacity: 1, y: 0 }}
                  exit={{ opacity: 0, y: -5 }}
                  transition={{ duration: 0.2 }}
                  className="absolute right-0 mt-2 w-48 bg-white dark:bg-gray-800 text-gray-900 dark:text-white rounded-lg shadow-xl overflow-hidden z-40"
                >
                  <div className="divide-y divide-gray-200 dark:divide-gray-700">
                    {[
                      "Server Restarted",
                      "New User Registered",
                      "New Comment Received",
                    ].map((note, i) => (
                      <div
                        key={i}
                        className="p-2 hover:bg-green-100 dark:hover:bg-green-700/50 cursor-pointer text-sm"
                      >
                        {note}
                      </div>
                    ))}
                  </div>
                </motion.div>
              )}
            </AnimatePresence>
          </div>

          {/* Dark Mode Button with Tooltip */}
          {mounted && (
            <motion.button
              aria-label="Toggle dark mode"
              whileHover={{ scale: 1.1 }}
              whileTap={{ scale: 0.9 }}
              onClick={() => setDarkMode(!darkMode)}
              className="relative group p-1 sm:p-2 rounded-lg hover:bg-white/10 dark:hover:bg-gray-700/50 transition-colors"
            >
              <AnimatePresence mode="wait">
                {darkMode ? (
                  <motion.div
                    key="sun"
                    initial={{ rotate: -90, opacity: 0 }}
                    animate={{ rotate: 0, opacity: 1 }}
                    exit={{ rotate: 90, opacity: 0 }}
                    transition={{ duration: 0.3 }}
                  >
                    <SunIcon className="w-4 h-4 sm:w-5 h-5 text-yellow-400" />
                  </motion.div>
                ) : (
                  <motion.div
                    key="moon"
                    initial={{ rotate: -90, opacity: 0 }}
                    animate={{ rotate: 0, opacity: 1 }}
                    exit={{ rotate: 90, opacity: 0 }}
                    transition={{ duration: 0.3 }}
                  >
                    <MoonIcon className="w-4 h-4 sm:w-5 h-5 text-white" />
                  </motion.div>
                )}
              </AnimatePresence>
              <div className="absolute top-full mt-2 px-2 py-1 text-xs bg-green-600/80 dark:bg-gray-800/80 text-white rounded shadow-lg opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none">
                Toggle Theme
              </div>
            </motion.button>
          )}
          {/* Full-Screen Button */}
          {mounted && (
            <motion.button
              whileHover={{ scale: 1.1 }}
              whileTap={{ scale: 0.9 }}
              onClick={toggleFullScreen}
              className="relative group p-1 sm:p-2 rounded-lg hover:bg-white/10 dark:hover:bg-gray-700/50 transition-colors"
            >
              {mounted &&
                (isFullScreen ? (
                  <Minimize2Icon className="w-4 h-4 sm:w-5 h-5 text-white" />
                ) : (
                  <Maximize2Icon className="w-4 h-4 sm:w-5 h-5 text-white" />
                ))}
              <div className="absolute top-full mt-2 px-2 py-1 text-xs bg-green-600/80 dark:bg-gray-800/80 text-white rounded shadow-lg opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none">
                {isFullScreen ? "Exit Full Screen" : "Full Screen"}
              </div>
            </motion.button>
          )}
          {/* Logout Button */}
          {mounted && (
            <motion.button
              whileHover={{ scale: 1.1 }}
              whileTap={{ scale: 0.9 }}
              onClick={handleLogout}
              className="relative group p-1 sm:p-2 rounded-lg hover:bg-white/10 dark:hover:bg-gray-700/50 transition-colors text-red-300 hover:text-red-100"
            >
              <LogOutIcon className="w-4 h-4 sm:w-5 h-5" />
              <div className="absolute top-full mt-2 px-2 py-1 text-xs bg-green-600/80 dark:bg-gray-800/80 text-white rounded shadow-lg opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none">
                Logout
              </div>
            </motion.button>
          )}

          {/* Profile Button with Dropdown */}
          <div className="relative" ref={profileRef}>
            {mounted && (
              <motion.button
                whileHover={{ scale: 1.1 }}
                whileTap={{ scale: 0.9 }}
                onClick={() => setIsProfileOpen(!isProfileOpen)}
                className="relative group p-1 sm:p-2 rounded-lg hover:bg-white/10 dark:hover:bg-gray-700/50 transition-colors profile-button"
              >
                <UserIcon className="w-4 h-4 sm:w-5 h-5 text-white" />
                <div className="absolute top-full mt-2 px-2 py-1 text-xs bg-green-600/80 dark:bg-gray-800/80 text-white rounded shadow-lg opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none">
                  Profile
                </div>
              </motion.button>
            )}
            <AnimatePresence>
              {isProfileOpen && (
                <motion.div
                  initial={{ opacity: 0, y: -5 }}
                  animate={{ opacity: 1, y: 0 }}
                  exit={{ opacity: 0, y: -5 }}
                  transition={{ duration: 0.2 }}
                  className="absolute right-0 mt-2 w-48 bg-white dark:bg-gray-800 text-gray-900 dark:text-white rounded-lg shadow-xl overflow-hidden z-40"
                >
                  <div className="flex items-center p-2 border-b border-gray-200 dark:border-gray-700">
                    <UserIcon className="w-10 h-10 text-gray-500 mr-2" />
                    <div>
                      <p className="text-sm font-medium">
                        {user?.name || "Guest"}
                      </p>
                      {/* <p className="text-xs text-gray-500 dark:text-gray-400">{user?.email || 'No email'}</p> */}
                    </div>
                  </div>
                  <button
                    onClick={handleProfileClick}
                    className="w-full px-4 py-2 text-left flex items-center gap-2 hover:bg-green-100 dark:hover:bg-green-700/50 transition-colors"
                  >
                    <UserIcon className="w-4 h-4" />
                    Profile
                  </button>
                  <button
                    onClick={handleSettingsClick}
                    className="w-full px-4 py-2 text-left flex items-center gap-2 hover:bg-green-100 dark:hover:bg-green-700/50 transition-colors"
                  >
                    <SettingsIcon className="w-4 h-4" />
                    Settings
                  </button>
                  <button
                    onClick={handleLogout}
                    className="w-full px-4 py-2 text-left flex items-center gap-2 text-red-600 dark:text-red-400 hover:bg-green-100 dark:hover:bg-green-700/50 transition-colors"
                  >
                    <LogOutIcon className="w-4 h-4" />
                    Logout
                  </button>
                </motion.div>
              )}
            </AnimatePresence>
          </div>
        </div>

        {/* Right: Dropdown Action Button for Mobile */}
        <div className="lg:hidden relative" ref={mobileMenuRef}>
          {mounted && (
            <motion.button
              whileHover={{ scale: 1.1 }}
              whileTap={{ scale: 0.95 }}
              onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
              className="p-2 rounded-full bg-white/10 hover:bg-white/20 dark:hover:bg-gray-700/50 transition"
            >
              <MoreHorizontal className="w-5 h-5 text-white" />
            </motion.button>
          )}
          {/* Mobile Menu Dropdown */}
          {isMobileMenuOpen && (
            <motion.div
              initial={{ opacity: 0, y: -10 }}
              animate={{ opacity: 1, y: 0 }}
              exit={{ opacity: 0, y: -10 }}
              transition={{ duration: 0.2 }}
              className="absolute right-0 mt-2 w-48 bg-white dark:bg-gray-800 text-gray-900 dark:text-white rounded-lg shadow-lg z-50 overflow-hidden"
            >
              <button
                onClick={() => {
                  setDarkMode(!darkMode);
                  setIsMobileMenuOpen(false);
                }}
                className="w-full px-4 py-2 flex items-center gap-2 hover:bg-green-100 dark:hover:bg-green-700/50"
              >
                {darkMode ? (
                  <SunIcon className="w-4 h-4" />
                ) : (
                  <MoonIcon className="w-4 h-4" />
                )}
                Toggle Theme
              </button>

              <button
                onClick={() => {
                  toggleFullScreen();
                  setIsMobileMenuOpen(false);
                }}
                className="w-full px-4 py-2 flex items-center gap-2 hover:bg-green-100 dark:hover:bg-green-700/50"
              >
                {isFullScreen ? (
                  <Minimize2Icon className="w-4 h-4" />
                ) : (
                  <Maximize2Icon className="w-4 h-4" />
                )}
                {isFullScreen ? "Exit Fullscreen" : "Enter Fullscreen"}
              </button>

              <button
                onClick={() => {
                  setIsNotificationsOpen(true);
                  setTimeout(() => setIsNotificationsOpen(false), 3000);
                  setIsMobileMenuOpen(false);
                }}
                className="w-full px-4 py-2 flex items-center gap-2 hover:bg-green-100 dark:hover:bg-green-700/50"
              >
                <BellIcon className="w-4 h-4" />
                Notifications
              </button>

              <button
                onClick={() => {
                  handleProfileClick();
                  setIsMobileMenuOpen(false);
                }}
                className="w-full px-4 py-2 flex items-center gap-2 hover:bg-green-100 dark:hover:bg-green-700/50"
              >
                <UserIcon className="w-4 h-4" />
                Profile
              </button>

              <button
                onClick={() => {
                  handleLogout();
                  setIsMobileMenuOpen(false);
                }}
                className="w-full px-4 py-2 flex items-center gap-2 text-red-600 dark:text-red-400 hover:bg-green-100 dark:hover:bg-green-700/50"
              >
                <LogOutIcon className="w-4 h-4" />
                Logout
              </button>
            </motion.div>
          )}
        </div>
      </div>
    </nav>
  );
}
