"use client";
import React, { useCallback, useEffect, useState } from "react";
import {
  ListTodo,
  Plus,
  Inbox,
  CircleCheck,
  Flag,
  CalendarDays,
  Ellipsis,
  Search,
  AlignLeft,
  X,
  Type,
  User,
  ListChecks,
  FileText,
  Eye,
  Edit,
  Trash2,
} from "lucide-react";
import { motion } from "framer-motion";
import Button from "antd/es/button";
import "antd/es/button/style";
import Table from "antd/es/table";
import "antd/es/table/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import dynamic from "next/dynamic";
import type { ColumnsType } from "antd/es/table";

// Dynamic import for performance optimization
const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

export default function TodoPage() {
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [todoTitle, setTodoTitle] = useState("");
  const [assignee, setAssignee] = useState("Choose");
  const [priority, setPriority] = useState("High");
  const [status, setStatus] = useState("In Progress");
  const [dueDate, setDueDate] = useState("");
  const [description, setDescription] = useState("");
  const [todos, setTodos] = useState<any[]>([]);
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });

  const handleAddTask = () => setIsModalOpen(true);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    const newTodo = {
      id: `${Date.now()}-${Math.random()}`, // Unique ID for each todo
      title: todoTitle,
      description,
      priority,
      status,
      assignedFrom: assignee,
      assignedTo: assignee,
      createdAt: new Date().toLocaleString(),
      dueDate,
    };
    setTodos([...todos, newTodo]);
    setIsModalOpen(false);
    setTodoTitle("");
    setAssignee("Choose");
    setPriority("High");
    setStatus("In Progress");
    setDueDate("");
    setDescription("");
  };

  const handleCancel = () => {
    setIsModalOpen(false);
    setTodoTitle("");
    setAssignee("Choose");
    setPriority("High");
    setStatus("In Progress");
    setDueDate("");
    setDescription("");
  };

  const handleDelete = (id: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Todo",
      message: "Are you sure you want to delete this todo?",
      onConfirm: () => {
        setTodos(todos.filter((todo) => todo.id !== id));
        setSelectedRows(selectedRows.filter((rowId) => rowId !== id));
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Todos",
      message: `Are you sure you want to delete ${selectedRows.length} todos?`,
      onConfirm: () => {
        setTodos(todos.filter((todo) => !selectedRows.includes(todo.id)));
        setSelectedRows([]);
      },
    });
  };

  const openEditModal = (todo: any) => {
    setTodoTitle(todo.title);
    setAssignee(todo.assignedTo);
    setPriority(todo.priority);
    setStatus(todo.status);
    setDueDate(todo.dueDate);
    setDescription(todo.description);
    setIsModalOpen(true);
    setTodos(todos.filter((t) => t.id !== todo.id)); // Remove the todo to be edited
  };

  const renderPriorityTag = (priority: string) => {
    const colorMap: Record<string, string> = {
      High: "red",
      Medium: "yellow",
      Low: "green",
    };
    return <Tag color={colorMap[priority] || "default"}>{priority}</Tag>;
  };

  const renderStatusTag = (status: string) => {
    const colorMap: Record<string, string> = {
      "In Progress": "blue",
      Pending: "orange",
      Onhold: "yellow",
      Done: "green",
    };
    return <Tag color={colorMap[status] || "default"}>{status}</Tag>;
  };

  const columns: ColumnsType<any> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? todos.map((t) => t.id) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id)}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.id]
                : prev.filter((id) => id !== record.id)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Title",
      dataIndex: "title",
      key: "title",
      sorter: (a, b) => a.title.localeCompare(b.title),
      render: (text) => (
        <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[200px]">{text}</span>
      ),
    },
    {
      title: "Description",
      dataIndex: "description",
      key: "description",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[200px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Priority",
      dataIndex: "priority",
      key: "priority",
      sorter: (a, b) => a.priority.localeCompare(b.priority),
      render: renderPriorityTag,
      filters: ["High", "Medium", "Low"].map((p) => ({ text: p, value: p })),
      onFilter: (value, record) => record.priority === value,
    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      sorter: (a, b) => a.status.localeCompare(b.status),
      render: renderStatusTag,
      filters: ["In Progress", "Pending", "Onhold", "Done"].map((s) => ({
        text: s,
        value: s,
      })),
      onFilter: (value, record) => record.status === value,
    },
    {
      title: "Assigner",
      dataIndex: "assigner",
      key: "assigner",
      sorter: (a, b) => a.assignedFrom.localeCompare(b.assignedFrom),
    },
    {
      title: "Assignee",
      dataIndex: "assignee",
      key: "assignee",
      sorter: (a, b) => a.assignedTo.localeCompare(b.assignedTo),
    },
    {
      title: "Due Date",
      dataIndex: "dueDate",
      key: "dueDate",
      sorter: (a, b) => (a.dueDate || "").localeCompare(b.dueDate || ""),
      render: (text) => text || "-",
    },
    {
      title: "Created At",
      dataIndex: "createdAt",
      key: "createdAt",
      sorter: (a, b) => a.createdAt.localeCompare(b.createdAt),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <div className="flex gap-2">
          <Tooltip title="View">
            <Button
              icon={<Eye className="w-4 h-4" />}
              size="small"
              className="text-blue-600"
              onClick={() => alert(`Viewing ${record.title}`)} // Placeholder for view action
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<Edit className="w-4 h-4" />}
              size="small"
              className="bg-blue-600 hover:bg-blue-700 text-white"
              onClick={() => openEditModal(record)}
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<Trash2 className="w-4 h-4" />}
              size="small"
              className="text-red-600"
              onClick={() => handleDelete(record.id)}
            />
          </Tooltip>
        </div>
      ),
    },
  ];

  const filteredTodos = todos.filter(
    (todo) =>
      todo.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
      todo.description.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const paginatedTodos = filteredTodos.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  return (
    <div className="flex flex-col h-screen bg-gray-100 dark:bg-gray-900 text-gray-800 dark:text-white overflow-hidden">
      {/* === HEADER === */}
      <div className="flex flex-wrap justify-between items-center px-4 sm:px-6 py-3 border-b bg-white dark:bg-gray-800">
        <div className="flex items-center gap-2 mb-2 sm:mb-0">
          <ListTodo className="w-6 h-6 text-blue-500" />
          <div>
            <h1 className="text-xl sm:text-2xl font-bold">Todo</h1>
            <p className="text-xs sm:text-sm text-gray-500">Manage your tasks</p>
          </div>
        </div>
        <div className="flex items-center gap-2 sm:gap-3">
          <button className="p-2 bg-gray-200 dark:bg-gray-700 rounded">
            <Ellipsis className="w-5 h-5" />
          </button>
          <button
            onClick={handleAddTask}
            className="bg-gradient-to-r from-cyan-500 via-sky-500 to-blue-600 hover:opacity-90 text-white px-3 sm:px-4 py-2 rounded flex items-center gap-2 shadow-md text-sm sm:text-base"
          >
            <Plus className="w-4 h-4" />
            Add Task
          </button>
        </div>
      </div>
      {/* === MAIN LAYOUT === */}
      <div className="flex flex-1 overflow-hidden flex-col sm:flex-row">
        {/* === SIDEBAR === */}
        <aside className="sm:w-64 flex-shrink-0 bg-white dark:bg-gray-800 p-4 border-b sm:border-r border-gray-200 dark:border-gray-700 overflow-y-auto">
          <h2 className="flex items-center gap-2 text-lg font-semibold mb-4">
            <FileText className="w-5 h-5 text-purple-500" />
            Todo List
          </h2>
          {/* Inbox & Completed */}
          <div className="space-y-2">
            <button className="flex items-center justify-between w-full bg-green-500 text-white px-4 py-2 rounded">
              <span className="flex items-center gap-2">
                <Inbox className="w-4 h-4" /> Inbox
              </span>
              <span className="bg-black/40 text-white text-xs px-2 py-0.5 rounded">
                {todos.length}
              </span>
            </button>
            <button className="flex items-center gap-2 w-full px-4 py-2 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded">
              <CircleCheck className="w-4 h-4" /> Completed
            </button>
          </div>
          <hr className="my-4 border-gray-300 dark:border-gray-700" />
          {/* Tags Section */}
          <div className="mb-4">
            <h3 className="font-semibold mb-2">Tags</h3>
            <ul className="space-y-1">
              {["Pending", "Onhold", "Inprogress", "Done"].map((tag) => {
                const tagData: Record<
                  string,
                  { color: string; icon: React.ElementType }
                > = {
                  Pending: {
                    color: "text-orange-500",
                    icon: require("lucide-react").Clock,
                  },
                  Onhold: {
                    color: "text-yellow-500",
                    icon: require("lucide-react").PauseCircle,
                  },
                  Inprogress: {
                    color: "text-pink-500",
                    icon: require("lucide-react").Loader2,
                  },
                  Done: {
                    color: "text-green-500",
                    icon: require("lucide-react").CheckCircle2,
                  },
                };
                const { color, icon: Icon } = tagData[tag];
                return (
                  <li
                    key={tag}
                    className={`flex items-center gap-2 text-sm ${color} hover:text-blue-600 transition-colors duration-300 cursor-pointer`}
                  >
                    <Icon className={`w-3 h-3 ${color}`} />
                    <span className="truncate">{tag}</span>
                  </li>
                );
              })}
            </ul>
          </div>
          {/* Priority Section */}
          <div>
            <h3 className="font-semibold mb-2">Priority</h3>
            <ul className="space-y-1">
              {["High", "Medium", "Low", "All"].map((level) => {
                const priorityData: Record<
                  string,
                  { color: string; icon: React.ElementType }
                > = {
                  High: {
                    color: "text-red-500",
                    icon: require("lucide-react").Flame,
                  },
                  Medium: {
                    color: "text-orange-500",
                    icon: require("lucide-react").Zap,
                  },
                  Low: {
                    color: "text-green-500",
                    icon: require("lucide-react").ArrowDown,
                  },
                  All: {
                    color: "text-purple-500",
                    icon: require("lucide-react").Filter,
                  },
                };
                const { color, icon: Icon } = priorityData[level];
                return (
                  <li
                    key={level}
                    className={`flex items-center gap-2 text-sm ${color} hover:text-blue-600 transition-colors duration-300 cursor-pointer`}
                  >
                    <Icon className={`w-3 h-3 ${color}`} />
                    <span className="truncate">{level}</span>
                  </li>
                );
              })}
            </ul>
          </div>
          {/* Clear Filters */}
          <button className="mt-4 bg-gradient-to-r from-pink-500 via-rose-500 to-fuchsia-500 text-white px-4 py-2 rounded text-sm w-full shadow-md hover:opacity-90 transition">
            Clear Filters
          </button>
        </aside>
        {/* === MAIN CONTENT === */}
        <main className="flex-1 p-4 sm:p-6 overflow-y-auto">
          {/* Search & Filters */}
          <div className="bg-white dark:bg-gray-800 p-4 rounded shadow mb-4 flex flex-col sm:flex-row justify-between items-center gap-4">
            <div className="flex gap-4 text-sm flex-wrap">
              <span className="bg-gray-100 dark:bg-gray-700 px-2 py-1 rounded">
                Total: {todos.length}
              </span>
              <span className="bg-orange-100 text-orange-600 dark:bg-orange-700 dark:text-white px-2 py-1 rounded">
                Pending: {todos.filter((todo) => todo.status === "Pending").length}
              </span>
              <span className="bg-green-100 text-green-600 dark:bg-green-700 dark:text-white px-2 py-1 rounded">
                Completed: {todos.filter((todo) => todo.status === "Done").length}
              </span>
              {selectedRows.length > 0 && (
                <Button
                  onClick={handleBulkDelete}
                  danger
                  icon={<Trash2 className="w-4 h-4" />}
                  className="text-red-600"
                >
                  Delete Selected ({selectedRows.length})
                </Button>
              )}
            </div>
            <div className="relative w-full sm:w-auto">
              <Search className="absolute left-2 top-2.5 w-4 h-4 text-gray-400" />
              <Input
                prefix={<Search className="w-4 h-4 text-gray-400" />}
                allowClear
                placeholder="Search todos..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-8 pr-3 py-2 border rounded w-full sm:w-64 dark:bg-gray-700 dark:border-gray-600"
              />
            </div>
          </div>
          {/* Todo Table */}
          {todos.length > 0 ? (
            <div className="overflow-x-auto">
              <Table
                columns={columns}
                dataSource={paginatedTodos}
                rowKey="id"
                pagination={false}
                scroll={{ x: "max-content" }}
                className="bg-white dark:bg-gray-800 rounded shadow"
              />
              <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300 mt-4">
                <div>
                  Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                  <strong>
                    {Math.min(currentPage * pageSize, filteredTodos.length)}
                  </strong>{" "}
                  of <strong>{filteredTodos.length}</strong> todos
                </div>
                <Pagination
                  current={currentPage}
                  pageSize={pageSize}
                  total={filteredTodos.length}
                  onChange={(page, size) => {
                    setCurrentPage(page);
                    setPageSize(size);
                  }}
                  showSizeChanger
                  pageSizeOptions={["10", "20", "50"]}
                  size="small"
                />
              </div>
            </div>
          ) : (
            <div className="flex justify-center items-center h-64">
              <p className="text-gray-500 dark:text-gray-400">No ToDos found</p>
            </div>
          )}
        </main>
      </div>
      {/* === ADD TASK MODAL === */}
      {isModalOpen && (
        <div className="fixed inset-0 bg-black/40 backdrop-blur-sm flex items-center justify-center z-50 px-4">
          <motion.div
            initial={{ opacity: 0, scale: 0.9, y: 30 }}
            animate={{ opacity: 1, scale: 1, y: 0 }}
            exit={{ opacity: 0, scale: 0.9, y: 30 }}
            transition={{ duration: 0.3, ease: "easeOut" }}
            className="bg-white dark:bg-gray-800 p-6 rounded-2xl shadow-xl w-full max-w-lg relative"
          >
            <button
              onClick={() => setIsModalOpen(false)}
              className="absolute top-3 right-3 text-gray-400 hover:text-red-500 transition"
            >
              <X className="w-5 h-5" />
            </button>
            <h2 className="text-xl font-semibold mb-5 flex items-center gap-2">
              <Plus className="w-5 h-5 text-red-600" />
              Add Todo
            </h2>
            {/* FORM */}
            <form onSubmit={handleSubmit} className="space-y-4">
              <div>
                <label className="flex items-center gap-2 text-sm font-medium mb-1">
                  <Type className="w-4 h-4 text-gray-500" />
                  Todo Title
                </label>
                <input
                  type="text"
                  value={todoTitle}
                  onChange={(e) => setTodoTitle(e.target.value)}
                  className="w-full p-2 border rounded-lg text-sm dark:bg-gray-700 dark:border-gray-600 focus:ring-2 focus:ring-red-500 outline-none transition"
                  placeholder="Title"
                  maxLength={60}
                />
              </div>
              <div>
                <label className="flex items-center gap-2 text-sm font-medium mb-1">
                  <User className="w-4 h-4 text-gray-500" />
                  Assignee
                </label>
                <select
                  value={assignee}
                  onChange={(e) => setAssignee(e.target.value)}
                  className="w-full p-2 border rounded-lg text-sm dark:bg-gray-700 dark:border-gray-600 focus:ring-2 focus:ring-red-500 outline-none transition"
                >
                  <option>Choose</option>
                </select>
              </div>
              <div className="flex flex-col sm:flex-row gap-4">
                <div className="sm:w-1/2">
                  <label className="flex items-center gap-2 text-sm font-medium mb-1">
                    <Flag className="w-4 h-4 text-red-500" />
                    Priority
                  </label>
                  <select
                    value={priority}
                    onChange={(e) => setPriority(e.target.value)}
                    className="w-full p-2 border rounded-lg text-sm dark:bg-gray-700 dark:border-gray-600 focus:ring-2 focus:ring-red-500 outline-none transition"
                  >
                    <option>High</option>
                    <option>Medium</option>
                    <option>Low</option>
                  </select>
                </div>
                <div className="sm:w-1/2">
                  <label className="flex items-center gap-2 text-sm font-medium mb-1">
                    <ListChecks className="w-4 h-4 text-blue-500" />
                    Status
                  </label>
                  <select
                    value={status}
                    onChange={(e) => setStatus(e.target.value)}
                    className="w-full p-2 border rounded-lg text-sm dark:bg-gray-700 dark:border-gray-600 focus:ring-2 focus:ring-blue-500 outline-none transition"
                  >
                    <option>In Progress</option>
                    <option>Pending</option>
                    <option>Onhold</option>
                    <option>Done</option>
                  </select>
                </div>
              </div>
              <div>
                <label className="flex items-center gap-2 text-sm font-medium mb-1">
                  <CalendarDays className="w-4 h-4 text-green-500" />
                  Due Date
                </label>
                <input
                  type="date"
                  value={dueDate}
                  onChange={(e) => setDueDate(e.target.value)}
                  className="w-full p-2 border rounded-lg text-sm dark:bg-gray-700 dark:border-gray-600 focus:ring-2 focus:ring-green-500 outline-none transition"
                />
              </div>
              <div>
                <label className="flex items-center gap-2 text-sm font-medium mb-1">
                  <AlignLeft className="w-4 h-4 text-purple-500" />
                  Description
                </label>
                <input
                  type="text"
                  value={description}
                  onChange={(e) => setDescription(e.target.value)}
                  className="w-full p-2 border rounded-lg text-sm dark:bg-gray-700 dark:border-gray-600 focus:ring-2 focus:ring-purple-500 outline-none transition"
                  placeholder="Enter details"
                />
                <p className="text-xs text-gray-500 mt-1">
                  Maximum 60 Characters
                </p>
              </div>
              <div className="flex justify-end gap-3 pt-4">
                <button
                  type="button"
                  onClick={handleCancel}
                  className="px-4 py-2 bg-gray-600 text-white rounded-lg text-sm hover:bg-gray-700 transition"
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  className="px-4 py-2 bg-green-600 text-white rounded-lg text-sm hover:bg-green-700 transition"
                >
                  Submit
                </button>
              </div>
            </form>
          </motion.div>
        </div>
      )}
      <ConfirmDialog
        open={confirmDialog.open}
        title={confirmDialog.title}
        message={confirmDialog.message}
        onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
        onConfirm={confirmDialog.onConfirm}
      />
    </div>
  );
}