// app/api/training-need.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// All interfaces (exactly as you wrote – NOT touched)
export interface TrainingNeed {
  employee_code?: string;
  training_topic: string;
  training_reason: string;
  priority_level: string;
  proposed_date: string;
  comments?: string | null;
}

export interface TrainingNeedData {
  training_code: string;
  employee_id?: number;
  employee_code?: string;
  employee_name?: string;
  training_topic: string;
  training_reason?: string;
  priority_level: string;
  proposed_date?: string;
  approval_status?: string;
  identified_by_user_id?: string;
  identified_by_name?: string;
  approved_by_user_id?: string | null;
  approved_by_name?: string | null;
  approved_date?: string | null;
  rejection_reason?: string | null;
  comments?: string | null;
  created_at?: string;
  updated_at?: string;
}

export interface CreateTrainingNeedResponse {
  status: string;
  message: string;
  data: {
    training_code: string;
    employee_code: string;
  };
}

export interface GetAllTrainingNeedsResponse {
  status: string;
  count: number;
  data: TrainingNeedData[];
}

export interface GetTrainingNeedByCodeResponse {
  status: string;
  message: string;
  data: TrainingNeedData;
}

export interface UpdateTrainingNeedResponse {
  status: string;
  message: string;
  updated_data: {
    training_code: string;
    training_topic: string;
    priority_level: string;
    approval_status: string;
  };
}

export interface UpdateApprovalStatusData {
  approval_status: "Approved" | "Rejected" | "Pending";
  rejection_reason?: string | null;
  comments?: string | null;
}

export interface UpdateApprovalStatusResponse {
  status: string;
  message: string;
  data: {
    training_code: string;
    approval_status: string;
    approved_by: string;
    approved_date: string;
    rejection_reason: string | null;
    comments: string | null;
  };
}

export interface DeleteTrainingNeedResponse {
  status: string;
  message: string;
}

// Get auth headers (your original logic – untouched)
const getAuthHeaders = () => {
  let token = localStorage.getItem("authToken") || "";

  if (!token.startsWith("Bearer ")) {
    token = `Bearer ${token}`;
  }

  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

// Helper to build training need form data (your original – untouched)
const buildTrainingNeedForm = (data: TrainingNeed): URLSearchParams => {
  return new URLSearchParams({
    ...(data.employee_code ? { employee_code: data.employee_code } : {}),
    training_topic: data.training_topic,
    training_reason: data.training_reason,
    priority_level: data.priority_level,
    proposed_date: data.proposed_date,
    comments: data.comments || "",
  });
};

// Helper to build approval status form data (your original – untouched)
const buildApprovalStatusForm = (
  data: UpdateApprovalStatusData
): URLSearchParams => {
  return new URLSearchParams({
    approval_status: data.approval_status,
    rejection_reason: data.rejection_reason || "",
    comments: data.comments || "",
  });
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE TRAINING NEED
export const createTrainingNeed = async (
  data: TrainingNeed
): Promise<CreateTrainingNeedResponse> => {
  try {
    const response = await axios.post(
      `${API_BASE_URL}/create_training_need`,
      buildTrainingNeedForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL TRAINING NEEDS
export const getAllTrainingNeeds = async (): Promise<TrainingNeedData[]> => {
  try {
    const response = await axios.get<GetAllTrainingNeedsResponse>(
      `${API_BASE_URL}/get_all_training_needs`,
      getAuthHeaders()
    );
    return response.data.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET TRAINING NEED BY CODE
export const getTrainingNeedByCode = async (
  trainingCode: string
): Promise<TrainingNeedData> => {
  try {
    const response = await axios.get<GetTrainingNeedByCodeResponse>(
      `${API_BASE_URL}/get_training_need_by_code/${trainingCode}`,
      getAuthHeaders()
    );
    return response.data.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE TRAINING NEED
export const updateTrainingNeed = async (
  trainingCode: string,
  data: TrainingNeed
): Promise<UpdateTrainingNeedResponse> => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update_training_need/${trainingCode}`,
      buildTrainingNeedForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE APPROVAL STATUS (Approve / Reject)
export const updateApprovalStatus = async (
  trainingCode: string,
  data: UpdateApprovalStatusData
): Promise<UpdateApprovalStatusResponse> => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update_approval_status/${trainingCode}`,
      buildApprovalStatusForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE TRAINING NEED
export const deleteTrainingNeed = async (
  trainingCode: string
): Promise<DeleteTrainingNeedResponse> => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete_training_need/${trainingCode}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
