// lib/api/supplier-visit.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Supplier Visit Interface (exactly as you wrote – NOT touched)
export interface SupplierVisit {
  supplierId: string;
  visit_type: string;
  visit_code?: string;
  place: string;
  material: string;
  auditor_name: string;
  scheduled_month: string;
  completion_date: string;
  Status: string;
  risk_classification: string;
  description: string;
  objective?: string;
  follow_up_date?: string;
  attacted_file?: File | string | null;
  is_confidential?: boolean;
  created_at?: string;
  updated_at?: string;
  created_by_id?: string;
  created_by_name?: string;
  updated_by_id?: string | null;
  updated_by_name?: string | null;
}

// Auth headers (your original)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE SUPPLIER VISIT
export const createSupplierVisit = async (data: SupplierVisit) => {
  try {
    const form = new FormData();
    form.append("place", data.place);
    form.append("material", data.material);
    form.append("auditor_name", data.auditor_name);
    form.append("supplier_id", data.supplierId);
    form.append("scheduled_month", data.scheduled_month);
    form.append("visit_type", data.visit_type);
    form.append("completion_date", data.completion_date);
    form.append("Status", data.Status);
    form.append("risk_classification", data.risk_classification);
    form.append("description", data.description);

    if (data.objective) form.append("objective", data.objective);
    if (data.follow_up_date) form.append("follow_up_date", data.follow_up_date);
    if (data.is_confidential !== undefined)
      form.append("is_confidential", String(data.is_confidential));
    if (data.attacted_file && data.attacted_file instanceof File)
      form.append("attacted_file", data.attacted_file);

    const response = await axios.post(
      `${API_BASE_URL}/create-suppliervisit`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL SUPPLIER VISITS
export const getAllSupplierVisits = async (): Promise<SupplierVisit[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-suppliervisit`,
      getAuthHeaders()
    );
    return response.data.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE SUPPLIER VISIT
export const updateSupplierVisit = async (
  supplierId: string,
  data: Partial<SupplierVisit>
) => {
  try {
    const form = new FormData();

    if (data.place) form.append("place", data.place);
    if (data.material) form.append("material", data.material);
    if (data.auditor_name) form.append("auditor_name", data.auditor_name);
    if (data.scheduled_month)
      form.append("scheduled_month", data.scheduled_month);
    if (data.visit_type) form.append("visit_type", data.visit_type);
    if (data.completion_date)
      form.append("completion_date", data.completion_date);
    if (data.Status) form.append("Status", data.Status);
    if (data.risk_classification)
      form.append("risk_classification", data.risk_classification);
    if (data.description) form.append("description", data.description);

    if (data.objective) form.append("objective", data.objective);
    if (data.follow_up_date) form.append("follow_up_date", data.follow_up_date);
    if (data.is_confidential !== undefined)
      form.append("is_confidential", String(data.is_confidential));
    if (data.attacted_file && data.attacted_file instanceof File) {
      form.append("attacted_file", data.attacted_file);
    }

    const response = await axios.put(
      `${API_BASE_URL}/update-suppliervisit/${supplierId}`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET SUPPLIER VISIT BY SUPPLIER ID
export const getSupplierVisitById = async (
  supplierId: string
): Promise<SupplierVisit[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-suppliervisit-by-supplier/${supplierId}`,
      getAuthHeaders()
    );
    return response.data.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE SUPPLIER VISIT BY SUPPLIER ID
export const deleteSupplierVisitById = async (supplierId: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-suppliervisit-by-supplier/${supplierId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD SUPPLIER VISIT DOCUMENT
export const downloadSupplierVisit = async (supplierId: string) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-supplier-vist/${supplierId}`,
      {
        ...getAuthHeaders(),
        responseType: "blob",
      }
    );

    // Extract filename if backend sends it
    const contentDisposition = response.headers["content-disposition"];
    let fileName = `SupplierVisit_${supplierId}.pdf`;
    if (contentDisposition) {
      const match = contentDisposition.match(/filename="(.+?)"/);
      if (match?.[1]) fileName = match[1];
    }

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", fileName);
    document.body.appendChild(link);
    link.click();
    link.remove();

    return true;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
