// lib/api/supplier-certification.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Types (exactly as you wrote – NOT touched)
export interface SupplierCertification {
  document_id?: string;
  supplier_id: string;
  supplier_name?: string;
  certification_type: string;
  certification_number: string;
  issuing_body: string;
  issue_date: string;
  expiry_date: string;
  scope: string;
  status: string;
  file_path?: string | File | null;
  created_at?: string;
  created_by_id?: string;
  created_by_name?: string;
  accreditation_body: string;
  country_of_issue: string;
  renewal_date: string;
  suspension_date: string;
  withdrawal_date: string;
  version_number: string;
  criticality: string;
  remarks: string;
}

// Auth header (your original)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
    },
  };
};

// Multipart header for uploads (your original)
const getUploadHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      "Content-Type": "multipart/form-data",
      Accept: "application/json",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// 1. UPLOAD CERTIFICATION
export const uploadSupplierCertification = async (
  formData: FormData
): Promise<{ Message: string; Detail: SupplierCertification }> => {
  try {
    const response = await axios.post(
      `${API_BASE_URL}/upload-document`,
      formData,
      getUploadHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// 2. GET ALL CERTIFICATIONS
export const getAllSupplierCertifications = async (): Promise<
  SupplierCertification[]
> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/view-all-documents`,
      getAuthHeaders()
    );
    return response.data.Documents;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// 3. UPDATE CERTIFICATION
export const updateSupplierCertification = async (
  documentId: string,
  formData: FormData
): Promise<{ Message: string; Detail: SupplierCertification }> => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update-document/${documentId}`,
      formData,
      getUploadHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// 4. DELETE CERTIFICATION
export const deleteSupplierCertification = async (
  documentId: string
): Promise<{ Message: string }> => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-document/${documentId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// 5. DOWNLOAD CERTIFICATION
export const downloadSupplierCertification = async (
  documentId: string
): Promise<Blob> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-document/${documentId}`,
      {
        ...getAuthHeaders(),
        responseType: "blob",
      }
    );

    // Auto download with proper filename
    const contentDisposition = response.headers["content-disposition"];
    let fileName = `Certification_${documentId}.pdf`;
    if (contentDisposition) {
      const match = contentDisposition.match(/filename="(.+?)"/);
      if (match?.[1]) fileName = match[1];
    }

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", fileName);
    document.body.appendChild(link);
    link.click();
    link.remove();

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
