// lib/api/supplier-action.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Supplier Action Interface (exactly as you wrote – NOT touched)
export interface SupplierAction {
  supplier_id: string;
  product_id: string;
  unitName: string;
  materialDescription?: string;
  invoiceNo: string;
  supplierLotNo: string;
  dateIdentified: string;
  dateResolved: string;
  nonConformanceDetails: string;
  capaNo: string;
  Status: string;
  priorityLevel: string;
  department: string;
  action_type: string;
  evidence_file?: File | string | null;
  description: string;
  root_cause_analysis: string;
  responsible_person: string;
  targetdate: string;
  duedate: string;
  action_taken: string;

  // system fields
  created_at?: string;
  updated_at?: string;
  created_id?: string;
  created_name?: string;
  updated_id?: string | null;
  updated_name?: string | null;
  overallScore?: number;
}

// Auth headers helper (your original)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
    },
  };
};

// Show exact backend error message only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE SUPPLIER ACTION
export const createSupplierAction = async (data: SupplierAction) => {
  try {
    const form = new FormData();

    form.append("supplier_id", data.supplier_id);
    form.append("product_id", data.product_id);
    form.append("unitName", data.unitName);
    if (data.materialDescription)
      form.append("materialDescription", data.materialDescription);
    form.append("invoiceNo", data.invoiceNo);
    form.append("supplierLotNo", data.supplierLotNo);
    form.append("dateIdentified", data.dateIdentified);
    form.append("dateResolved", data.dateResolved);
    form.append("nonConformanceDetails", data.nonConformanceDetails);
    form.append("capaNo", data.capaNo);
    form.append("Status", data.Status);
    form.append("priorityLevel", data.priorityLevel);
    form.append("department", data.department);
    form.append("action_type", data.action_type);
    form.append("description", data.description);
    form.append("root_cause_analysis", data.root_cause_analysis);
    form.append("responsible_person", data.responsible_person);
    form.append("targetdate", data.targetdate);
    form.append("duedate", data.duedate);
    form.append("action_taken", data.action_taken);

    if (data.evidence_file && data.evidence_file instanceof File) {
      form.append("evidence_file", data.evidence_file);
    }

    const response = await axios.post(
      `${API_BASE_URL}/create-supplier-action`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL SUPPLIER ACTIONS
export const getAllSupplierActions = async (): Promise<SupplierAction[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-all-supplier-actions`, getAuthHeaders() );
    return response.data?.Data ?? [];
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET SUPPLIER ACTION BY SUPPLIER + PRODUCT
export const getSupplierAction = async (
  supplierId: string,
  productId: string
): Promise<SupplierAction> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-supplier-action/${supplierId}/${productId}`,
      getAuthHeaders()
    );
    return response.data.Detail;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE SUPPLIER ACTION
export const updateSupplierAction = async (
  supplierId: string,
  productId: string,
  data: Partial<SupplierAction>
) => {
  try {
    const form = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      if (value !== undefined && value !== null) {
        if (key === "evidence_file" && value instanceof File) {
          form.append("evidence_file", value);
        } else {
          form.append(key, String(value));
        }
      }
    });

    const response = await axios.put(
      `${API_BASE_URL}/update-supplier-action/${supplierId}/${productId}`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE SUPPLIER ACTION
export const deleteSupplierAction = async (
  supplierId: string,
  productId: string
) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-supplier-action/${supplierId}/${productId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD SUPPLIER ACTION DOCUMENT
export const downloadSupplierAction = async (
  supplierId: string,
  productId: string
) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-supplier-action-document/${supplierId}-${productId}`,
      {
        ...getAuthHeaders(),
        responseType: "blob",
      }
    );

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute(
      "download",
      `SupplierAction_${supplierId}_${productId}.pdf`
    );
    document.body.appendChild(link);
    link.click();
    link.remove();

    return true;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};