// lib/api/purchaserequisition.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Type for Purchase Requisition (exactly as you wrote – NOT touched)
export interface PurchaseRequisition {
  purchase_requisition_id?: string;
  purchase_requisition_date: string | null;
  product_id: string | null;
  product_name: string | null;
  quantity: number | null;
  department: string | null;
  description: string | null;
  unit_of_measure: string | null;
  expected_date: string | null;
  estimated_cost: string | null;
  justification: string | null;
  priority: string | null;
  Status: string | null;
  supported_file?: File | string | null;
  created_at?: string;
  updated_at?: string;
  approved_at?: string;
  requested_by_id?: string;
  requested_by_name?: string;
  approved_by_id?: string;
  approved_by_name?: string;
}

// Get auth headers (your original)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// Build FormData (your original logic – kept exactly)
const buildRequisitionForm = (data: PurchaseRequisition): FormData => {
  const form = new FormData();
  form.append("product_id", data.product_id || "");
  form.append(
    "purchase_requisition_date",
    data.purchase_requisition_date || ""
  );
  form.append("department", data.department || "");
  form.append("quantity", data.quantity?.toString() || "");
  form.append("description", data.description || "");
  form.append("unit_of_measure", data.unit_of_measure || "");
  form.append("expected_date", data.expected_date || "");
  form.append("estimated_cost", data.estimated_cost || "");
  form.append("justification", data.justification || "");
  form.append("priority", data.priority || "");
  form.append("Status", data.Status || "");
  if (data.supported_file && data.supported_file instanceof File) {
    form.append("supported_file", data.supported_file);
  }
  return form;
};

// CREATE
export const createPurchaseRequisition = async (data: PurchaseRequisition) => {
  try {
    const response = await axios.post(
      `${API_BASE_URL}/create-purchase-requisition`,
      buildRequisitionForm(data),
      {
        ...getAuthHeaders(),
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL
export const getPurchaseRequisitions = async (): Promise<
  PurchaseRequisition[]
> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-all-purchase-requisition`,
      getAuthHeaders()
    );
    return response.data.Details;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET BY ID
export const getPurchaseRequisitionById = async (
  id: string
): Promise<PurchaseRequisition> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-purchase-requisition/${id}`,
      getAuthHeaders()
    );
    return response.data.Details;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE
export const updatePurchaseRequisition = async (
  id: string,
  data: PurchaseRequisition
) => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update-purchase-requisition/${id}`,
      buildRequisitionForm(data),
      {
        ...getAuthHeaders(),
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE
export const deletePurchaseRequisition = async (id: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-purchase-requisition/${id}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD DOCUMENT
export const downloadPurchaseRequisitionDocument = async (id: string) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-purchase-requisition-document/${id}`,
      {
        ...getAuthHeaders(),
        responseType: "blob",
      }
    );

    const contentDisposition = response.headers["content-disposition"];
    let fileName = `PurchaseRequisition_${id}`;
    if (contentDisposition) {
      const match = contentDisposition.match(/filename="(.+)"/);
      if (match?.[1]) {
        fileName = match[1];
      }
    }

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", fileName);
    document.body.appendChild(link);
    link.click();
    link.remove();

    return true;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
