// lib/api/purchase-order.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// ------------------ Interfaces (100% untouched – exactly as you wrote) ------------------
export interface PurchaseOrder {
  purchase_order_id?: string;
  purchase_requisition_id: string;
  supplier_id: string;
  supplier_type?: string;
  supplier_name?: string;
  product_id?: string;
  product_name?: string;
  po_date?: string | null;
  expected_delivery_date?: string | null;
  payment_terms?: string | null;
  shipping_address?: string | null;
  billing_address?: string | null;
  quantity_ordered?: number | null;
  unit_of_measure?: string | null;
  price_per_unit?: string | null;
  total_amount?: string | null;
  remarks?: string | null;
  Approve_status: string;
  Status: string;
  supported_file?: File | string | null;
  created_at?: string | null;
  updated_at?: string | null;
  approved_at?: string | null;
  created_by_id?: string | null;
  created_by_name?: string | null;
  approved_by_id?: string | null;
  approved_by_name?: string | null;
}

export interface PurchaseOrderCreate {
  purchase_requisition_id: string;
  supplier_id: string;
  Approve_status: string;
  Status: string;
  po_date?: string | null;
  expected_delivery_date?: string | null;
  payment_terms?: string | null;
  shipping_address?: string | null;
  billing_address?: string | null;
  quantity_ordered?: number | null;
  unit_of_measure?: string | null;
  price_per_unit?: string | null;
  total_amount?: string | null;
  remarks?: string | null;
  supported_file?: File | null;
}

export type PurchaseOrderUpdate = Partial<PurchaseOrder>;

// ------------------ Auth Headers (your original) ------------------
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

// ------------------ Show exact backend error only ------------------
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE PURCHASE ORDER
export const createPurchaseOrder = async (data: PurchaseOrderCreate) => {
  try {
    const form = new FormData();

    form.append("purchase_requisition_id", data.purchase_requisition_id);
    form.append("supplier_id", data.supplier_id);
    form.append("Approve_status", data.Approve_status);
    form.append("Status", data.Status);

    if (data.po_date) form.append("po_date", data.po_date);
    if (data.expected_delivery_date)
      form.append("expected_delivery_date", data.expected_delivery_date);
    if (data.payment_terms) form.append("payment_terms", data.payment_terms);
    if (data.shipping_address)
      form.append("shipping_address", data.shipping_address);
    if (data.billing_address)
      form.append("billing_address", data.billing_address);
    if (data.quantity_ordered !== undefined)
      form.append("quantity_ordered", String(data.quantity_ordered));
    if (data.unit_of_measure)
      form.append("unit_of_measure", data.unit_of_measure);
    if (data.price_per_unit) form.append("price_per_unit", data.price_per_unit);
    if (data.total_amount) form.append("total_amount", data.total_amount);
    if (data.remarks) form.append("remarks", data.remarks);
    if (data.supported_file) {
      form.append("supported_file", data.supported_file);
    }

    const response = await axios.post(
      `${API_BASE_URL}/create-purchase-order`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data.Detail as PurchaseOrder;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL PURCHASE ORDERS
export const getAllPurchaseOrders = async (): Promise<PurchaseOrder[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-all-purchase-orders`,
      getAuthHeaders()
    );
    return response.data.Data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET PURCHASE ORDER BY ID
export const getPurchaseOrderById = async (
  purchase_order_id: string
): Promise<PurchaseOrder> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-purchase-order/${purchase_order_id}`,
      getAuthHeaders()
    );
    return response.data.Detail;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE PURCHASE ORDER
export const updatePurchaseOrder = async (
  purchase_order_id: string,
  data: PurchaseOrderUpdate
) => {
  try {
    const form = new FormData();

    if (data.purchase_requisition_id)
      form.append("purchase_requisition_id", data.purchase_requisition_id);
    if (data.supplier_id) form.append("supplier_id", data.supplier_id);
    if (data.po_date) form.append("po_date", data.po_date);
    if (data.expected_delivery_date)
      form.append("expected_delivery_date", data.expected_delivery_date);
    if (data.payment_terms) form.append("payment_terms", data.payment_terms);
    if (data.shipping_address)
      form.append("shipping_address", data.shipping_address);
    if (data.billing_address)
      form.append("billing_address", data.billing_address);
    if (data.quantity_ordered !== undefined)
      form.append("quantity_ordered", String(data.quantity_ordered));
    if (data.unit_of_measure)
      form.append("unit_of_measure", data.unit_of_measure);
    if (data.price_per_unit) form.append("price_per_unit", data.price_per_unit);
    if (data.total_amount) form.append("total_amount", data.total_amount);
    if (data.remarks) form.append("remarks", data.remarks);
    if (data.Approve_status) form.append("Approve_status", data.Approve_status);
    if (data.Status) form.append("Status", data.Status);
    if (data.supported_file && data.supported_file instanceof File) {
      form.append("supported_file", data.supported_file);
    }

    const response = await axios.put(
      `${API_BASE_URL}/update-purchase-order/${purchase_order_id}`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data.Detail as PurchaseOrder;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE PURCHASE ORDER
export const deletePurchaseOrderById = async (purchase_order_id: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-purchase-order/${purchase_order_id}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD PURCHASE ORDER DOCUMENT
export const downloadPurchaseOrderDocument = async (
  purchaseOrderId: string
) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-purchase-order-document/${purchaseOrderId}`,
      {
        ...getAuthHeaders(),
        responseType: "blob",
      }
    );

    const contentDisposition = response.headers["content-disposition"];
    let fileName = `PurchaseOrder_${purchaseOrderId}.pdf`;
    if (contentDisposition) {
      const match = contentDisposition.match(/filename="(.+)"/);
      if (match?.[1]) {
        fileName = match[1].endsWith(".pdf") ? match[1] : `${match[1]}.pdf`;
      }
    }

    const url = window.URL.createObjectURL(
      new Blob([response.data], {
        type: response.data.type || "application/pdf",
      })
    );
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", fileName);
    document.body.appendChild(link);
    link.click();
    link.remove();

    return true;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
