// app/api/minimumqualificationcriteria/minimumqualificationcriteria.ts

import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner"; // ← Added

const API_BASE_URL = API_BASE_PATH;

/* ------------------------------------------------------
   TYPE – Minimum Qualification
------------------------------------------------------ */
export interface MinimumQualification {
  message?: string;
  criteria_code?: string;
  designation: string;
  department: string;
  minimum_education: string;
  minimum_experience: number | string;
  special_skills?: string;
  specific_knowledge?: string;
  certifications_required?: string;
  remarks?: string;
  status: string;
  rejection_reason?: string;
  created_at?: string;
  updated_at?: string;
  created_by_name?: string;
  approved_by_name?: string;
}

/* ------------------------------------------------------
   AUTH HEADERS
------------------------------------------------------ */
const getAuthHeaders = () => {
  let token = localStorage.getItem("authToken") || "";
  if (token && !token.startsWith("Bearer ")) {
    token = `Bearer ${token}`;
  }

  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

/* ------------------------------------------------------
   TOAST HELPER – Show exact backend message
------------------------------------------------------ */
const showSuccess = (message: string) => {
  toast.success(message || "Operation successful");
};

const showError = (error: any) => {
  const msg =
    error?.response?.data?.message ||
    error?.response?.data?.detail ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

/* ------------------------------------------------------
   CREATE MQ
------------------------------------------------------ */
export const createMinimumQualification = async (
  data: MinimumQualification
) => {
  const form = new URLSearchParams();
  form.append("designation", data.designation);
  form.append("department", data.department);
  form.append("minimum_education", data.minimum_education);
  form.append("minimum_experience", String(data.minimum_experience));
  form.append("special_skills", data.special_skills || "");
  form.append("specific_knowledge", data.specific_knowledge || "");
  form.append("certifications_required", data.certifications_required || "");
  form.append("remarks", data.remarks || "");
  form.append("status", data.status);

  try {
    const response = await axios.post(
      `${API_BASE_URL}/minimum-qualification/create`,
      form,
      getAuthHeaders()
    );
    showSuccess(response.data.message || "Criteria created successfully");
    return response.data;
  } catch (error: any) {
    showError(error);
    throw error;
  }
};

/* ------------------------------------------------------
   UPDATE MQ
------------------------------------------------------ */
export const updateMinimumQualification = async (
  criteria_code: string,
  data: MinimumQualification
) => {
  const form = new URLSearchParams();
  form.append("designation", data.designation);
  form.append("department", data.department);
  form.append("minimum_education", data.minimum_education);
  form.append("minimum_experience", String(data.minimum_experience));
  form.append("special_skills", data.special_skills || "");
  form.append("specific_knowledge", data.specific_knowledge || "");
  form.append("certifications_required", data.certifications_required || "");
  form.append("remarks", data.remarks || "");
  form.append("status", data.status);

  try {
    const response = await axios.put(
      `${API_BASE_URL}/minimum-qualification/update/${criteria_code}`,
      form,
      getAuthHeaders()
    );
    showSuccess(response.data.message || "Criteria updated successfully");
    return response.data;
  } catch (error: any) {
    showError(error);
    throw error;
  }
};

/* ------------------------------------------------------
   APPROVE / REJECT / SUBMIT MQ
------------------------------------------------------ */
export const approveOrRejectMinimumQualification = async (
  criteria_code: string,
  status: "Approved" | "Rejected" | "Submitted",
  rejection_reason?: string
) => {
  const form = new URLSearchParams();
  form.append("status", status);
  if (rejection_reason) {
    form.append("rejection_reason", rejection_reason);
  }

  try {
    const response = await axios.put(
      `${API_BASE_URL}/minimum-qualification/approve/${criteria_code}`,
      form,
      getAuthHeaders()
    );

    const msg = response.data.message;
    if (status === "Approved") showSuccess(msg || "Criteria Approved");
    else if (status === "Rejected") showError(msg || "Criteria Rejected");
    else showSuccess(msg || "Status updated to Submitted");

    return response.data;
  } catch (error: any) {
    showError(error);
    throw error;
  }
};

/* ------------------------------------------------------
   GET ALL MQ
------------------------------------------------------ */
export const getAllMinimumQualifications = async (): Promise<
  MinimumQualification[]
> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/minimum-qualification`,
      getAuthHeaders()
    );
    return response.data.data;
  } catch (error: any) {
    showError(error);
    throw error;
  }
};

/* ------------------------------------------------------
   DELETE MQ (only Draft allowed)
------------------------------------------------------ */
export const deleteMinimumQualification = async (criteria_code: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/minimum-qualification/${criteria_code}`,
      getAuthHeaders()
    );
    showSuccess(response.data.message || "Criteria deleted successfully");
    return response.data;
  } catch (error: any) {
    showError(error);
    throw error;
  }
};

/* ------------------------------------------------------
   COMPARE EMPLOYEE QUALIFICATION
------------------------------------------------------ */
export const compareEmployeeQualification = async (
  employee_code: string,
  criteria_code: string
) => {
  const form = new URLSearchParams();
  form.append("employee_code", employee_code);
  form.append("criteria_code", criteria_code);

  try {
    const response = await axios.post(
      `${API_BASE_URL}/compare-employee-qualification`,
      form,
      getAuthHeaders()
    );
    showSuccess("Comparison completed successfully");
    return response.data;
  } catch (error: any) {
    showError(error);
    throw error;
  }
};
