// app/api/employee/employee.ts

import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// ==============================
// Employee Interface (matches your backend)
// ==============================
export interface Employee {
  employee_code: string;
  first_name: string;
  last_name?: string | null;
  designation?: string | null;
  department?: string | null;
  email: string;
  phone: string;
  date_of_birth?: string | null;
  date_of_joining?: string | null;
  qualification?: string | null;
  experience_years?: number | null;
  employee_type?: string | null;
  employee_status?: string | null;
  reporting_manager_id?: string | null;
  reporting_manager_name?: string | null;
  remarks?: string | null;

  // File fields (can be File or string path after upload)
  resume_path?: File | string | null;
  qualification_doc_path?: File | string | null;
  id_proof_path?: File | string | null;

  // Response-only fields
  created_by_name?: string | null;
  updated_by_name?: string | null;
  created_at?: string;
  updated_at?: string;
}

// Auth headers
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  const authHeader = token.startsWith("Bearer ") ? token : `Bearer ${token}`;
  return {
    headers: {
      Authorization: authHeader,
      Accept: "application/json",
    },
  };
};

// Show exact backend error
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE EMPLOYEE
export const createEmployee = async (data: Employee) => {
  try {
    const form = new FormData();

    // Required fields
    form.append("first_name", data.first_name);
    form.append("email", data.email);
    form.append("phone", data.phone);

    // Optional fields
    if (data.last_name) form.append("last_name", data.last_name);
    if (data.designation) form.append("designation", data.designation);
    if (data.department) form.append("department", data.department);
    if (data.date_of_birth) form.append("date_of_birth", data.date_of_birth);
    if (data.date_of_joining)
      form.append("date_of_joining", data.date_of_joining);
    if (data.qualification) form.append("qualification", data.qualification);
    if (data.experience_years !== undefined && data.experience_years !== null)
      form.append("experience_years", String(data.experience_years));
    if (data.employee_type) form.append("employee_type", data.employee_type);
    if (data.employee_status)
      form.append("employee_status", data.employee_status);
    if (data.reporting_manager_id)
      form.append("reporting_manager_id", data.reporting_manager_id);
    if (data.remarks) form.append("remarks", data.remarks);

    // File uploads
    if (data.resume_path && data.resume_path instanceof File) {
      form.append("resume_path", data.resume_path);
    }
    if (
      data.qualification_doc_path &&
      data.qualification_doc_path instanceof File
    ) {
      form.append("qualification_doc_path", data.qualification_doc_path);
    }
    if (data.id_proof_path && data.id_proof_path instanceof File) {
      form.append("id_proof_path", data.id_proof_path);
    }

    const response = await axios.post(`${API_BASE_URL}/create_employee`, form, {
      headers: {
        ...getAuthHeaders().headers,
        "Content-Type": "multipart/form-data",
      },
    });

    toast.success("Employee created successfully");
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL EMPLOYEES
export const getAllEmployees = async (): Promise<Employee[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get_employee_detail`,
      getAuthHeaders()
    );

    if (response.data?.status === "success" || response.data?.data) {
      return response.data.data || [];
    }
    return [];
  } catch (error: any) {
    showBackendError(error);
    return [];
  }
};

// GET EMPLOYEE BY CODE
export const getEmployeeByCode = async (
  employee_code: string
): Promise<Employee> => {
  if (!employee_code) throw new Error("Employee code is required");

  try {
    const response = await axios.get(
      `${API_BASE_URL}/get_employee_detail/${employee_code}`,
      getAuthHeaders()
    );

    if (response.data?.status === "success" && response.data.data) {
      return response.data.data;
    }
    throw new Error("Employee not found");
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE EMPLOYEE - Fixed field names
export const updateEmployee = async (employee_code: string, data: any) => {
  try {
    const form = new FormData();

    if (data.first_name) form.append("first_name", data.first_name);
    if (data.last_name !== undefined)
      form.append("last_name", data.last_name || "");
    if (data.designation !== undefined)
      form.append("designation", data.designation || "");
    if (data.department !== undefined)
      form.append("department", data.department || "");
    if (data.email) form.append("email", data.email);
    if (data.phone) form.append("phone", data.phone);
    if (data.date_of_birth) form.append("date_of_birth", data.date_of_birth);
    if (data.date_of_joining)
      form.append("date_of_joining", data.date_of_joining);
    if (data.qualification !== undefined)
      form.append("qualification", data.qualification || "");
    if (data.experience_years !== undefined)
      form.append("experience_years", String(data.experience_years || ""));
    if (data.employee_type !== undefined)
      form.append("employee_type", data.employee_type || "");
    if (data.employee_status !== undefined)
      form.append("employee_status", data.employee_status || "");
    if (data.reporting_manager_id)
      form.append("reporting_manager_id", data.reporting_manager_id); // ← FIXED
    if (data.remarks !== undefined) form.append("remarks", data.remarks || "");

    // Files - correct names
    if (data.resume_path instanceof File)
      form.append("resume_path", data.resume_path);
    if (data.qualification_doc_path instanceof File)
      form.append("qualification_doc_path", data.qualification_doc_path);
    if (data.id_proof_path instanceof File)
      form.append("id_proof_path", data.id_proof_path);

    const response = await axios.put(
      `${API_BASE_URL}/update_employee_detail/${employee_code}`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    toast.success("Employee updated successfully");
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE EMPLOYEE
export const deleteEmployee = async (employee_code: string) => {
  if (!employee_code) throw new Error("Employee code is required");

  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete_employee_id${employee_code}`, 
      getAuthHeaders()
    );

    toast.success(response.data.message || "Employee deleted successfully");
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD EMPLOYEE FILE (Resume / Qualification / ID Proof)
export const downloadEmployeeFile = async (
  employee_code: string,
  fileType: "resume" | "qualification" | "id_proof"
) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-employee-file/${employee_code}`,
      {
        ...getAuthHeaders(),
        params: { file_type: fileType },
        responseType: "blob",
      }
    );

    const blob = new Blob([response.data]);
    const url = window.URL.createObjectURL(blob);
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", `${employee_code}_${fileType}.xlsx`);
    document.body.appendChild(link);
    link.click();
    link.remove();
    window.URL.revokeObjectURL(url);

    toast.success(
      `${fileType.charAt(0).toUpperCase() + fileType.slice(1)} downloaded`
    );
  } catch (error: any) {
    showBackendError(error);
  }
};

// COMPARE EMPLOYEE QUALIFICATION (Bonus: as per your last endpoint)
export const compareEmployeeQualification = async (
  employee_code: string,
  criteria_code: string
) => {
  try {
    const params = new URLSearchParams();
    params.append("employee_code", employee_code);
    params.append("criteria_code", criteria_code);

    const response = await axios.post(
      `${API_BASE_URL}/compare-employee-qualification`,
      params,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "application/x-www-form-urlencoded",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
